# ONVIF Python

[![License](https://img.shields.io/badge/License-MIT-blue)](https://github.com/nirsimetri/onvif-python?tab=MIT-1-ov-file)
[![DeepWiki](https://img.shields.io/badge/DeepWiki-AI%20Wiki-orange)](https://deepwiki.com/nirsimetri/onvif-python)
[![Release](https://img.shields.io/badge/Release-v0.0.3-red?logo=archive)](https://github.com/nirsimetri/onvif-python/releases)
<br>
[![PyPI](https://img.shields.io/badge/PyPI-0.0.3-yellow?logo=archive)](https://pypi.org/project/onvif-python/)
[![Downloads](https://img.shields.io/pypi/dm/onvif-python?label=PyPI%20Downloads)](https://pypi.org/project/onvif-python/)

Are you having trouble finding a Python ONVIF library that supports your device?  
Are you confused about the compatibility of the various ONVIF versions, which are updated every six months?

**This project provides a comprehensive and developer-friendly Python library for working with ONVIF-compliant devices.** It is designed to be reliable, easy to integrate, and flexible enough to support a wide range of ONVIF profiles and services.  

**[ONVIF](https://www.onvif.org) (Open Network Video Interface Forum)** is a global standard for the interface of IP-based physical security products, including network cameras, video recorders, and related systems.  

Behind the scenes, ONVIF communication relies on **[SOAP](https://en.wikipedia.org/wiki/SOAP) (Simple Object Access Protocol)** — an [XML](https://en.wikipedia.org/wiki/XML)-based messaging protocol with strict schema definitions ([WSDL](https://en.wikipedia.org/wiki/Web_Services_Description_Language)/[XSD](https://en.wikipedia.org/wiki/XML_Schema_(W3C))). SOAP ensures interoperability, but when used directly it can be verbose, complex, and error-prone.  

This library simplifies that process by wrapping SOAP communication into a clean, Pythonic API. You no longer need to handle low-level XML parsing, namespaces, or security tokens manually — the library takes care of it, letting you focus on building functionality.  

## Key Features
- Full implementation of ONVIF core services and profiles  
- Support for device discovery, media streaming, PTZ control, event management, and more  
- Pythonic abstraction over SOAP requests and responses (no need to handcraft XML)  
- Extensible architecture for custom ONVIF extensions  
- Compatible with multiple ONVIF specification versions  
- Example scripts and tests included  

## Who Is It For?
- **Individual developers** exploring ONVIF or building hobby projects  
- **Companies** building video intelligence, analytics, or VMS platforms  
- **Security integrators** who need reliable ONVIF interoperability across devices

## Installation

From official [PyPI](https://pypi.org/project/onvif-python/):
```bash
pip install onvif-python
```
Or clone this repository and install locally:
```bash
git clone https://github.com/nirsimetri/onvif-python
cd onvif-python
pip install .
```

## Usage Example

> [!TIP]
> You can view the complete documentation automatically generated by DeepWiki via the [onvif-python AI Wiki](https://deepwiki.com/nirsimetri/onvif-python) link. We currently do not have an official documentation site. Help us create more examples and helpful documentation by [contributing](https://github.com/nirsimetri/onvif-python?tab=contributing-ov-file).

Below are simple examples to help you get started with the ONVIF Python library. These demonstrate how to connect to an ONVIF-compliant device and retrieve basic device information.

**1. Initialize the ONVIFClient**

Create an instance of `ONVIFClient` by providing your device's IP address, port, username, and password:

```python
from onvif import ONVIFClient

client = ONVIFClient("192.168.1.17", 8000, "admin", "admin123")
```

**2. Create Service Instance**

`ONVIFClient` provides several main services that can be accessed via the following methods:

- `client.devicemgmt()` — Device Management
- `client.events()` — Events
- `client.imaging()` — Imaging
- `client.media()` — Media
- `client.ptz()` — PTZ (Pan-Tilt-Zoom)
- `client.analytics()` — Analytics

and so on, check [Implemented ONVIF Services](https://github.com/nirsimetri/onvif-python?tab=readme-ov-file#implemented-onvif-services) for more details

Example usage:
```python
device = client.devicemgmt()      # Device Management (Core)
media = client.media()            # Media
```

**3. Get Device Information**

Retrieve basic information about the device, such as manufacturer, model, firmware version, and serial number using `devicemgmt()` service:

```python
info = device.GetDeviceInformation()
print(info)
# Example output: {'Manufacturer': '..', 'Model': '..', 'FirmwareVersion': '..', 'SerialNumber': '..'}
```

**4. Get RTSP URL**

Retrieve the RTSP stream URL for live video streaming from the device using `media()` service:

```python
profile = media.GetProfiles()[0]  # use the first profile
stream = media.GetStreamUri(
    ProfileToken=profile.token, 
	StreamSetup={"Stream": "RTP-Unicast", "Transport": {"Protocol": "RTSP"}}
)
print(stream)
# Example output: {'Uri': 'rtsp://192.168.1.17:8554/Streaming/Channels/101', ...}
```

Explore more advanced usage and service-specific operations in the [`examples/`](./examples/) folder.

> [!IMPORTANT]
> If you're new to ONVIF and want to learn more, we highly recommend taking the official free online course provided by ONVIF at [Introduction to ONVIF Course](https://www.onvif.org/about/introduction-to-onvif-course). Please note that we are not endorsed or sponsored by ONVIF, see [Legal Notice](#legal-notice) for details.

## Device Verification: Why Use GetCapabilities First?

> [!WARNING]
> Before performing any operations on an ONVIF device, it is highly recommended to verify which capabilities and services are available and supported by the device using the `GetCapabilities` method from `devicemgmt()` service instance. This step ensures that your application interacts only with features that the device actually implements, preventing errors and improving compatibility.

**Why verify device capabilities with GetCapabilities?**

- **Device Diversity:** Not all ONVIF devices support every capability or service. Capabilities may vary by manufacturer, model, firmware, or configuration.
- **Error Prevention:** Attempting to use unsupported features can result in failed requests, exceptions, or undefined behavior.
- **Dynamic Feature Detection:** Devices may enable or disable capabilities over time (e.g., after firmware updates or configuration changes).
- **Optimized Integration:** By checking available capabilities, your application can adapt its workflow and UI to match the device's actual features.

**How to verify device capabilities:**

Call `GetCapabilities` on your `devicemgmt()` instance:

```python
from onvif import ONVIFClient

client = ONVIFClient("192.168.1.17", 8000, "admin", "admin123")
capabilities = client.devicemgmt().GetCapabilities()
print(capabilities)
# Example output: {'Media': {'XAddr': 'http://192.168.1.17:8000/onvif/media_service', ...}, 'PTZ': {...}, ...}
```

Review the returned dictionary to determine which capabilities and services (e.g., Media, PTZ, Analytics) are available before invoking further operations.

## Tested Devices

This library has been tested with a variety of ONVIF-compliant devices. For the latest and most complete list of devices that have been verified to work with this library, please refer to:

- [List of tested devices (device-test)](https://github.com/nirsimetri/onvif-products-directory/blob/main/device-test)

If your device is not listed right now, feel free to contribute your test results or feedback via Issues or Discussions at [onvif-products-directory](https://github.com/nirsimetri/onvif-products-directory). Your contribution will be invaluable to the community and the public.

> [!IMPORTANT]
> Device testing contributions must be made with a real device and use the scripts provided in the [onvif-products-directory](https://github.com/nirsimetri/onvif-products-directory) repo. Please be sure to contribute using a device model not already listed.

## Supported ONVIF Profiles

This library fully supports all major ONVIF Profiles listed below. Each profile represents a standardized set of features and use cases, ensuring interoperability between ONVIF-compliant devices and clients. You can use this library to integrate with devices and systems that implement any of these profiles.

| Name      | Specifications | Main Features | Typical Use Case | Support |
|-----------|----------------|---------------|------------------|---------|
| Profile_S | [Document](https://www.onvif.org/wp-content/uploads/2019/12/ONVIF_Profile_-S_Specification_v1-3.pdf) | Video streaming, PTZ, audio, multicasting | Network video transmitters (cameras) and receivers (recorders, VMS) | ✅ Yes |
| Profile_G | [Document](https://www.onvif.org/wp-content/uploads/2017/01/ONVIF_Profile_G_Specification_v1-0.pdf) | Recording, search, replay, video storage | Video recorders, storage devices | ✅ Yes |
| Profile_T | [Document](https://www.onvif.org/wp-content/uploads/2018/09/ONVIF_Profile_T_Specification_v1-0.pdf) | Advanced video streaming (H.265, analytics metadata, motion detection) | Modern cameras and clients | ✅ Yes |
| Profile_C | [Document](https://www.onvif.org/wp-content/uploads/2017/01/2013_12_ONVIF_Profile_C_Specification_v1-0.pdf) | Access control, door monitoring | Door controllers, access systems | ✅ Yes |
| Profile_A | [Document](https://www.onvif.org/wp-content/uploads/2017/06/ONVIF_Profile_A_Specification_v1-0.pdf) | Advanced access control configuration, credential management | Access control clients and devices | ✅ Yes |
| Profile_D | [Document](https://www.onvif.org/wp-content/uploads/2021/06/onvif-profile-d-specification-v1-0.pdf) | Access control peripherals (locks, sensors, relays) | Peripheral devices for access control | ✅ Yes |
| Profile_M | [Document](https://www.onvif.org/wp-content/uploads/2024/04/onvif-profile-m-specification-v1-1.pdf) | Metadata, analytics events, object detection | Analytics devices, metadata clients | ✅ Yes |

For a full description of each profile and its features, visit [ONVIF Profiles](https://www.onvif.org/profiles/).

## Implemented ONVIF Services

> [!NOTE]
> For details about the available service functions and methods already implemented in this library, see the source code in [`onvif/services/`](./onvif/services). Or if you want to read in a more proper format visit [onvif-python AI Wiki](https://deepwiki.com/nirsimetri/onvif-python).

Below is a list of ONVIF services implemented and supported by this library, along with links to the official specifications, service definitions, and schema files as referenced from the [ONVIF Developer Specs](https://developer.onvif.org/pub/specs/branches/development/doc/index.html). This table provides a quick overview of the available ONVIF features and their technical documentation for integration and development purposes.

| Service                | Specifications                | Service Definitions         | Schema Files                        | Status     |
|------------------------|-------------------------------|-----------------------------|-------------------------------------|------------|
| Device Management      | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Core.xml)                   | [device.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/device/wsdl/devicemgmt.wsdl)    | [onvif.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/onvif.xsd) <br> [common.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/common.xsd)                | ✅ Complete |
| Events                 | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Core.xml)                   | [event.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/events/wsdl/event.wsdl)    | [onvif.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/onvif.xsd) <br> [common.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/common.xsd)                | ⚠️ Partial |
| Access Control         | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AccessControl.xml)         | [accesscontrol.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/pacs/accesscontrol.wsdl)         | [types.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/pacs/types.xsd)                            | ✅ Complete |
| Access Rules           | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AccessRules.xml)           | [accessrules.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/accessrules/wsdl/accessrules.wsdl)           | -                                      | ✅ Complete |
| Action Engine          | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/ActionEngine.xml)          | [actionengine.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/actionengine.wsdl)          | -                                      | ✅ Complete |
| Analytics              | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Analytics.xml)             | [analytics.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/wsdl/analytics.wsdl)             | [rules.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/rules.xsd) <br> [humanbody.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/humanbody.xsd) <br> [humanface.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/humanface.xsd) | ✅ Complete |
| Application Management | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AppMgmt.xml)               | [appmgmt.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/appmgmt/wsdl/appmgmt.wsdl)               | -                                     | ✅ Complete |
| Authentication Behavior| [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AuthenticationBehavior.xml) | [authenticationbehavior.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/authenticationbehavior/wsdl/authenticationbehavior.wsdl) | -                                     | ✅ Complete |
| Cloud Integration      | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/CloudIntegration.xml)      | [cloudintegration.yaml](https://developer.onvif.org/pub/specs/branches/development/doc/yaml.php?yaml=cloudintegration.yaml)      | -                                     | ❌ Not yet |
| Credential             | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Credential.xml)            | [credential.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/credential/wsdl/credential.wsdl)            | -                                     | ✅ Complete |
| Device IO              | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/DeviceIo.xml)              | [deviceio.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/deviceio.wsdl)              |-                                      | ✅ Complete |
| Display                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Display.xml)               | [display.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/display.wsdl)               | -                                     | ✅ Complete |
| Door Control           | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/DoorControl.xml)           | [doorcontrol.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/pacs/doorcontrol.wsdl)           | -                                     | ✅ Complete |
| Imaging                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Imaging.xml)               | [imaging.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/imaging/wsdl/imaging.wsdl)               | -                                     | ✅ Complete |
| Media                  | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Media.xml)                 | [media.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/media/wsdl/media.wsdl)                 | -                                     | ✅ Complete |
| Media 2                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Media2.xml)                | [media2.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/media/wsdl/media.wsdl)                | -                                     | ✅ Complete |
| Provisioning           | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Provisioning.xml)          | [provisioning.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/provisioning/wsdl/provisioning.wsdl)          | -                                     | ✅ Complete |
| PTZ                    | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/PTZ.xml)                    | [ptz.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/ptz/wsdl/ptz.wsdl)                   | -                                     | ✅ Complete |
| Receiver               | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Receiver.xml)               | [receiver.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/receiver.wsdl)              | -                                     | ✅ Complete |
| Recording Control      | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/RecordingControl.xml)       | [recording.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/recording.wsdl)             | -                                     | ✅ Complete |
| Recording Search       | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/RecordingSearch.xml)        | [search.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/search.wsdl)                | -                                     | ✅ Complete |
| Replay Control         | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Replay.xml)                 | [replay.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/replay.wsdl)                | -                                     | ✅ Complete |
| Resource Query         | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/ResourceQuery.xml)          | -                           |                                      | ❌ Any idea? |
| Schedule               | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Schedule.xml)               | [schedule.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schedule/wsdl/schedule.wsdl)              | -                                     | ✅ Complete |
| Security               | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Security.xml)               | [advancedsecurity.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/advancedsecurity/wsdl/advancedsecurity.wsdl)      | -                                     | ✅ Complete |
| Thermal                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Thermal.xml)                | [thermal.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/thermal/wsdl/thermal.wsdl)               | [radiometry.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/radiometry.xsd)                       | ✅ Complete |
| Uplink                 | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Uplink.xml)                 | [uplink.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/uplink/wsdl/uplink.wsdl)                | -                                     | ✅ Complete |
| WebRTC                 | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/WebRTC.xml)                 | -                            | -                                     | ❌ Any idea? |

## Service Bindings in ONVIF

ONVIF services are defined by WSDL bindings. In this library, there are two main patterns:

### 1. Single Binding Services

Most ONVIF services use a single binding, mapping directly to one endpoint. These are accessed via simple client methods, and the binding/xAddr is always known from device capabilities.

<details>
<summary>Examples:</summary>

```python
client.devicemgmt()   # DeviceBinding
client.media()        # MediaBinding
client.ptz()          # PTZBinding
...
```

✅ These are considered fixed and always accessed directly.

</details>

### 2. Multi-Binding Services

Some ONVIF services have multiple bindings in the same WSDL. These typically include:
- A **root binding** (main entry point)
- One or more **sub-bindings**, discovered or created dynamically (e.g. after subscription/configuration creation)

<details>
<summary>Examples:</summary>

1. **Events**
   - **Root:** `EventBinding`
   - **Sub-bindings:**
     - `PullPointSubscriptionBinding` (created via `CreatePullPointSubscription`)
     - `SubscriptionManagerBinding` (manages existing subscriptions)
     - `NotificationProducerBinding`

   Usage in library:
   ```python
   client.events()                    # root binding
   client.pullpoint(subscription)     # sub-binding (dynamic, via SubscriptionReference.Address)
   client.subscription(subscription)  # sub-binding (dynamic, via SubscriptionReference.Address)
   ```

2. **Security (Advanced Security)**
   - **Root:** `AdvancedSecurityServiceBinding`
   - **Sub-bindings:**
     - `AuthorizationServerBinding`
     - `KeystoreBinding`
     - `CredentialBinding`
     - `JWTBinding`
     - `Dot1XBinding`
     - `TLSServerBinding`
     - `MediaSigningBinding`

   Usage in library:
   ```python
   client.security()                  # root binding
   client.authorizationserver(xaddr)  # sub-binding accessor (requires xAddr)
   client.keystore(xaddr)
   client.jwt(xaddr)
   client.dot1x(xaddr)
   client.tlsserver(xaddr)
   client.mediasigning(xaddr)
   ```

3. **Analytics**
   - **Root:** `AnalyticsEngineBinding`
   - **Sub-bindings:**
     - `RuleEngineBinding`

   Usage in library:
   ```python
   client.analytics()   # root binding
   client.ruleengine()  # sub-binding accessor
   ```
</details>

### Summary

- **Single binding services:** Always accessed directly (e.g. `client.media()`).
- **Multi-binding services:** Have a root + sub-binding(s). Root is fixed; sub-bindings may require dynamic creation or explicit xAddr (e.g. `client.pullpoint(subscription)`, `client.authorizationserver(xaddr)`).

## Future Improvements (Stay tuned and star ⭐ this repo)

- [ ] Implement structured data models for ONVIF Schemas using [xsdata](https://github.com/tefra/xsdata).
- [ ] Integrate [xmltodict](https://github.com/martinblech/xmltodict) for simplified XML parsing and conversion.
- [ ] Add functionality for `ONVIFClient` to accept a custom `wsdl_path` service.
- [ ] Enhance documentation with API references and diagrams (not from [AI Wiki](https://deepwiki.com/nirsimetri/onvif-python)).
- [ ] Add more usage examples for advanced features.
- [ ] Add benchmarking and performance metrics.
- [ ] Add community-contributed device configuration templates.
- [ ] Implement missing or partial ONVIF services.
- [ ] Add function to expose ONVIF devices (for debugging purposes by the community).

## Related Projects

- [onvif-products-directory](https://github.com/nirsimetri/onvif-products-directory):
	This project is a comprehensive ONVIF data aggregation and management suite, designed to help developers explore, analyze, and process ONVIF-compliant product information from hundreds of manufacturers worldwide. It provides a unified structure for device, client, and company data, making it easier to perform research, build integrations, and generate statistics for ONVIF ecosystem analysis.

- (soon) [onvif-rest-server](https://github.com/nirsimetri/onvif-rest-server):
	A RESTful API server for ONVIF devices, enabling easy integration of ONVIF device management, media streaming, and other capabilities into web applications and services.

- (soon) [onvif-mcp](https://github.com/nirsimetri/onvif-mcp):
	A Model Context Protocol (MCP) server for ONVIF, providing a unified API and context-based integration for ONVIF devices, clients, and services. It enables advanced automation, orchestration, and interoperability across ONVIF-compliant devices and clients.

## Alternatives

If you are looking for other ONVIF Python libraries, here are some alternatives:

- [python-onvif-zeep](https://github.com/FalkTannhaeuser/python-onvif-zeep):
	A synchronous ONVIF client library for Python, using Zeep for SOAP communication. Focuses on compatibility and ease of use for standard ONVIF device operations. Good for scripts and applications where async is not required.

- [python-onvif-zeep-async](https://github.com/openvideolibs/python-onvif-zeep-async):
	An asynchronous ONVIF client library for Python, based on Zeep and asyncio. Suitable for applications requiring non-blocking operations and concurrent device communication. Supports many ONVIF services and is actively maintained.

## References
- [ONVIF Official Specifications](https://www.onvif.org/profiles/specifications/specification-history/)
- [ONVIF Official Specs Repository](https://github.com/onvif/specs)
- [ONVIF 2.0 Service Operation Index](https://www.onvif.org/onvif/ver20/util/operationIndex.html)
- [Usage Examples](./examples/)

## Legal Notice

This project is an **independent open-source implementation** of the [ONVIF](https://www.onvif.org) specifications. It is **not affiliated with, endorsed by, or sponsored by ONVIF** or its member companies.

- The name **“ONVIF”** and the ONVIF logo are registered trademarks of the ONVIF organization.  
- Any references to ONVIF within this project are made strictly for the purpose of describing interoperability with ONVIF-compliant devices and services.  
- Use of the ONVIF trademark in this repository is solely nominative and does not imply any partnership, certification, or official status.
- This project includes WSDL/XSD/HTML files from the official ONVIF specifications.
- These files are © ONVIF and are redistributed here for interoperability purposes.
- All rights to the ONVIF specifications are reserved by ONVIF.

If you require certified ONVIF-compliant devices or clients, please refer to the official [ONVIF conformant product list](https://www.onvif.org/conformant-products/). For authoritative reference and the latest official ONVIF specifications, please consult the [ONVIF Official Specifications](https://www.onvif.org/profiles/specifications/specification-history/).

Use of this library is at your own risk. The authors and contributors assume no liability for any damages, direct or indirect, arising from its use.

## License

This project is licensed under the MIT License. See [LICENSE](./LICENSE.md) for details.