# ONVIF Python

[![License](https://img.shields.io/badge/License-MIT-blue)](https://github.com/nirsimetri/onvif-python?tab=MIT-1-ov-file)
[![DeepWiki](https://img.shields.io/badge/DeepWiki-AI%20Wiki-orange)](https://deepwiki.com/nirsimetri/onvif-python)
[![Release](https://img.shields.io/badge/Release-v0.1.0-red?logo=archive)](https://github.com/nirsimetri/onvif-python/releases)
<br>
[![PyPI](https://img.shields.io/badge/PyPI-0.1.0-yellow?logo=archive)](https://pypi.org/project/onvif-python/)
[![Downloads](https://img.shields.io/pypi/dm/onvif-python?label=PyPI%20Downloads)](https://clickpy.clickhouse.com/dashboard/onvif-python)

**This project provides a comprehensive and developer-friendly Python library for working with ONVIF-compliant devices.** It is designed to be reliable, easy to integrate, and flexible enough to support a wide range of ONVIF profiles and services.  

**[ONVIF](https://www.onvif.org) (Open Network Video Interface Forum)** is a global standard for the interface of IP-based physical security products, including network cameras, video recorders, and related systems.  

Behind the scenes, ONVIF communication relies on **[SOAP](https://en.wikipedia.org/wiki/SOAP) (Simple Object Access Protocol)** — an [XML](https://en.wikipedia.org/wiki/XML)-based messaging protocol with strict schema definitions ([WSDL](https://en.wikipedia.org/wiki/Web_Services_Description_Language)/[XSD](https://en.wikipedia.org/wiki/XML_Schema_(W3C))). SOAP ensures interoperability, but when used directly it can be verbose, complex, and error-prone.  

This library simplifies that process by wrapping SOAP communication into a clean, Pythonic API. You no longer need to handle low-level XML parsing, namespaces, or security tokens manually — the library takes care of it, letting you focus on building functionality.  

## Key Features
- Full implementation of ONVIF core services and profiles  
- Support for device discovery, media streaming, PTZ control, event management, and more  
- Pythonic abstraction over SOAP requests and responses (no need to handcraft XML)  
- Extensible architecture for custom ONVIF extensions  
- Compatible with multiple ONVIF specification versions  
- Example scripts and tests included  

## Who Is It For?
- **Individual developers** exploring ONVIF or building hobby projects  
- **Companies** building video intelligence, analytics, or VMS platforms  
- **Security integrators** who need reliable ONVIF interoperability across devices

## Installation

From official [PyPI](https://pypi.org/project/onvif-python/):
```bash
pip install onvif-python
```
Or clone this repository and install locally:
```bash
git clone https://github.com/nirsimetri/onvif-python
cd onvif-python
pip install .
```

## Usage Example

> [!TIP]
> You can view the complete documentation automatically generated by DeepWiki via the [onvif-python AI Wiki](https://deepwiki.com/nirsimetri/onvif-python) link. We currently do not have an official documentation site. Help us create more examples and helpful documentation by [contributing](https://github.com/nirsimetri/onvif-python?tab=contributing-ov-file).

Below are simple examples to help you get started with the ONVIF Python library. These demonstrate how to connect to an ONVIF-compliant device and retrieve basic device information.

**1. Initialize the ONVIFClient**

Create an instance of `ONVIFClient` by providing your device's IP address, port, username, and password:

```python
from onvif import ONVIFClient

# Basic connection
client = ONVIFClient("192.168.1.17", 8000, "admin", "admin123")

# With custom WSDL directory (optional)
client = ONVIFClient(
    "192.168.1.17", 8000, "admin", "admin123",
    wsdl_dir="/path/to/custom/wsdl"  # Use custom WSDL files in this path
)
```

**2. Create Service Instance**

`ONVIFClient` provides several main services that can be accessed via the following methods:

- `client.devicemgmt()` — Device Management
- `client.events()` — Events
- `client.imaging()` — Imaging
- `client.media()` — Media
- `client.ptz()` — PTZ (Pan-Tilt-Zoom)
- `client.analytics()` — Analytics

and so on, check [Implemented ONVIF Services](https://github.com/nirsimetri/onvif-python?tab=readme-ov-file#implemented-onvif-services) for more details

Example usage:
```python
device = client.devicemgmt()      # Device Management (Core)
media = client.media()            # Media
```

**3. Get Device Information**

Retrieve basic information about the device, such as manufacturer, model, firmware version, and serial number using `devicemgmt()` service:

```python
info = device.GetDeviceInformation()
print(info)
# Example output: {'Manufacturer': '..', 'Model': '..', 'FirmwareVersion': '..', 'SerialNumber': '..'}
```

**4. Get RTSP URL**

Retrieve the RTSP stream URL for live video streaming from the device using `media()` service:

```python
profile = media.GetProfiles()[0]  # use the first profile
stream = media.GetStreamUri(
    ProfileToken=profile.token, 
	StreamSetup={"Stream": "RTP-Unicast", "Transport": {"Protocol": "RTSP"}}
)
print(stream)
# Example output: {'Uri': 'rtsp://192.168.1.17:8554/Streaming/Channels/101', ...}
```

Explore more advanced usage and service-specific operations in the [`examples/`](./examples/) folder.

> [!IMPORTANT]
> If you're new to ONVIF and want to learn more, we highly recommend taking the official free online course provided by ONVIF at [Introduction to ONVIF Course](https://www.onvif.org/about/introduction-to-onvif-course). Please note that we are not endorsed or sponsored by ONVIF, see [Legal Notice](#legal-notice) for details.

## ONVIFClient Parameters

The `ONVIFClient` class provides various configuration options to customize the connection behavior, caching strategy, security settings, and debugging capabilities. Below is a detailed description of all available parameters:


<details>
<summary><b>Basic Parameters</b></summary>

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `host` | `str` | ✅ Yes | - | IP address or hostname of the ONVIF device (e.g., `"192.168.1.17"`) |
| `port` | `int` | ✅ Yes | - | Port number for ONVIF service (common ports: `80`, `8000`, `8080`) |
| `username` | `str` | ✅ Yes | - | Username for device authentication (use digest authentication) |
| `password` | `str` | ✅ Yes | - | Password for device authentication |

</details>

<details>
<summary><b>Connection Parameters</b></summary>

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `timeout` | `int` | ❌ No | `10` | Connection timeout in seconds for SOAP requests |
| `use_https` | `bool` | ❌ No | `False` | Use HTTPS instead of HTTP for secure communication |
| `verify_ssl` | `bool` | ❌ No | `True` | Verify SSL certificates when using HTTPS (set to `False` for self-signed certificates) |

</details>

<details>
<summary><b>Caching Parameters</b></summary>

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `cache` | `CacheMode` | ❌ No | `CacheMode.ALL` | WSDL caching strategy (see [Cache Modes](#cache-modes) below) |

</details>

<details>
<summary><b>Feature Parameters</b></summary>

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `apply_patch` | `bool` | ❌ No | `True` | Enable zeep patching for better `xsd:any` field parsing and automatic flattening, applied at (>= [`v0.0.4`](https://github.com/nirsimetri/onvif-python/releases/tag/v0.0.4)) |
| `capture_xml` | `bool` | ❌ No | `False` | Enable XML capture plugin for debugging SOAP requests/responses, applied at (>= [`v0.0.6`](https://github.com/nirsimetri/onvif-python/releases/tag/v0.0.6)) |
| `wsdl_dir`    | `str`  | ❌ No | `None` | Custom WSDL directory path for using external WSDL files instead of built-in ones (e.g., `/path/to/custom/wsdl`), applied at (>= [`v0.1.0`](https://github.com/nirsimetri/onvif-python/releases/tag/v0.1.0)) |

</details>

<details>
<summary><b>Cache Modes</b></summary> 

The library provides four caching strategies via the `CacheMode` enum:

| Mode | Description | Best For | Startup Speed | Disk Usage | Memory Usage |
|------|-------------|----------|---------------|------------|--------------|
| `CacheMode.ALL` | In-memory + disk cache (SQLite) | Production servers, multi-device apps | Fast | High | High |
| `CacheMode.DB` | Disk cache only (SQLite) | Batch jobs, CLI tools | Medium | Medium | Low |
| `CacheMode.MEM` | In-memory cache only | Short-lived scripts, demos | Medium | None | Medium |
| `CacheMode.NONE` | No caching | Testing, debugging | Slow | None | Low |

**Recommendation:** Use `CacheMode.ALL` (default) for production applications to maximize performance.

</details>

<details>
<summary><b>Usage Examples</b></summary>

**Basic Connection:**
```python
from onvif import ONVIFClient

# Minimal configuration
client = ONVIFClient("192.168.1.17", 80, "admin", "password")
```

**Secure Connection (HTTPS):**
```python
from onvif import ONVIFClient

# Connect via HTTPS with custom timeout
client = ONVIFClient(
    "your-cctv-node.viewplexus.com", 
    443,  # HTTPS port
    "admin", 
    "password",
    timeout=30,
    use_https=True
)
```

**Performance Optimized (Memory Cache):**
```python
from onvif import ONVIFClient, CacheMode

# Use memory-only cache for quick scripts
client = ONVIFClient(
    "192.168.1.17", 
    80, 
    "admin", 
    "password",
    cache=CacheMode.MEM
)
```

**No Caching and No Zeep Patching (Testing):**
```python
from onvif import ONVIFClient, CacheMode

# Disable all caching for testing
client = ONVIFClient(
    "192.168.1.17", 
    80, 
    "admin", 
    "password",
    cache=CacheMode.NONE,
    apply_patch=False  # Use original zeep behavior
)
```

**Debugging Mode (XML Capture):**
```python
from onvif import ONVIFClient

# Enable XML capture for debugging
client = ONVIFClient(
    "192.168.1.17", 
    80, 
    "admin", 
    "password",
    capture_xml=True  # Captures all SOAP requests/responses
)

# Make some ONVIF calls
device = client.devicemgmt()
info = device.GetDeviceInformation()
services = device.GetCapabilities()

# Access the XML capture plugin
if client.xml_plugin:
    # Get last captured request/response
    print("Last Request XML:")
    print(client.xml_plugin.last_sent_xml)
    
    print("\nLast Response XML:")
    print(client.xml_plugin.last_received_xml)
    
    print(f"\nLast Operation: {client.xml_plugin.last_operation}")
    
    # Get complete history of all requests/responses
    print(f"\nTotal captured operations: {len(client.xml_plugin.history)}")
    for item in client.xml_plugin.history:
        print(f"  - {item['operation']} ({item['type']})")
    
    # Save captured XML to files
    client.xml_plugin.save_to_file(
        request_file="last_request.xml",
        response_file="last_response.xml"
    )
    
    # Clear history when done
    client.xml_plugin.clear_history()
```

> **XML Capture Plugin Methods:**
> - `last_sent_xml` - Get the last SOAP request XML
> - `last_received_xml` - Get the last SOAP response XML
> - `last_operation` - Get the name of the last operation
> - `history` - List of all captured requests/responses with metadata
> - `get_last_request()` - Method to get last request
> - `get_last_response()` - Method to get last response
> - `get_history()` - Method to get all history
> - `save_to_file(request_file, response_file)` - Save XML to files
> - `clear_history()` - Clear captured history

**Custom WSDL Directory:**
```python
from onvif import ONVIFClient

# Use custom WSDL files instead of built-in ones
client = ONVIFClient(
    "192.168.1.17", 
    80, 
    "admin", 
    "password",
    wsdl_dir="/path/to/custom/wsdl"  # Custom WSDL directory
)

# All services will automatically use custom WSDL files
device = client.devicemgmt()
media = client.media()
ptz = client.ptz()

# The custom WSDL directory should have a flat structure:
# /path/to/custom/wsdl/
# ├── devicemgmt.wsdl
# ├── media.wsdl
# ├── ptz.wsdl
# ├── imaging.wsdl
# └── ... (other WSDL files)
```

</details>

<details>
<summary><b>Production Configuration</b></summary>

```python
from onvif import ONVIFClient, CacheMode

# Recommended production settings
client = ONVIFClient(
    host="your-cctv-node.viewplexus.com",
    port=443,
    username="admin",
    password="secure_password",
    timeout=15,
    cache=CacheMode.ALL,        # Maximum performance (default)
    use_https=True,             # Secure communication
    verify_ssl=True,            # Verify certificates (default)
    apply_patch=True,           # Enhanced parsing (default)
    capture_xml=False,          # Disable debug mode (default)
    wsdl_dir=None               # Use built-in WSDL files (default)
)
```
</details>

### Notes

- **Authentication:** This library uses **WS-UsernameToken with Digest** authentication by default, which is the standard for ONVIF devices.
- **Patching:** The `apply_patch=True` (default) enables custom zeep patching that improves `xsd:any` field parsing. This is recommended for better compatibility with ONVIF responses.
- **XML Capture:** Only use `capture_xml=True` during development/debugging as it increases memory usage and may expose sensitive data in logs.
- **Custom WSDL:** Use `wsdl_dir` parameter to specify a custom directory containing WSDL files. The directory should have a flat structure with WSDL files directly in the root (e.g., `/path/to/custom/wsdl/devicemgmt.wsdl`, `/path/to/custom/wsdl/media.wsdl`, etc.).
- **Cache Location:** Disk cache (when using `CacheMode.DB` or `CacheMode.ALL`) is stored in `~/.onvif-python/onvif_zeep_cache.sqlite`.

## Service Discovery: Understanding Device Capabilities

> [!WARNING]
> Before performing any operations on an ONVIF device, it is highly recommended to discover which services are available and supported by the device. This library automatically performs comprehensive service discovery during initialization using a robust fallback mechanism.

**Why discover device services?**

- **Device Diversity:** Not all ONVIF devices support every service. Available services may vary by manufacturer, model, firmware, or configuration.
- **Error Prevention:** Attempting to use unsupported services can result in failed requests, exceptions, or undefined behavior.
- **Dynamic Feature Detection:** Devices may enable or disable services over time (e.g., after firmware updates or configuration changes).
- **Optimized Integration:** By checking available services, your application can adapt its workflow and UI to match the device's actual features.

**How service discovery works in this library:**

The `ONVIFClient` uses a **3-tier discovery approach** to maximize device compatibility:

1. **GetServices (Preferred)** - Tries `GetServices` first for detailed service information
2. **GetCapabilities (Fallback)** - Falls back to `GetCapabilities` if `GetServices` is not supported
3. **Default URLs (Final Fallback)** - Uses standard ONVIF URLs as last resort

```python
from onvif import ONVIFClient

client = ONVIFClient("192.168.1.17", 8000, "admin", "admin123")

# Check what discovery method was used
if client.services:
    print("Service discovery: GetServices (preferred)")
    print("Discovered services:", len(client.services))
    print("Service map:", client._service_map)
elif client.capabilities:
    print("Service discovery: GetCapabilities (fallback)")
    print("Available capabilities:", client.capabilities)
else:
    print("Service discovery: Using default URLs")
```

**Why this approach?**

- **GetServices** provides the most accurate and detailed service information, but it's **optional** in the ONVIF specification
- **GetCapabilities** is **mandatory** for all ONVIF-compliant devices, ensuring broader compatibility
- **Default URLs** guarantee basic connectivity even with non-compliant devices

**Get detailed service information:**

If you need comprehensive service details, you can manually call `GetServices` with capabilities:

```python
device = client.devicemgmt()

try:
    # Try to get detailed service information
    services = device.GetServices(IncludeCapability=True)
    for service in services:
        print(f"Service: {service.Namespace}")
        print(f"XAddr: {service.XAddr}")
        if hasattr(service, 'Capabilities'):
            print(f"Capabilities: {service.Capabilities}")
except Exception:
    # Fallback to GetCapabilities for legacy devices
    capabilities = device.GetCapabilities()
    print("Capabilities:", capabilities)
```

**Access capabilities information:**

When using `GetCapabilities` fallback, you can access capability information:

```python
device = client.devicemgmt()

try:
    # Get capabilities information from device
    capabilities = device.GetCapabilities()
    
    # Main services (always available)
    print("Media XAddr:", getattr(capabilities.Media, 'XAddr', 'Not available'))
    print("PTZ XAddr:", getattr(capabilities.PTZ, 'XAddr', 'Not available'))
    
    # Extension services (device-dependent)
    ext = getattr(capabilities, 'Extension', None)
    if ext:
        print("DeviceIO XAddr:", getattr(ext.DeviceIO, 'XAddr', 'Not available'))
        print("Recording XAddr:", getattr(ext.Recording, 'XAddr', 'Not available'))
        print("Search XAddr:", getattr(ext.Search, 'XAddr', 'Not available'))
        print("Replay XAddr:", getattr(ext.Replay, 'XAddr', 'Not available'))
except Exception as e:
    print(f"Error getting capabilities: {e}")
```

> [!TIP]
> The library handles service discovery automatically with intelligent fallback. You typically don't need to call discovery methods manually unless you need detailed capability information or want to refresh the service list after device configuration changes.

## Tested Devices

This library has been tested with a variety of ONVIF-compliant devices. For the latest and most complete list of devices that have been verified to work with this library, please refer to:

- [List of tested devices (device-test)](https://github.com/nirsimetri/onvif-products-directory/blob/main/device-test)

If your device is not listed right now, feel free to contribute your test results or feedback via Issues or Discussions at [onvif-products-directory](https://github.com/nirsimetri/onvif-products-directory). Your contribution will be invaluable to the community and the public.

> [!IMPORTANT]
> Device testing contributions must be made with a real device and use the scripts provided in the [onvif-products-directory](https://github.com/nirsimetri/onvif-products-directory) repo. Please be sure to contribute using a device model not already listed.

## Supported ONVIF Profiles

This library fully supports all major ONVIF Profiles listed below. Each profile represents a standardized set of features and use cases, ensuring interoperability between ONVIF-compliant devices and clients. You can use this library to integrate with devices and systems that implement any of these profiles.

| Name      | Specifications | Main Features | Typical Use Case | Support |
|-----------|----------------|---------------|------------------|---------|
| Profile_S | [Document](https://www.onvif.org/wp-content/uploads/2019/12/ONVIF_Profile_-S_Specification_v1-3.pdf) | Video streaming, PTZ, audio, multicasting | Network video transmitters (cameras) and receivers (recorders, VMS) | ✅ Yes |
| Profile_G | [Document](https://www.onvif.org/wp-content/uploads/2017/01/ONVIF_Profile_G_Specification_v1-0.pdf) | Recording, search, replay, video storage | Video recorders, storage devices | ✅ Yes |
| Profile_T | [Document](https://www.onvif.org/wp-content/uploads/2018/09/ONVIF_Profile_T_Specification_v1-0.pdf) | Advanced video streaming (H.265, analytics metadata, motion detection) | Modern cameras and clients | ✅ Yes |
| Profile_C | [Document](https://www.onvif.org/wp-content/uploads/2017/01/2013_12_ONVIF_Profile_C_Specification_v1-0.pdf) | Access control, door monitoring | Door controllers, access systems | ✅ Yes |
| Profile_A | [Document](https://www.onvif.org/wp-content/uploads/2017/06/ONVIF_Profile_A_Specification_v1-0.pdf) | Advanced access control configuration, credential management | Access control clients and devices | ✅ Yes |
| Profile_D | [Document](https://www.onvif.org/wp-content/uploads/2021/06/onvif-profile-d-specification-v1-0.pdf) | Access control peripherals (locks, sensors, relays) | Peripheral devices for access control | ✅ Yes |
| Profile_M | [Document](https://www.onvif.org/wp-content/uploads/2024/04/onvif-profile-m-specification-v1-1.pdf) | Metadata, analytics events, object detection | Analytics devices, metadata clients | ✅ Yes |

For a full description of each profile and its features, visit [ONVIF Profiles](https://www.onvif.org/profiles/).

## Implemented ONVIF Services

> [!NOTE]
> For details about the available service functions and methods already implemented in this library, see the source code in [`onvif/services/`](./onvif/services). Or if you want to read in a more proper format visit [onvif-python AI Wiki](https://deepwiki.com/nirsimetri/onvif-python).

Below is a list of ONVIF services implemented and supported by this library, along with links to the official specifications, service definitions, and schema files as referenced from the [ONVIF Developer Specs](https://developer.onvif.org/pub/specs/branches/development/doc/index.html). This table provides a quick overview of the available ONVIF features and their technical documentation for integration and development purposes.

| Service                | Specifications                | Service Definitions         | Schema Files                        | Status     |
|------------------------|-------------------------------|-----------------------------|-------------------------------------|------------|
| Device Management      | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Core.xml)                   | [device.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/device/wsdl/devicemgmt.wsdl)    | [onvif.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/onvif.xsd) <br> [common.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/common.xsd)                | ✅ Complete |
| Events                 | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Core.xml)                   | [event.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/events/wsdl/event.wsdl)    | [onvif.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/onvif.xsd) <br> [common.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schema/common.xsd)                | ⚠️ Partial |
| Access Control         | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AccessControl.xml)         | [accesscontrol.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/pacs/accesscontrol.wsdl)         | [types.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/pacs/types.xsd)                            | ✅ Complete |
| Access Rules           | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AccessRules.xml)           | [accessrules.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/accessrules/wsdl/accessrules.wsdl)           | -                                      | ✅ Complete |
| Action Engine          | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/ActionEngine.xml)          | [actionengine.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/actionengine.wsdl)          | -                                      | ✅ Complete |
| Analytics              | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Analytics.xml)             | [analytics.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/wsdl/analytics.wsdl)             | [rules.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/rules.xsd) <br> [humanbody.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/humanbody.xsd) <br> [humanface.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/humanface.xsd) | ✅ Complete |
| Application Management | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AppMgmt.xml)               | [appmgmt.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/appmgmt/wsdl/appmgmt.wsdl)               | -                                     | ✅ Complete |
| Authentication Behavior| [Document](https://developer.onvif.org/pub/specs/branches/development/doc/AuthenticationBehavior.xml) | [authenticationbehavior.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/authenticationbehavior/wsdl/authenticationbehavior.wsdl) | -                                     | ✅ Complete |
| Cloud Integration      | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/CloudIntegration.xml)      | [cloudintegration.yaml](https://developer.onvif.org/pub/specs/branches/development/doc/yaml.php?yaml=cloudintegration.yaml)      | -                                     | ❌ Not yet |
| Credential             | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Credential.xml)            | [credential.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/credential/wsdl/credential.wsdl)            | -                                     | ✅ Complete |
| Device IO              | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/DeviceIo.xml)              | [deviceio.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/deviceio.wsdl)              |-                                      | ✅ Complete |
| Display                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Display.xml)               | [display.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/display.wsdl)               | -                                     | ✅ Complete |
| Door Control           | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/DoorControl.xml)           | [doorcontrol.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/pacs/doorcontrol.wsdl)           | -                                     | ✅ Complete |
| Imaging                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Imaging.xml)               | [imaging.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/imaging/wsdl/imaging.wsdl)               | -                                     | ✅ Complete |
| Media                  | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Media.xml)                 | [media.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/media/wsdl/media.wsdl)                 | -                                     | ✅ Complete |
| Media 2                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Media2.xml)                | [media2.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/media/wsdl/media.wsdl)                | -                                     | ✅ Complete |
| Provisioning           | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Provisioning.xml)          | [provisioning.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/provisioning/wsdl/provisioning.wsdl)          | -                                     | ✅ Complete |
| PTZ                    | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/PTZ.xml)                    | [ptz.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/ptz/wsdl/ptz.wsdl)                   | -                                     | ✅ Complete |
| Receiver               | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Receiver.xml)               | [receiver.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/receiver.wsdl)              | -                                     | ✅ Complete |
| Recording Control      | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/RecordingControl.xml)       | [recording.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/recording.wsdl)             | -                                     | ✅ Complete |
| Recording Search       | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/RecordingSearch.xml)        | [search.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/search.wsdl)                | -                                     | ✅ Complete |
| Replay Control         | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Replay.xml)                 | [replay.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/replay.wsdl)                | -                                     | ✅ Complete |
| Resource Query         | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/ResourceQuery.xml)          | -                           |                                      | ❌ Any idea? |
| Schedule               | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Schedule.xml)               | [schedule.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/schedule/wsdl/schedule.wsdl)              | -                                     | ✅ Complete |
| Security               | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Security.xml)               | [advancedsecurity.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/advancedsecurity/wsdl/advancedsecurity.wsdl)      | -                                     | ✅ Complete |
| Thermal                | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Thermal.xml)                | [thermal.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/thermal/wsdl/thermal.wsdl)               | [radiometry.xsd](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver20/analytics/radiometry.xsd)                       | ✅ Complete |
| Uplink                 | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/Uplink.xml)                 | [uplink.wsdl](https://developer.onvif.org/pub/specs/branches/development/wsdl/ver10/uplink/wsdl/uplink.wsdl)                | -                                     | ✅ Complete |
| WebRTC                 | [Document](https://developer.onvif.org/pub/specs/branches/development/doc/WebRTC.xml)                 | -                            | -                                     | ❌ Any idea? |

## Service Bindings in ONVIF

ONVIF services are defined by WSDL bindings. In this library, there are two main patterns:

### 1. Single Binding Services

Most ONVIF services use a single binding, mapping directly to one endpoint. These are accessed via simple client methods, and the binding/xAddr is always known from device capabilities.

<details>
<summary>Examples:</summary>

```python
client.devicemgmt()   # DeviceBinding
client.media()        # MediaBinding
client.ptz()          # PTZBinding
...
```

✅ These are considered fixed and always accessed directly.

</details>

### 2. Multi-Binding Services

Some ONVIF services have multiple bindings in the same WSDL. These typically include:
- A **root binding** (main entry point)
- One or more **sub-bindings**, discovered or created dynamically (e.g. after subscription/configuration creation)

<details>
<summary>Examples:</summary>

1. **Events**
   - **Root:** `EventBinding`
   - **Sub-bindings:**
     - `PullPointSubscriptionBinding` (created via `CreatePullPointSubscription`)
     - `SubscriptionManagerBinding` (manages existing subscriptions)
     - `NotificationProducerBinding`

   Usage in library:
   ```python
   client.events()                    # root binding
   client.pullpoint(subscription)     # sub-binding (dynamic, via SubscriptionReference.Address)
   client.subscription(subscription)  # sub-binding (dynamic, via SubscriptionReference.Address)
   client.notification()              # sub-binding accessor
   ```

2. **Security (Advanced Security)**
   - **Root:** `AdvancedSecurityServiceBinding`
   - **Sub-bindings:**
     - `AuthorizationServerBinding`
     - `KeystoreBinding`
     - `JWTBinding`
     - `Dot1XBinding`
     - `TLSServerBinding`
     - `MediaSigningBinding`

   Usage in library:
   ```python
   client.security()                  # root binding
   client.authorizationserver(xaddr)  # sub-binding accessor (requires xAddr)
   client.keystore(xaddr)             # ..
   client.jwt(xaddr)
   client.dot1x(xaddr)
   client.tlsserver(xaddr)
   client.mediasigning(xaddr)
   ```

3. **Analytics**
   - **Root:** `AnalyticsEngineBinding`
   - **Sub-bindings:**
     - `RuleEngineBinding`

   Usage in library:
   ```python
   client.analytics()   # root binding
   client.ruleengine()  # sub-binding accessor
   ```
</details>

### Summary

- **Single binding services:** Always accessed directly (e.g. `client.media()`).
- **Multi-binding services:** Have a root + sub-binding(s). Root is fixed; sub-bindings may require dynamic creation or explicit xAddr (e.g. `client.pullpoint(subscription)`, `client.authorizationserver(xaddr)`).

## Future Improvements (Stay tuned and star ⭐ this repo)

- [x] ~~Add debugging mode with raw xml on SOAP requests and responses.~~ ([c258162](https://github.com/nirsimetri/onvif-python/commit/c258162))
- [x] ~~Add functionality for `ONVIFClient` to accept a custom `wsdl_dir` service.~~ ([65f2570](https://github.com/nirsimetri/onvif-python/commit/65f257092e4c9daa23dd0d00825ed38a45d23b70))
- [ ] Add `ONVIF CLI` program to interact directly with ONVIF devices via terminal.
- [ ] Add asynchronous (async/await) support for non-blocking ONVIF operations and concurrent device communication.
- [ ] Implement structured data models for ONVIF Schemas using [xsdata](https://github.com/tefra/xsdata).
- [ ] Integrate [xmltodict](https://github.com/martinblech/xmltodict) for simplified XML parsing and conversion.
- [ ] Enhance documentation with API references and diagrams (not from [AI Wiki](https://deepwiki.com/nirsimetri/onvif-python)).
- [ ] Add more usage examples for advanced features.
- [ ] Add benchmarking and performance metrics.
- [ ] Add community-contributed device configuration templates.
- [ ] Implement missing or partial ONVIF services.
- [ ] Add function to expose ONVIF devices (for debugging purposes by the community).

## Related Projects

- [onvif-products-directory](https://github.com/nirsimetri/onvif-products-directory):
	This project is a comprehensive ONVIF data aggregation and management suite, designed to help developers explore, analyze, and process ONVIF-compliant product information from hundreds of manufacturers worldwide. It provides a unified structure for device, client, and company data, making it easier to perform research, build integrations, and generate statistics for ONVIF ecosystem analysis.

- (soon) [onvif-rest-server](https://github.com/nirsimetri/onvif-rest-server):
	A RESTful API server for ONVIF devices, enabling easy integration of ONVIF device management, media streaming, and other capabilities into web applications and services.

- (soon) [onvif-mcp](https://github.com/nirsimetri/onvif-mcp):
	A Model Context Protocol (MCP) server for ONVIF, providing a unified API and context-based integration for ONVIF devices, clients, and services. It enables advanced automation, orchestration, and interoperability across ONVIF-compliant devices and clients.

## Alternatives

If you are looking for other ONVIF Python libraries, here are some alternatives:

- [python-onvif-zeep](https://github.com/FalkTannhaeuser/python-onvif-zeep):
	A synchronous ONVIF client library for Python, using Zeep for SOAP communication. Focuses on compatibility and ease of use for standard ONVIF device operations. Good for scripts and applications where async is not required.

- [python-onvif-zeep-async](https://github.com/openvideolibs/python-onvif-zeep-async):
	An asynchronous ONVIF client library for Python, based on Zeep and asyncio. Suitable for applications requiring non-blocking operations and concurrent device communication. Supports many ONVIF services and is actively maintained.

## References
- [ONVIF Official Specifications](https://www.onvif.org/profiles/specifications/specification-history/)
- [ONVIF Official Specs Repository](https://github.com/onvif/specs)
- [ONVIF 2.0 Service Operation Index](https://www.onvif.org/onvif/ver20/util/operationIndex.html)
- [Usage Examples](./examples/)

## Legal Notice

This project is an **independent open-source implementation** of the [ONVIF](https://www.onvif.org) specifications. It is **not affiliated with, endorsed by, or sponsored by ONVIF** or its member companies.

- The name **“ONVIF”** and the ONVIF logo are registered trademarks of the ONVIF organization.  
- Any references to ONVIF within this project are made strictly for the purpose of describing interoperability with ONVIF-compliant devices and services.  
- Use of the ONVIF trademark in this repository is solely nominative and does not imply any partnership, certification, or official status.
- This project includes WSDL/XSD/HTML files from the official ONVIF specifications.
- These files are © ONVIF and are redistributed here for interoperability purposes.
- All rights to the ONVIF specifications are reserved by ONVIF.

If you require certified ONVIF-compliant devices or clients, please refer to the official [ONVIF conformant product list](https://www.onvif.org/conformant-products/). For authoritative reference and the latest official ONVIF specifications, please consult the [ONVIF Official Specifications](https://www.onvif.org/profiles/specifications/specification-history/).

## License

This project is licensed under the MIT License. See [LICENSE](./LICENSE.md) for details.