import inspect
import json
import openai
from functools import wraps
import re

class OpenAIWrapper:
    """Central class to manage OpenAI/Gemini configuration"""
    api_key = None
    base_url = None
    
    @classmethod
    def set_api_key(cls, api_key: str) -> None:
        """Set the API key globally (works for both OpenAI and Gemini)"""
        cls.api_key = api_key
        openai.api_key = api_key
        
    @classmethod
    def set_base_url(cls, base_url: str) -> None:
        """Set a custom base URL for the API"""
        cls.base_url = base_url
        openai.base_url = base_url

class FunctionWrapper:
    """
    Wraps a function to enable API function tool creation and execution.
    When the function is called without arguments, it uses the API to gather parameters.
    """
    def __init__(self, func, **config):
        self.func = func
        self.config = config
        self.params_schema = self._create_parameters_schema()
        self.tool_schema = {
            "name": func.__name__,
            "description": inspect.getdoc(func) or "",
            "parameters": self.params_schema
        }
        
        # Preserve function metadata
        wraps(func)(self)
        
        # Check for API key if needed
        if not openai.api_key and not self.config.get('api_key'):
            raise ValueError("API key not set. Use OpenAIWrapper.set_api_key() or pass api_key in decorator.")

    def _parse_docstring(self):
        """Parse docstring to extract parameter descriptions"""
        doc = inspect.getdoc(self.func) or ""
        param_descriptions = {}
        
        # Parse :param <name>: <description> format
        pattern = r":param\s+(?P<name>\w+):\s*(?P<desc>[^\n]*)"
        matches = re.findall(pattern, doc)
        for match in matches:
            param_descriptions[match[0]] = match[1]
        
        return param_descriptions

    def _create_parameters_schema(self):
        """Generate JSON schema for function parameters with descriptions"""
        sig = inspect.signature(self.func)
        properties = {}
        required = []
        
        # Get parameter descriptions from docstring
        param_descriptions = self._parse_docstring()
        
        for name, param in sig.parameters.items():
            # Map Python types to JSON schema types
            if param.annotation is int:
                prop_type = "integer"
            elif param.annotation is float:
                prop_type = "number"
            elif param.annotation is bool:
                prop_type = "boolean"
            elif param.annotation is list:
                prop_type = "array"
            elif param.annotation is dict:
                prop_type = "object"
            else:  # Default to string
                prop_type = "string"
                
            # Create property schema
            prop_schema = {"type": prop_type}
            
            # Add description if available
            if name in param_descriptions:
                prop_schema["description"] = param_descriptions[name]
                
            properties[name] = prop_schema
            
            # Mark as required if no default value
            if param.default is param.empty:
                required.append(name)
                
        schema = {
            "type": "object",
            "properties": properties,
            "required": required
        }
        return schema

    def __call__(self, *args, **kwargs):
        """Call the function normally if arguments are provided"""
        if args or kwargs:
            return self.func(*args, **kwargs)
        
        # Allow query via a keyword if function requires arguments
        if not self.params_schema["required"]:
            raise ValueError("Function must be called with arguments or via query()")
        return self.query(None)

    def query(self, context: str = None, model: str = None, **input_kwargs):
        """
        Execute the function using the API to gather parameters.
        Uses Google Gemini by default with the OpenAI-compatible endpoint.
        
        Args:
            context: User prompt to determine parameters
            model: Optional override of the model specified in decorator
            input_kwargs: Parameters starting with "input_" to be directly
                          fed into the model prompt

        Returns:
            Function result using parameters generated by the API
        """
        # Configure API key - use from config, then global
        api_key = self.config.get('api_key') or OpenAIWrapper.api_key
        if not api_key:
            raise ValueError("API key not set. Use OpenAIWrapper.set_api_key() or pass api_key in decorator.")
            
        # Use model from config unless overridden (default: gemini-1.5-flash-latest)
        model = model or self.config.get("model", "gemini-1.5-flash-latest")
        
        # Use base URL from config, then global
        base_url = self.config.get('base_url') or OpenAIWrapper.base_url
        
        # Default to Gemini OpenAI-compatible endpoint if not specified
        if not base_url and "gemini" in model:
            base_url = "https://generativelanguage.googleapis.com/v1beta/openai/"
        
        # Prepare message content for user input
        user_content = self.get_user_content_from_input(input_kwargs)
        
        # Add context as additional text input
        if context:
            user_content.append({"type": "text", "text": context})
        
        # Prepare request payload
        payload = {
            "model": model,
            "messages": [
                {"role": "system", "content": "Call function with parameters based on user input"},
                {"role": "user", "content": user_content}
            ],
            "tools": [{"type": "function", "function": self.tool_schema}],
            "tool_choice": {"type": "function", "function": {"name": self.func.__name__}},
            **self.config
        }

        # Make API request with API key and base URL
        client = openai.OpenAI(api_key=api_key, base_url=base_url)
        response = client.chat.completions.create(**payload)
        
        # Extract function call arguments
        message = response.choices[0].message
        if not message.tool_calls:
            raise ValueError("API did not call the function")
        
        function_call = message.tool_calls[0].function
        if function_call.name != self.func.__name__:
            raise ValueError(f"Unexpected function call: {function_call.name}")
        
        args_dict = json.loads(function_call.arguments)
        return self.func(**args_dict)

    def get_user_content_from_input(self, input_kwargs):
        user_content = []
        
        # Process input arguments starting with "input_"
        for key, value in input_kwargs.items():
            if key.startswith("input_"):
                # Extract content type from parameter name
                content_type = key[6:]  # Remove "input_" prefix
                
                # Create content object based on type
                if content_type == "text":
                    user_content.append({"type": "text", "text": value})
                elif content_type.endswith("_url"):
                    # Determine media type from content_type
                    media_type = content_type.replace("_url", "")
                    user_content.append({
                        "type": f"{media_type}_url",
                        f"{media_type}_url": {"url": value}
                    })
                elif content_type == "list":
                    for list_item in value:
                        user_content += self.get_user_content_from_input(list_item)
                else:
                    # Fallback to text representation
                    user_content.append({"type": "text", "text": f"{content_type}: {value}"})
        return user_content

def openai_function(func=None, **config):
    """
    Decorator to create API function tools from Python functions.
    Uses Google Gemini by default.
    
    Usage:
        @openai_function()
        def my_function(...):
            ...
            
    Args:
        func: Function to decorate
        config: Configuration for API calls (model, temperature, etc)
    """
    if func is None:
        return lambda f: openai_function(f, **config)
    return FunctionWrapper(func, **config)
