"""
Base client with common functionality for all Open-Meteo API clients.
"""

import requests
import time
import logging
from typing import Dict, List, Optional, Any, Union, Literal
from abc import ABC
from dataclasses import dataclass, field

from openmeteo.validators import (
    CoordinateValidator,
    DateValidator,
    ResponseValidator,
    InputSanitizer,
    ValidationError,
)

logger = logging.getLogger(__name__)

# Type alias for response format
ResponseFormat = Literal["json", "flatbuffers"]


@dataclass
class APIConfig:
    """
    Configuration for API client.

    Attributes:
        timeout: Request timeout in seconds
        retry_attempts: Number of retry attempts for failed requests
        retry_delay: Base delay between retries (multiplied by attempt number)
        api_key: API key for commercial use
        format: Response format ('json' or 'flatbuffers')
    """
    timeout: int = 30
    retry_attempts: int = 3
    retry_delay: float = 1.0
    api_key: Optional[str] = None
    format: ResponseFormat = "json"


class OpenMeteoBaseClient(ABC):
    """
    Abstract base client for all Open-Meteo API endpoints.

    Provides common functionality like request handling, retries, and validation.
    Supports both JSON and FlatBuffers response formats.

    FlatBuffers support requires installing with:
        pip install openmeteo-python[fast]
    """

    BASE_URL: str = ""

    def __init__(self, config: Optional[APIConfig] = None):
        """
        Initialize the base client.

        Args:
            config: API configuration options
        """
        self.config = config or APIConfig()
        self._session = requests.Session()
        self._flatbuffers_parser = None

        # Initialize FlatBuffers parser if needed
        if self.config.format == "flatbuffers":
            self._init_flatbuffers_parser()

    def _init_flatbuffers_parser(self) -> None:
        """Initialize the FlatBuffers parser (lazy loading)."""
        if self._flatbuffers_parser is None:
            try:
                from openmeteo.flatbuffers_parser import FlatBuffersParser
                self._flatbuffers_parser = FlatBuffersParser()
            except ImportError as e:
                raise ImportError(
                    "FlatBuffers support requires additional dependencies. "
                    "Install with: pip install openmeteo-python[fast]"
                ) from e

    def _build_params(self, **kwargs) -> Dict[str, Any]:
        """
        Build request parameters, filtering out None values.

        Args:
            **kwargs: Parameter key-value pairs

        Returns:
            Dictionary of non-None parameters
        """
        params = {}
        for key, value in kwargs.items():
            if value is not None:
                if isinstance(value, (list, tuple)):
                    params[key] = ",".join(str(v) for v in value)
                elif isinstance(value, bool):
                    params[key] = str(value).lower()
                else:
                    params[key] = value

        if self.config.api_key:
            params["apikey"] = self.config.api_key

        # Add format parameter for FlatBuffers
        if self.config.format == "flatbuffers":
            params["format"] = "flatbuffers"

        return params

    def _make_request(
        self,
        url: str,
        params: Dict[str, Any],
        method: str = "GET"
    ) -> Dict[str, Any]:
        """
        Make HTTP request with retry logic.

        Args:
            url: API endpoint URL
            params: Request parameters
            method: HTTP method

        Returns:
            Response data (dict for JSON, parsed for FlatBuffers)

        Raises:
            requests.RequestException: If request fails after retries
            ValueError: If API returns an error response
        """
        last_exception = None

        for attempt in range(1, self.config.retry_attempts + 1):
            try:
                if method.upper() == "GET":
                    response = self._session.get(
                        url,
                        params=params,
                        timeout=self.config.timeout
                    )
                else:
                    response = self._session.post(
                        url,
                        json=params,
                        timeout=self.config.timeout
                    )

                response.raise_for_status()

                # Parse response based on format
                if self.config.format == "flatbuffers":
                    return self._parse_flatbuffers_response(response.content)
                else:
                    data = response.json()

                    # Check for API error response
                    if "error" in data and data.get("error"):
                        error_msg = data.get("reason", "Unknown API error")
                        raise ValueError(f"API Error: {error_msg}")

                    return data

            except requests.RequestException as e:
                last_exception = e
                if attempt < self.config.retry_attempts:
                    logger.warning(
                        f"Request failed (attempt {attempt}/{self.config.retry_attempts}): {e}"
                    )
                    time.sleep(self.config.retry_delay * attempt)
                else:
                    logger.error(f"Request failed after {self.config.retry_attempts} attempts")

            except ValueError:
                raise

        raise last_exception

    def _parse_flatbuffers_response(self, content: bytes) -> Dict[str, Any]:
        """
        Parse FlatBuffers response content.

        Args:
            content: Raw bytes from API response

        Returns:
            Dictionary with parsed data (compatible with JSON format)
        """
        if self._flatbuffers_parser is None:
            self._init_flatbuffers_parser()

        from openmeteo.flatbuffers_parser import response_to_dict

        responses = self._flatbuffers_parser.parse(content)

        if not responses:
            raise ValueError("Empty FlatBuffers response")

        # Return first response for single location
        # For multiple locations, this would need to be handled differently
        return response_to_dict(responses[0])

    def _make_request_raw(
        self,
        url: str,
        params: Dict[str, Any],
        method: str = "GET"
    ) -> bytes:
        """
        Make HTTP request and return raw bytes (for FlatBuffers).

        Args:
            url: API endpoint URL
            params: Request parameters
            method: HTTP method

        Returns:
            Raw response bytes

        Raises:
            requests.RequestException: If request fails after retries
        """
        last_exception = None

        for attempt in range(1, self.config.retry_attempts + 1):
            try:
                if method.upper() == "GET":
                    response = self._session.get(
                        url,
                        params=params,
                        timeout=self.config.timeout
                    )
                else:
                    response = self._session.post(
                        url,
                        json=params,
                        timeout=self.config.timeout
                    )

                response.raise_for_status()
                return response.content

            except requests.RequestException as e:
                last_exception = e
                if attempt < self.config.retry_attempts:
                    logger.warning(
                        f"Request failed (attempt {attempt}/{self.config.retry_attempts}): {e}"
                    )
                    time.sleep(self.config.retry_delay * attempt)
                else:
                    logger.error(f"Request failed after {self.config.retry_attempts} attempts")

        raise last_exception

    def _validate_coordinates(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]]
    ) -> None:
        """
        Validate latitude and longitude values.

        Args:
            latitude: Single value or list of latitudes
            longitude: Single value or list of longitudes

        Raises:
            ValidationError: If coordinates are invalid
        """
        CoordinateValidator.validate_coordinates(latitude, longitude)

    def _validate_date(self, date_str: str, param_name: str = "date") -> None:
        """
        Validate date string format (YYYY-MM-DD).

        Args:
            date_str: Date string to validate
            param_name: Parameter name for error messages

        Raises:
            ValidationError: If date format is invalid
        """
        DateValidator.parse_date(date_str, param_name)

    def _validate_date_range(
        self,
        start_date: str,
        end_date: str,
        min_date=None,
        max_date=None
    ) -> None:
        """
        Validate date range.

        Args:
            start_date: Start date string
            end_date: End date string
            min_date: Minimum allowed date
            max_date: Maximum allowed date

        Raises:
            ValidationError: If date range is invalid
        """
        DateValidator.validate_date_range(start_date, end_date, min_date, max_date)

    def _validate_response(self, data: Dict[str, Any]) -> None:
        """
        Validate response data structure.

        Args:
            data: Response data dictionary

        Raises:
            ValidationError: If response is invalid
        """
        ResponseValidator.validate_response_structure(
            data,
            required_fields=["latitude", "longitude"]
        )

    def _sanitize_timezone(self, timezone: str) -> str:
        """Sanitize timezone parameter."""
        return InputSanitizer.sanitize_timezone(timezone)

    def _sanitize_unit(self, unit: str, valid_units: List[str], default: str) -> str:
        """Sanitize unit parameter."""
        return InputSanitizer.sanitize_unit(unit, valid_units, default)

    def close(self):
        """Close the session."""
        self._session.close()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()


def is_flatbuffers_available() -> bool:
    """
    Check if FlatBuffers support is available.

    Returns:
        True if flatbuffers and numpy are installed
    """
    try:
        from openmeteo.flatbuffers_parser import FLATBUFFERS_AVAILABLE
        return FLATBUFFERS_AVAILABLE
    except ImportError:
        return False
