"""
Command-line interface for OpenMeteo Python.

Provides CLI access to all Open-Meteo APIs with CSV/JSON output support.
"""

import argparse
import sys
import json
from datetime import datetime, timedelta
from typing import Optional
import pandas as pd

from openmeteo import OpenMeteo
from openmeteo.base import APIConfig


def create_parser() -> argparse.ArgumentParser:
    """Create the main argument parser."""
    parser = argparse.ArgumentParser(
        prog="openmeteo",
        description="OpenMeteo Python - Access weather data from command line",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Get current weather for Berlin
  openmeteo forecast --lat 52.52 --lon 13.41 --current temperature_2m,weather_code

  # Get 7-day hourly forecast
  openmeteo forecast --lat 52.52 --lon 13.41 --hourly temperature_2m,precipitation -o forecast.csv

  # Get historical data for January 2024
  openmeteo historical --lat 52.52 --lon 13.41 --start 2024-01-01 --end 2024-01-31 --hourly temperature_2m

  # Get air quality data
  openmeteo air-quality --lat 52.52 --lon 13.41 --hourly pm10,pm2_5,european_aqi

  # Search for a location
  openmeteo geocoding --search "New York"

  # Get elevation
  openmeteo elevation --lat 52.52 --lon 13.41

  # Generate standalone DAG script
  openmeteo generate-dag --apis forecast,air_quality --locations "Berlin:52.52:13.41,Paris:48.85:2.35" -o my_dag.py

  # Generate DAG with custom schedule
  openmeteo generate-dag --apis historical --locations "NYC:40.71:-74.01" --schedule "@weekly" -o weekly_dag.py
        """
    )

    parser.add_argument(
        "--version",
        action="version",
        version="%(prog)s 1.0.0"
    )

    parser.add_argument(
        "--api-key",
        type=str,
        help="API key for commercial use"
    )

    parser.add_argument(
        "--timeout",
        type=int,
        default=30,
        help="Request timeout in seconds (default: 30)"
    )

    subparsers = parser.add_subparsers(dest="command", help="Available commands")

    # Forecast command
    forecast_parser = subparsers.add_parser(
        "forecast",
        help="Get weather forecast data"
    )
    add_location_args(forecast_parser)
    add_output_args(forecast_parser)
    forecast_parser.add_argument(
        "--hourly",
        type=str,
        help="Comma-separated hourly variables (e.g., temperature_2m,precipitation)"
    )
    forecast_parser.add_argument(
        "--daily",
        type=str,
        help="Comma-separated daily variables"
    )
    forecast_parser.add_argument(
        "--current",
        type=str,
        help="Comma-separated current variables"
    )
    forecast_parser.add_argument(
        "--forecast-days",
        type=int,
        default=7,
        help="Number of forecast days (default: 7)"
    )
    forecast_parser.add_argument(
        "--timezone",
        type=str,
        default="UTC",
        help="Timezone for timestamps (default: UTC)"
    )
    forecast_parser.add_argument(
        "--temperature-unit",
        type=str,
        choices=["celsius", "fahrenheit"],
        default="celsius",
        help="Temperature unit (default: celsius)"
    )

    # Historical command
    historical_parser = subparsers.add_parser(
        "historical",
        help="Get historical weather data"
    )
    add_location_args(historical_parser)
    add_output_args(historical_parser)
    historical_parser.add_argument(
        "--start",
        type=str,
        required=True,
        help="Start date (YYYY-MM-DD)"
    )
    historical_parser.add_argument(
        "--end",
        type=str,
        required=True,
        help="End date (YYYY-MM-DD)"
    )
    historical_parser.add_argument(
        "--hourly",
        type=str,
        help="Comma-separated hourly variables"
    )
    historical_parser.add_argument(
        "--daily",
        type=str,
        help="Comma-separated daily variables"
    )
    historical_parser.add_argument(
        "--timezone",
        type=str,
        default="UTC",
        help="Timezone for timestamps (default: UTC)"
    )
    historical_parser.add_argument(
        "--temperature-unit",
        type=str,
        choices=["celsius", "fahrenheit"],
        default="celsius",
        help="Temperature unit (default: celsius)"
    )

    # Air quality command
    air_parser = subparsers.add_parser(
        "air-quality",
        help="Get air quality data"
    )
    add_location_args(air_parser)
    add_output_args(air_parser)
    air_parser.add_argument(
        "--hourly",
        type=str,
        help="Comma-separated hourly variables (e.g., pm10,pm2_5,european_aqi)"
    )
    air_parser.add_argument(
        "--current",
        type=str,
        help="Comma-separated current variables"
    )
    air_parser.add_argument(
        "--forecast-days",
        type=int,
        default=5,
        help="Number of forecast days (default: 5)"
    )
    air_parser.add_argument(
        "--timezone",
        type=str,
        default="UTC",
        help="Timezone for timestamps (default: UTC)"
    )

    # Marine command
    marine_parser = subparsers.add_parser(
        "marine",
        help="Get marine weather data"
    )
    add_location_args(marine_parser)
    add_output_args(marine_parser)
    marine_parser.add_argument(
        "--hourly",
        type=str,
        help="Comma-separated hourly variables (e.g., wave_height,sea_surface_temperature)"
    )
    marine_parser.add_argument(
        "--daily",
        type=str,
        help="Comma-separated daily variables"
    )
    marine_parser.add_argument(
        "--current",
        type=str,
        help="Comma-separated current variables"
    )
    marine_parser.add_argument(
        "--forecast-days",
        type=int,
        default=5,
        help="Number of forecast days (default: 5)"
    )
    marine_parser.add_argument(
        "--timezone",
        type=str,
        default="UTC",
        help="Timezone for timestamps (default: UTC)"
    )

    # Flood command
    flood_parser = subparsers.add_parser(
        "flood",
        help="Get flood/river discharge data"
    )
    add_location_args(flood_parser)
    add_output_args(flood_parser)
    flood_parser.add_argument(
        "--daily",
        type=str,
        default="river_discharge",
        help="Comma-separated daily variables (default: river_discharge)"
    )
    flood_parser.add_argument(
        "--forecast-days",
        type=int,
        default=92,
        help="Number of forecast days (default: 92)"
    )
    flood_parser.add_argument(
        "--start",
        type=str,
        help="Start date for historical data (YYYY-MM-DD)"
    )
    flood_parser.add_argument(
        "--end",
        type=str,
        help="End date for historical data (YYYY-MM-DD)"
    )

    # Climate command
    climate_parser = subparsers.add_parser(
        "climate",
        help="Get climate change projection data"
    )
    add_location_args(climate_parser)
    add_output_args(climate_parser)
    climate_parser.add_argument(
        "--start",
        type=str,
        required=True,
        help="Start date (YYYY-MM-DD)"
    )
    climate_parser.add_argument(
        "--end",
        type=str,
        required=True,
        help="End date (YYYY-MM-DD)"
    )
    climate_parser.add_argument(
        "--models",
        type=str,
        default="EC_Earth3P_HR",
        help="Comma-separated climate models (default: EC_Earth3P_HR)"
    )
    climate_parser.add_argument(
        "--daily",
        type=str,
        default="temperature_2m_max,temperature_2m_min,precipitation_sum",
        help="Comma-separated daily variables"
    )
    climate_parser.add_argument(
        "--temperature-unit",
        type=str,
        choices=["celsius", "fahrenheit"],
        default="celsius",
        help="Temperature unit (default: celsius)"
    )

    # Ensemble command
    ensemble_parser = subparsers.add_parser(
        "ensemble",
        help="Get ensemble model forecast data"
    )
    add_location_args(ensemble_parser)
    add_output_args(ensemble_parser)
    ensemble_parser.add_argument(
        "--models",
        type=str,
        default="icon_seamless",
        help="Comma-separated ensemble models (default: icon_seamless)"
    )
    ensemble_parser.add_argument(
        "--hourly",
        type=str,
        help="Comma-separated hourly variables"
    )
    ensemble_parser.add_argument(
        "--daily",
        type=str,
        help="Comma-separated daily variables"
    )
    ensemble_parser.add_argument(
        "--forecast-days",
        type=int,
        default=7,
        help="Number of forecast days (default: 7)"
    )
    ensemble_parser.add_argument(
        "--timezone",
        type=str,
        default="UTC",
        help="Timezone for timestamps (default: UTC)"
    )

    # Geocoding command
    geocoding_parser = subparsers.add_parser(
        "geocoding",
        help="Search for locations"
    )
    geocoding_parser.add_argument(
        "--search",
        type=str,
        required=True,
        help="Location name to search"
    )
    geocoding_parser.add_argument(
        "--count",
        type=int,
        default=10,
        help="Maximum number of results (default: 10)"
    )
    geocoding_parser.add_argument(
        "--language",
        type=str,
        default="en",
        help="Result language (default: en)"
    )
    geocoding_parser.add_argument(
        "--country",
        type=str,
        help="Filter by country code (e.g., US, DE)"
    )
    add_output_args(geocoding_parser)

    # Elevation command
    elevation_parser = subparsers.add_parser(
        "elevation",
        help="Get elevation data"
    )
    add_location_args(elevation_parser)
    add_output_args(elevation_parser)

    # Generate DAG command
    generate_parser = subparsers.add_parser(
        "generate-dag",
        help="Generate standalone Python DAG script",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description="""
Generate a standalone Python script for fetching weather data.

The generated script:
- Works as a standalone Python script
- Works as an Airflow DAG (auto-detected)
- Has no dependency on openmeteo-python package
- Can be customized after generation

Available APIs: forecast, historical, air_quality, marine, flood, climate, ensemble
        """
    )
    generate_parser.add_argument(
        "--apis",
        type=str,
        required=True,
        help="Comma-separated list of APIs (forecast,historical,air_quality,marine,flood,climate,ensemble)"
    )
    generate_parser.add_argument(
        "--locations",
        type=str,
        required=True,
        help="Locations in format 'Name:lat:lon,Name2:lat2:lon2' (e.g., 'Berlin:52.52:13.41,Paris:48.85:2.35')"
    )
    generate_parser.add_argument(
        "-o", "--output",
        type=str,
        required=True,
        help="Output Python file path (e.g., my_weather_dag.py)"
    )
    generate_parser.add_argument(
        "--dag-id",
        type=str,
        help="Airflow DAG ID (auto-generated if not provided)"
    )
    generate_parser.add_argument(
        "--schedule",
        type=str,
        default="@daily",
        help="Cron schedule or Airflow preset (default: @daily)"
    )
    generate_parser.add_argument(
        "--output-dir",
        type=str,
        default="./output",
        help="Output directory for generated data files (default: ./output)"
    )
    generate_parser.add_argument(
        "--output-format",
        type=str,
        choices=["csv", "json"],
        default="csv",
        help="Output format for data files (default: csv)"
    )
    generate_parser.add_argument(
        "--no-airflow",
        action="store_true",
        help="Generate script without Airflow DAG wrapper"
    )

    # List APIs command
    list_apis_parser = subparsers.add_parser(
        "list-apis",
        help="List available APIs for DAG generation"
    )

    # Generate standalone script command (no Airflow)
    script_parser = subparsers.add_parser(
        "generate-script",
        help="Generate standalone Python script (no Airflow)",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description="""
Generate a simple standalone Python script for fetching weather data.

Unlike generate-dag, this creates a minimal script that:
- Runs directly with Python
- Has no Airflow dependencies or DAG wrapper
- Is simpler and easier to customize
- Perfect for cron jobs, one-off runs, or integration into other systems

Available APIs: forecast, historical, air_quality, marine, flood, climate, ensemble

Examples:
  # Simple forecast script
  openmeteo generate-script --api forecast --locations "Berlin:52.52:13.41" -o forecast.py

  # Historical data with date range options
  openmeteo generate-script --api historical --locations "NYC:40.71:-74.01,LA:34.05:-118.24" -o historical.py

  # Multiple locations for air quality
  openmeteo generate-script --api air_quality --locations "Paris:48.85:2.35" -o air_quality.py --format json
        """
    )
    script_parser.add_argument(
        "--api",
        type=str,
        required=True,
        choices=["forecast", "historical", "air_quality", "marine", "flood", "climate", "ensemble"],
        help="API to use (forecast, historical, air_quality, marine, flood, climate, ensemble)"
    )
    script_parser.add_argument(
        "--locations",
        type=str,
        required=True,
        help="Locations in format 'Name:lat:lon,Name2:lat2:lon2' (e.g., 'Berlin:52.52:13.41')"
    )
    script_parser.add_argument(
        "-o", "--output",
        type=str,
        required=True,
        help="Output Python file path (e.g., my_script.py)"
    )
    script_parser.add_argument(
        "--output-dir",
        type=str,
        default="./output",
        help="Output directory for generated data files (default: ./output)"
    )
    script_parser.add_argument(
        "--format",
        type=str,
        choices=["csv", "json"],
        default="csv",
        help="Output format for data files (default: csv)"
    )

    return parser


def add_location_args(parser: argparse.ArgumentParser):
    """Add common location arguments."""
    location_group = parser.add_argument_group("location")
    location_group.add_argument(
        "--lat",
        type=float,
        help="Latitude (-90 to 90)"
    )
    location_group.add_argument(
        "--lon",
        type=float,
        help="Longitude (-180 to 180)"
    )
    location_group.add_argument(
        "--location",
        type=str,
        help="Location name (will be geocoded)"
    )


def add_output_args(parser: argparse.ArgumentParser):
    """Add common output arguments."""
    output_group = parser.add_argument_group("output")
    output_group.add_argument(
        "-o", "--output",
        type=str,
        help="Output file path (supports .csv and .json)"
    )
    output_group.add_argument(
        "--format",
        type=str,
        choices=["csv", "json", "table"],
        default="table",
        help="Output format (default: table)"
    )


def parse_variables(var_string: Optional[str]) -> Optional[list]:
    """Parse comma-separated variable string."""
    if var_string:
        return [v.strip() for v in var_string.split(",")]
    return None


def resolve_location(client: OpenMeteo, args) -> tuple:
    """Resolve location from args to coordinates."""
    if args.lat is not None and args.lon is not None:
        return args.lat, args.lon

    if hasattr(args, 'location') and args.location:
        coords = client.geocoding.get_coordinates(args.location)
        if coords:
            print(f"Resolved '{args.location}' to coordinates: {coords[0]}, {coords[1]}")
            return coords
        else:
            print(f"Error: Could not find location '{args.location}'", file=sys.stderr)
            sys.exit(1)

    print("Error: Either --lat/--lon or --location is required", file=sys.stderr)
    sys.exit(1)


def output_result(df: pd.DataFrame, args, default_name: str = "output"):
    """Output DataFrame in requested format."""
    # Determine output format
    output_format = args.format
    if args.output:
        if args.output.endswith(".csv"):
            output_format = "csv"
        elif args.output.endswith(".json"):
            output_format = "json"

    if output_format == "csv":
        if args.output:
            df.to_csv(args.output, index=False)
            print(f"Data saved to {args.output}")
        else:
            print(df.to_csv(index=False))
    elif output_format == "json":
        json_data = df.to_json(orient="records", date_format="iso")
        if args.output:
            with open(args.output, "w") as f:
                f.write(json_data)
            print(f"Data saved to {args.output}")
        else:
            print(json_data)
    else:
        print(df.to_string())


def cmd_forecast(client: OpenMeteo, args):
    """Handle forecast command."""
    lat, lon = resolve_location(client, args)

    hourly = parse_variables(args.hourly)
    daily = parse_variables(args.daily)
    current = parse_variables(args.current)

    if not hourly and not daily and not current:
        hourly = ["temperature_2m", "precipitation", "weather_code"]

    response = client.forecast.get(
        latitude=lat,
        longitude=lon,
        hourly=hourly,
        daily=daily,
        current=current,
        forecast_days=args.forecast_days,
        timezone=args.timezone,
        temperature_unit=args.temperature_unit,
    )

    if current and response.current:
        print("\nCurrent Weather:")
        print(f"  Time: {response.current.time}")
        for key, value in response.current.data.items():
            unit = response.current.units.get(key, "")
            print(f"  {key}: {value} {unit}")
        print()

    if hourly and response.hourly:
        df = response.to_dataframe("hourly")
        output_result(df, args, "forecast_hourly")
    elif daily and response.daily:
        df = response.to_dataframe("daily")
        output_result(df, args, "forecast_daily")


def cmd_historical(client: OpenMeteo, args):
    """Handle historical command."""
    lat, lon = resolve_location(client, args)

    hourly = parse_variables(args.hourly)
    daily = parse_variables(args.daily)

    if not hourly and not daily:
        hourly = ["temperature_2m", "precipitation"]

    response = client.historical.get(
        latitude=lat,
        longitude=lon,
        start_date=args.start,
        end_date=args.end,
        hourly=hourly,
        daily=daily,
        timezone=args.timezone,
        temperature_unit=args.temperature_unit,
    )

    if hourly and response.hourly:
        df = response.to_dataframe("hourly")
    elif daily and response.daily:
        df = response.to_dataframe("daily")
    else:
        print("No data returned")
        return

    output_result(df, args, "historical")


def cmd_air_quality(client: OpenMeteo, args):
    """Handle air-quality command."""
    lat, lon = resolve_location(client, args)

    hourly = parse_variables(args.hourly)
    current = parse_variables(args.current)

    if not hourly and not current:
        hourly = ["pm10", "pm2_5", "european_aqi"]

    response = client.air_quality.get(
        latitude=lat,
        longitude=lon,
        hourly=hourly,
        current=current,
        forecast_days=args.forecast_days,
        timezone=args.timezone,
    )

    if current and response.current:
        print("\nCurrent Air Quality:")
        print(f"  Time: {response.current.time}")
        for key, value in response.current.data.items():
            unit = response.current.units.get(key, "")
            print(f"  {key}: {value} {unit}")
        print()

    if hourly and response.hourly:
        df = response.to_dataframe("hourly")
        output_result(df, args, "air_quality")


def cmd_marine(client: OpenMeteo, args):
    """Handle marine command."""
    lat, lon = resolve_location(client, args)

    hourly = parse_variables(args.hourly)
    daily = parse_variables(args.daily)
    current = parse_variables(args.current)

    if not hourly and not daily and not current:
        hourly = ["wave_height", "wave_direction", "sea_surface_temperature"]

    response = client.marine.get(
        latitude=lat,
        longitude=lon,
        hourly=hourly,
        daily=daily,
        current=current,
        forecast_days=args.forecast_days,
        timezone=args.timezone,
    )

    if current and response.current:
        print("\nCurrent Marine Conditions:")
        print(f"  Time: {response.current.time}")
        for key, value in response.current.data.items():
            unit = response.current.units.get(key, "")
            print(f"  {key}: {value} {unit}")
        print()

    if hourly and response.hourly:
        df = response.to_dataframe("hourly")
        output_result(df, args, "marine")
    elif daily and response.daily:
        df = response.to_dataframe("daily")
        output_result(df, args, "marine")


def cmd_flood(client: OpenMeteo, args):
    """Handle flood command."""
    lat, lon = resolve_location(client, args)

    daily = parse_variables(args.daily)

    kwargs = {
        "latitude": lat,
        "longitude": lon,
        "daily": daily,
    }

    if args.start and args.end:
        kwargs["start_date"] = args.start
        kwargs["end_date"] = args.end
    else:
        kwargs["forecast_days"] = args.forecast_days

    response = client.flood.get(**kwargs)

    if response.daily:
        df = response.to_dataframe()
        output_result(df, args, "flood")
    else:
        print("No data returned")


def cmd_climate(client: OpenMeteo, args):
    """Handle climate command."""
    lat, lon = resolve_location(client, args)

    models = parse_variables(args.models)
    daily = parse_variables(args.daily)

    response = client.climate.get(
        latitude=lat,
        longitude=lon,
        start_date=args.start,
        end_date=args.end,
        models=models,
        daily=daily,
        temperature_unit=args.temperature_unit,
    )

    if response.daily:
        df = response.to_dataframe("daily")
        output_result(df, args, "climate")
    else:
        print("No data returned")


def cmd_ensemble(client: OpenMeteo, args):
    """Handle ensemble command."""
    lat, lon = resolve_location(client, args)

    models = parse_variables(args.models)
    hourly = parse_variables(args.hourly)
    daily = parse_variables(args.daily)

    if not hourly and not daily:
        hourly = ["temperature_2m", "precipitation"]

    response = client.ensemble.get(
        latitude=lat,
        longitude=lon,
        models=models,
        hourly=hourly,
        daily=daily,
        forecast_days=args.forecast_days,
        timezone=args.timezone,
    )

    if hourly and response.hourly:
        df = response.to_dataframe("hourly")
        output_result(df, args, "ensemble")
    elif daily and response.daily:
        df = response.to_dataframe("daily")
        output_result(df, args, "ensemble")


def cmd_geocoding(client: OpenMeteo, args):
    """Handle geocoding command."""
    results = client.geocoding.search(
        name=args.search,
        count=args.count,
        language=args.language,
        country_code=args.country,
    )

    if not results:
        print(f"No results found for '{args.search}'")
        return

    data = []
    for r in results:
        data.append({
            "id": r.id,
            "name": r.name,
            "latitude": r.latitude,
            "longitude": r.longitude,
            "elevation": r.elevation,
            "country": r.country,
            "country_code": r.country_code,
            "timezone": r.timezone,
            "admin1": r.admin1,
            "population": r.population,
        })

    df = pd.DataFrame(data)
    output_result(df, args, "geocoding")


def cmd_elevation(client: OpenMeteo, args):
    """Handle elevation command."""
    lat, lon = resolve_location(client, args)

    response = client.elevation.get(latitude=lat, longitude=lon)

    data = [{
        "latitude": lat,
        "longitude": lon,
        "elevation_m": response.elevations[0],
    }]

    df = pd.DataFrame(data)
    output_result(df, args, "elevation")


def cmd_generate_dag(args):
    """Handle generate-dag command."""
    from openmeteo.generator import DagGenerator

    # Parse APIs
    apis = [api.strip() for api in args.apis.split(",")]

    # Parse locations (format: "Name:lat:lon,Name2:lat2:lon2")
    locations = []
    for loc_str in args.locations.split(","):
        parts = loc_str.strip().split(":")
        if len(parts) != 3:
            print(f"Error: Invalid location format '{loc_str}'. Use 'Name:lat:lon'", file=sys.stderr)
            sys.exit(1)
        try:
            locations.append({
                "name": parts[0].strip(),
                "lat": float(parts[1].strip()),
                "lon": float(parts[2].strip()),
            })
        except ValueError:
            print(f"Error: Invalid coordinates in '{loc_str}'", file=sys.stderr)
            sys.exit(1)

    try:
        generator = DagGenerator(
            apis=apis,
            locations=locations,
            output_format=args.output_format,
            output_dir=args.output_dir,
            dag_id=args.dag_id,
            schedule=args.schedule,
            include_airflow=not args.no_airflow,
        )

        script_content = generator.generate()

        # Write to file
        with open(args.output, "w") as f:
            f.write(script_content)

        print(f"Generated DAG script: {args.output}")
        print(f"  APIs: {', '.join(apis)}")
        print(f"  Locations: {', '.join(loc['name'] for loc in locations)}")
        print(f"  Schedule: {args.schedule}")
        print(f"  Output format: {args.output_format}")
        if not args.no_airflow:
            print(f"  Airflow DAG ID: {generator.dag_id}")
        print(f"\nUsage:")
        print(f"  python {args.output}              # Run standalone")
        if not args.no_airflow:
            print(f"  # Or copy to Airflow DAGs folder for automatic scheduling")

    except ValueError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def cmd_list_apis(args):
    """Handle list-apis command."""
    from openmeteo.generator import DagGenerator

    print("Available APIs for DAG generation:")
    print("-" * 50)
    for api in DagGenerator.get_available_apis():
        desc = DagGenerator.get_api_description(api)
        print(f"  {api:15} - {desc}")
    print("-" * 50)
    print("\nExample:")
    print('  openmeteo generate-dag --apis forecast,air_quality --locations "Berlin:52.52:13.41" -o my_dag.py')


def cmd_generate_script(args):
    """Handle generate-script command."""
    from openmeteo.generator import StandaloneScriptGenerator

    # Parse locations (format: "Name:lat:lon,Name2:lat2:lon2")
    locations = []
    for loc_str in args.locations.split(","):
        parts = loc_str.strip().split(":")
        if len(parts) != 3:
            print(f"Error: Invalid location format '{loc_str}'. Use 'Name:lat:lon'", file=sys.stderr)
            sys.exit(1)
        try:
            locations.append({
                "name": parts[0].strip(),
                "lat": float(parts[1].strip()),
                "lon": float(parts[2].strip()),
            })
        except ValueError:
            print(f"Error: Invalid coordinates in '{loc_str}'", file=sys.stderr)
            sys.exit(1)

    try:
        generator = StandaloneScriptGenerator(
            api=args.api,
            locations=locations,
            output_format=args.format,
            output_dir=args.output_dir,
        )

        script_content = generator.generate()

        # Write to file
        with open(args.output, "w") as f:
            f.write(script_content)

        print(f"Generated standalone script: {args.output}")
        print(f"  API: {args.api}")
        print(f"  Locations: {', '.join(loc['name'] for loc in locations)}")
        print(f"  Output format: {args.format}")
        print(f"  Output directory: {args.output_dir}")
        print(f"\nUsage:")
        print(f"  python {args.output}                    # Run with defaults")
        if args.api in ["historical", "climate"]:
            print(f"  python {args.output} --start-date 2024-01-01 --end-date 2024-01-31")
        print(f"  python {args.output} --help             # Show all options")

    except ValueError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def main():
    """Main CLI entry point."""
    parser = create_parser()
    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    # Commands that don't need the API client
    if args.command == "generate-dag":
        cmd_generate_dag(args)
        return

    if args.command == "generate-script":
        cmd_generate_script(args)
        return

    if args.command == "list-apis":
        cmd_list_apis(args)
        return

    # Create client with config for API commands
    config = APIConfig(
        timeout=args.timeout,
        api_key=args.api_key,
    )
    client = OpenMeteo(config=config)

    try:
        # Route to command handler
        commands = {
            "forecast": cmd_forecast,
            "historical": cmd_historical,
            "air-quality": cmd_air_quality,
            "marine": cmd_marine,
            "flood": cmd_flood,
            "climate": cmd_climate,
            "ensemble": cmd_ensemble,
            "geocoding": cmd_geocoding,
            "elevation": cmd_elevation,
        }

        handler = commands.get(args.command)
        if handler:
            handler(client, args)
        else:
            print(f"Unknown command: {args.command}", file=sys.stderr)
            sys.exit(1)

    except ValueError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    finally:
        client.close()


if __name__ == "__main__":
    main()
