"""
Main OpenMeteo client that provides access to all API endpoints.
"""

from typing import Optional
from openmeteo.base import APIConfig
from openmeteo.clients.forecast import ForecastClient
from openmeteo.clients.historical import HistoricalClient
from openmeteo.clients.air_quality import AirQualityClient
from openmeteo.clients.marine import MarineClient
from openmeteo.clients.flood import FloodClient
from openmeteo.clients.climate import ClimateClient
from openmeteo.clients.ensemble import EnsembleClient
from openmeteo.clients.geocoding import GeocodingClient
from openmeteo.clients.elevation import ElevationClient


class OpenMeteo:
    """
    Main client for Open-Meteo API.

    Provides unified access to all Open-Meteo API endpoints.

    Example:
        >>> client = OpenMeteo()
        >>>
        >>> # Get weather forecast
        >>> forecast = client.forecast.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     hourly=["temperature_2m", "precipitation"]
        ... )
        >>>
        >>> # Get historical data
        >>> historical = client.historical.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     start_date="2023-01-01",
        ...     end_date="2023-01-31",
        ...     hourly=["temperature_2m"]
        ... )
        >>>
        >>> # Search for a location
        >>> locations = client.geocoding.search("Berlin")
        >>> print(locations[0].latitude, locations[0].longitude)

    With API key (commercial use):
        >>> from openmeteo import OpenMeteo
        >>> from openmeteo.base import APIConfig
        >>>
        >>> config = APIConfig(api_key="your-api-key")
        >>> client = OpenMeteo(config=config)
    """

    def __init__(self, config: Optional[APIConfig] = None):
        """
        Initialize the OpenMeteo client.

        Args:
            config: Optional API configuration with timeout, retry settings, and API key.
        """
        self.config = config or APIConfig()

        # Initialize all API clients
        self._forecast = None
        self._historical = None
        self._air_quality = None
        self._marine = None
        self._flood = None
        self._climate = None
        self._ensemble = None
        self._geocoding = None
        self._elevation = None

    @property
    def forecast(self) -> ForecastClient:
        """Weather Forecast API client."""
        if self._forecast is None:
            self._forecast = ForecastClient(self.config)
        return self._forecast

    @property
    def historical(self) -> HistoricalClient:
        """Historical Weather API client."""
        if self._historical is None:
            self._historical = HistoricalClient(self.config)
        return self._historical

    @property
    def air_quality(self) -> AirQualityClient:
        """Air Quality API client."""
        if self._air_quality is None:
            self._air_quality = AirQualityClient(self.config)
        return self._air_quality

    @property
    def marine(self) -> MarineClient:
        """Marine Weather API client."""
        if self._marine is None:
            self._marine = MarineClient(self.config)
        return self._marine

    @property
    def flood(self) -> FloodClient:
        """Flood API client."""
        if self._flood is None:
            self._flood = FloodClient(self.config)
        return self._flood

    @property
    def climate(self) -> ClimateClient:
        """Climate Change API client."""
        if self._climate is None:
            self._climate = ClimateClient(self.config)
        return self._climate

    @property
    def ensemble(self) -> EnsembleClient:
        """Ensemble Models API client."""
        if self._ensemble is None:
            self._ensemble = EnsembleClient(self.config)
        return self._ensemble

    @property
    def geocoding(self) -> GeocodingClient:
        """Geocoding API client."""
        if self._geocoding is None:
            self._geocoding = GeocodingClient(self.config)
        return self._geocoding

    @property
    def elevation(self) -> ElevationClient:
        """Elevation API client."""
        if self._elevation is None:
            self._elevation = ElevationClient(self.config)
        return self._elevation

    def close(self):
        """Close all client sessions."""
        clients = [
            self._forecast,
            self._historical,
            self._air_quality,
            self._marine,
            self._flood,
            self._climate,
            self._ensemble,
            self._geocoding,
            self._elevation,
        ]
        for client in clients:
            if client is not None:
                client.close()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()
