"""
Air Quality API client.

Provides access to air quality forecasts including pollutants, pollen, and AQI indices.
"""

from typing import List, Optional, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import WeatherResponse


class AirQualityVariables:
    """Available variables for Air Quality API."""

    # Pollutants
    POLLUTANTS = [
        "pm10",
        "pm2_5",
        "carbon_monoxide",
        "nitrogen_dioxide",
        "sulphur_dioxide",
        "ozone",
        "ammonia",
        "methane",
        "aerosol_optical_depth",
        "dust",
        "formaldehyde",
        "glyoxal",
        "non_methane_volatile_organic_compounds",
        "carbon_dioxide",
    ]

    # Pollen (Europe only)
    POLLEN = [
        "alder_pollen",
        "birch_pollen",
        "grass_pollen",
        "mugwort_pollen",
        "olive_pollen",
        "ragweed_pollen",
    ]

    # Air Quality Indices
    AQI_EUROPEAN = [
        "european_aqi",
        "european_aqi_pm2_5",
        "european_aqi_pm10",
        "european_aqi_nitrogen_dioxide",
        "european_aqi_ozone",
        "european_aqi_sulphur_dioxide",
    ]

    AQI_US = [
        "us_aqi",
        "us_aqi_pm2_5",
        "us_aqi_pm10",
        "us_aqi_nitrogen_dioxide",
        "us_aqi_carbon_monoxide",
        "us_aqi_ozone",
        "us_aqi_sulphur_dioxide",
    ]

    # UV Index
    UV = [
        "uv_index",
        "uv_index_clear_sky",
    ]

    # All hourly variables
    HOURLY_ALL = POLLUTANTS + POLLEN + AQI_EUROPEAN + AQI_US + UV

    # Current variables
    CURRENT_ALL = HOURLY_ALL

    # Common presets
    BASIC = ["pm10", "pm2_5", "european_aqi", "us_aqi"]
    POLLUTANTS_BASIC = ["pm10", "pm2_5", "carbon_monoxide", "nitrogen_dioxide", "ozone"]
    POLLEN_ALL = POLLEN


class AirQualityClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Air Quality API.

    Provides air quality forecasts including pollutants, pollen, and AQI indices.

    Example:
        >>> client = AirQualityClient()
        >>> response = client.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     hourly=["pm10", "pm2_5", "european_aqi"]
        ... )
        >>> df = response.to_dataframe()
    """

    BASE_URL = "https://air-quality-api.open-meteo.com/v1/air-quality"

    # Available domains
    DOMAINS = ["auto", "cams_europe", "cams_global"]

    def __init__(self, config: Optional[APIConfig] = None):
        super().__init__(config)
        self.variables = AirQualityVariables

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        hourly: Optional[List[str]] = None,
        current: Optional[List[str]] = None,
        timezone: str = "UTC",
        forecast_days: Optional[int] = None,
        forecast_hours: Optional[int] = None,
        past_days: Optional[int] = None,
        past_hours: Optional[int] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
        start_hour: Optional[str] = None,
        end_hour: Optional[str] = None,
        domains: str = "auto",
        cell_selection: str = "nearest",
    ) -> WeatherResponse:
        """
        Get air quality data.

        Args:
            latitude: Latitude (-90 to 90). Can be a list for multiple locations.
            longitude: Longitude (-180 to 180). Can be a list for multiple locations.
            hourly: List of hourly variables to fetch.
            current: List of current variables to fetch.
            timezone: Timezone for timestamps (default: UTC).
            forecast_days: Number of forecast days (0-7).
            forecast_hours: Number of forecast hours.
            past_days: Number of past days (0-92).
            past_hours: Number of past hours.
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            start_hour: Start hour (YYYY-MM-DDTHH:MM).
            end_hour: End hour (YYYY-MM-DDTHH:MM).
            domains: Data domain ('auto', 'cams_europe', 'cams_global').
            cell_selection: Grid cell selection ('nearest', 'land', 'sea').

        Returns:
            WeatherResponse with requested data.
        """
        self._validate_coordinates(latitude, longitude)

        if start_date:
            self._validate_date(start_date, "start_date")
        if end_date:
            self._validate_date(end_date, "end_date")

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
            hourly=hourly,
            current=current,
            timezone=timezone,
            forecast_days=forecast_days,
            forecast_hours=forecast_hours,
            past_days=past_days,
            past_hours=past_hours,
            start_date=start_date,
            end_date=end_date,
            start_hour=start_hour,
            end_hour=end_hour,
            domains=domains,
            cell_selection=cell_selection,
        )

        data = self._make_request(self.BASE_URL, params)
        return WeatherResponse.from_response(data)

    def get_current(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        timezone: str = "UTC",
    ) -> WeatherResponse:
        """
        Get current air quality conditions.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of current variables (default: basic set).
            timezone: Timezone for timestamps.

        Returns:
            WeatherResponse with current conditions.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            current=variables,
            timezone=timezone,
        )

    def get_hourly(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        forecast_days: int = 5,
        timezone: str = "UTC",
    ) -> WeatherResponse:
        """
        Get hourly air quality forecast.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of hourly variables (default: basic set).
            forecast_days: Number of forecast days (0-7).
            timezone: Timezone for timestamps.

        Returns:
            WeatherResponse with hourly data.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            hourly=variables,
            forecast_days=forecast_days,
            timezone=timezone,
        )

    def get_pollen(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        forecast_days: int = 5,
        timezone: str = "UTC",
    ) -> WeatherResponse:
        """
        Get pollen forecast (Europe only).

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            forecast_days: Number of forecast days (0-7).
            timezone: Timezone for timestamps.

        Returns:
            WeatherResponse with pollen data.
        """
        return self.get(
            latitude=latitude,
            longitude=longitude,
            hourly=self.variables.POLLEN,
            forecast_days=forecast_days,
            timezone=timezone,
            domains="cams_europe",
        )
