"""
Climate Change API client.

Provides access to climate model data from 1950 to 2050.
"""

from typing import List, Optional, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import WeatherResponse


class ClimateVariables:
    """Available variables for Climate API."""

    # Daily temperature
    DAILY_TEMPERATURE = [
        "temperature_2m_max",
        "temperature_2m_min",
        "temperature_2m_mean",
        "apparent_temperature_max",
        "apparent_temperature_min",
        "apparent_temperature_mean",
    ]

    # Daily humidity and pressure
    DAILY_HUMIDITY_PRESSURE = [
        "relative_humidity_2m_max",
        "relative_humidity_2m_min",
        "relative_humidity_2m_mean",
        "pressure_msl_mean",
    ]

    # Daily precipitation
    DAILY_PRECIPITATION = [
        "precipitation_sum",
        "rain_sum",
        "snowfall_sum",
        "precipitation_hours",
    ]

    # Daily wind
    DAILY_WIND = [
        "wind_speed_10m_max",
        "wind_speed_10m_mean",
    ]

    # Daily clouds
    DAILY_CLOUDS = [
        "cloud_cover_mean",
    ]

    # Daily solar
    DAILY_SOLAR = [
        "shortwave_radiation_sum",
        "et0_fao_evapotranspiration",
    ]

    # Daily soil
    DAILY_SOIL = [
        "soil_moisture_0_to_10cm_mean",
    ]

    # All daily variables
    DAILY_ALL = (
        DAILY_TEMPERATURE
        + DAILY_HUMIDITY_PRESSURE
        + DAILY_PRECIPITATION
        + DAILY_WIND
        + DAILY_CLOUDS
        + DAILY_SOLAR
        + DAILY_SOIL
    )

    # Common presets
    BASIC = [
        "temperature_2m_max",
        "temperature_2m_min",
        "precipitation_sum",
    ]
    TEMPERATURE = DAILY_TEMPERATURE
    PRECIPITATION = DAILY_PRECIPITATION


class ClimateClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Climate Change API.

    Provides climate model projections from 1950 to 2050.

    Example:
        >>> client = ClimateClient()
        >>> response = client.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     start_date="2020-01-01",
        ...     end_date="2020-12-31",
        ...     models=["EC_Earth3P_HR"],
        ...     daily=["temperature_2m_max", "temperature_2m_min"]
        ... )
        >>> df = response.to_dataframe("daily")
    """

    BASE_URL = "https://climate-api.open-meteo.com/v1/climate"

    # Available climate models
    MODELS = [
        "CMCC_CM2_VHR4",
        "FGOALS_f3_H",
        "HiRAM_SIT_HR",
        "MRI_AGCM3_2_S",
        "EC_Earth3P_HR",
        "MPI_ESM1_2_XR",
        "NICAM16_8S",
    ]

    def __init__(self, config: Optional[APIConfig] = None):
        super().__init__(config)
        self.variables = ClimateVariables

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        start_date: str,
        end_date: str,
        models: List[str],
        daily: Optional[List[str]] = None,
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
        timeformat: str = "iso8601",
        disable_bias_correction: bool = False,
        cell_selection: str = "land",
    ) -> WeatherResponse:
        """
        Get climate model data.

        Args:
            latitude: Latitude (-90 to 90). Can be a list for multiple locations.
            longitude: Longitude (-180 to 180). Can be a list for multiple locations.
            start_date: Start date (YYYY-MM-DD). Data available from 1950.
            end_date: End date (YYYY-MM-DD). Data available until 2050.
            models: List of climate models to use.
            daily: List of daily variables to fetch.
            temperature_unit: Temperature unit ('celsius' or 'fahrenheit').
            wind_speed_unit: Wind speed unit ('kmh', 'ms', 'mph', 'kn').
            precipitation_unit: Precipitation unit ('mm' or 'inch').
            timeformat: Time format ('iso8601' or 'unixtime').
            disable_bias_correction: Disable bias correction (default: False).
            cell_selection: Grid cell selection ('land', 'sea', 'nearest').

        Returns:
            WeatherResponse with requested data.
        """
        self._validate_coordinates(latitude, longitude)
        self._validate_date(start_date, "start_date")
        self._validate_date(end_date, "end_date")

        if not models:
            raise ValueError("At least one climate model must be specified")

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
            start_date=start_date,
            end_date=end_date,
            models=models,
            daily=daily,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
            timeformat=timeformat,
            disable_bias_correction=disable_bias_correction,
            cell_selection=cell_selection,
        )

        data = self._make_request(self.BASE_URL, params)
        return WeatherResponse.from_response(data)

    def get_projection(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        start_date: str,
        end_date: str,
        models: Optional[List[str]] = None,
        variables: Optional[List[str]] = None,
        temperature_unit: str = "celsius",
    ) -> WeatherResponse:
        """
        Get climate projection data.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            models: List of climate models (default: EC_Earth3P_HR).
            variables: List of daily variables (default: basic set).
            temperature_unit: Temperature unit.

        Returns:
            WeatherResponse with projection data.
        """
        if models is None:
            models = ["EC_Earth3P_HR"]
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            start_date=start_date,
            end_date=end_date,
            models=models,
            daily=variables,
            temperature_unit=temperature_unit,
        )

    def compare_models(
        self,
        latitude: float,
        longitude: float,
        start_date: str,
        end_date: str,
        variables: Optional[List[str]] = None,
    ) -> WeatherResponse:
        """
        Compare all available climate models.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            variables: List of daily variables (default: basic set).

        Returns:
            WeatherResponse with data from all models.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            start_date=start_date,
            end_date=end_date,
            models=self.MODELS,
            daily=variables,
        )
