"""
Elevation API client.

Provides terrain elevation data for coordinates.
"""

from typing import List, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import ElevationResponse


class ElevationClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Elevation API.

    Provides terrain elevation data based on 90m resolution DEM.

    Example:
        >>> client = ElevationClient()
        >>> response = client.get(latitude=52.52, longitude=13.41)
        >>> print(response.elevations[0])
        34.0
    """

    BASE_URL = "https://api.open-meteo.com/v1/elevation"

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
    ) -> ElevationResponse:
        """
        Get elevation data for coordinates.

        Args:
            latitude: Latitude (-90 to 90). Can be a list (max 100).
            longitude: Longitude (-180 to 180). Can be a list (max 100).

        Returns:
            ElevationResponse with elevation values in meters.

        Raises:
            ValueError: If more than 100 coordinates provided.
        """
        self._validate_coordinates(latitude, longitude)

        # Check batch limit
        lats = [latitude] if isinstance(latitude, (int, float)) else latitude
        if len(lats) > 100:
            raise ValueError("Maximum 100 coordinates per request")

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
        )

        data = self._make_request(self.BASE_URL, params)
        return ElevationResponse.from_response(data)

    def get_elevation(
        self,
        latitude: float,
        longitude: float,
    ) -> float:
        """
        Get single elevation value.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).

        Returns:
            Elevation in meters.
        """
        response = self.get(latitude=latitude, longitude=longitude)
        return response.elevations[0]

    def get_batch(
        self,
        coordinates: List[tuple],
    ) -> List[float]:
        """
        Get elevations for multiple coordinates.

        Args:
            coordinates: List of (latitude, longitude) tuples.

        Returns:
            List of elevations in meters.

        Raises:
            ValueError: If more than 100 coordinates provided.
        """
        if len(coordinates) > 100:
            raise ValueError("Maximum 100 coordinates per request")

        latitudes = [c[0] for c in coordinates]
        longitudes = [c[1] for c in coordinates]

        response = self.get(latitude=latitudes, longitude=longitudes)
        return response.elevations
