"""
Ensemble Models API client.

Provides access to ensemble weather model predictions for probabilistic forecasting.
"""

from typing import List, Optional, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import WeatherResponse


class EnsembleVariables:
    """Available variables for Ensemble API."""

    # Hourly temperature
    HOURLY_TEMPERATURE = [
        "temperature_2m",
        "apparent_temperature",
        "dew_point_2m",
    ]

    # Hourly humidity and pressure
    HOURLY_HUMIDITY_PRESSURE = [
        "relative_humidity_2m",
        "surface_pressure",
        "pressure_msl",
    ]

    # Hourly precipitation
    HOURLY_PRECIPITATION = [
        "precipitation",
        "rain",
        "snowfall",
        "snow_depth",
        "freezing_level_height",
    ]

    # Hourly wind
    HOURLY_WIND = [
        "wind_speed_10m",
        "wind_speed_80m",
        "wind_direction_10m",
        "wind_direction_80m",
        "wind_gusts_10m",
    ]

    # Hourly clouds
    HOURLY_CLOUDS = [
        "cloud_cover",
        "visibility",
    ]

    # Hourly solar
    HOURLY_SOLAR = [
        "shortwave_radiation",
        "direct_radiation",
        "diffuse_radiation",
        "direct_normal_irradiance",
        "global_tilted_irradiance",
    ]

    # Hourly soil
    HOURLY_SOIL = [
        "soil_temperature_0_to_7cm",
        "soil_temperature_7_to_28cm",
        "soil_temperature_28_to_100cm",
        "soil_temperature_100_to_255cm",
        "soil_moisture_0_to_7cm",
        "soil_moisture_7_to_28cm",
        "soil_moisture_28_to_100cm",
        "soil_moisture_100_to_255cm",
    ]

    # Hourly other
    HOURLY_OTHER = [
        "weather_code",
        "cape",
        "lifted_index",
        "evapotranspiration",
        "et0_fao_evapotranspiration",
        "sunshine_duration",
    ]

    # All hourly variables
    HOURLY_ALL = (
        HOURLY_TEMPERATURE
        + HOURLY_HUMIDITY_PRESSURE
        + HOURLY_PRECIPITATION
        + HOURLY_WIND
        + HOURLY_CLOUDS
        + HOURLY_SOLAR
        + HOURLY_SOIL
        + HOURLY_OTHER
    )

    # Daily variables
    DAILY_ALL = [
        "weather_code",
        "temperature_2m_max",
        "temperature_2m_min",
        "temperature_2m_mean",
        "apparent_temperature_max",
        "apparent_temperature_min",
        "apparent_temperature_mean",
        "sunrise",
        "sunset",
        "daylight_duration",
        "sunshine_duration",
        "precipitation_sum",
        "rain_sum",
        "snowfall_sum",
        "precipitation_hours",
        "wind_speed_10m_max",
        "wind_gusts_10m_max",
        "wind_direction_10m_dominant",
        "shortwave_radiation_sum",
        "et0_fao_evapotranspiration",
    ]

    # Common presets
    BASIC = ["temperature_2m", "precipitation", "wind_speed_10m", "weather_code"]
    TEMPERATURE = HOURLY_TEMPERATURE
    PRECIPITATION = HOURLY_PRECIPITATION


class EnsembleClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Ensemble Models API.

    Provides ensemble weather predictions for probabilistic forecasting.

    Example:
        >>> client = EnsembleClient()
        >>> response = client.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     models=["icon_seamless"],
        ...     hourly=["temperature_2m", "precipitation"]
        ... )
        >>> df = response.to_dataframe()
    """

    BASE_URL = "https://ensemble-api.open-meteo.com/v1/ensemble"

    # Available ensemble models
    MODELS = [
        # DWD ICON
        "icon_seamless",
        "icon_global",
        "icon_eu",
        "icon_d2",
        # GFS
        "gfs_seamless",
        "gfs025",
        "gfs05",
        # ECMWF
        "ecmwf_ifs04",
        "ecmwf_aifs025",
        # GEM (Canada)
        "gem_global",
        # BOM (Australia)
        "bom_access_global_ensemble",
        # UK Met Office
        "ukmo_global_ensemble",
        # MeteoSwiss
        "meteoswiss_icon_ch_ensemble",
    ]

    def __init__(self, config: Optional[APIConfig] = None):
        super().__init__(config)
        self.variables = EnsembleVariables

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        models: List[str],
        hourly: Optional[List[str]] = None,
        daily: Optional[List[str]] = None,
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
        timezone: str = "UTC",
        forecast_days: Optional[int] = None,
        forecast_hours: Optional[int] = None,
        past_days: Optional[int] = None,
        past_hours: Optional[int] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
        start_hour: Optional[str] = None,
        end_hour: Optional[str] = None,
        cell_selection: str = "land",
        elevation: Optional[float] = None,
    ) -> WeatherResponse:
        """
        Get ensemble model data.

        Args:
            latitude: Latitude (-90 to 90). Can be a list for multiple locations.
            longitude: Longitude (-180 to 180). Can be a list for multiple locations.
            models: List of ensemble models to use.
            hourly: List of hourly variables to fetch.
            daily: List of daily variables to fetch.
            temperature_unit: Temperature unit ('celsius' or 'fahrenheit').
            wind_speed_unit: Wind speed unit ('kmh', 'ms', 'mph', 'kn').
            precipitation_unit: Precipitation unit ('mm' or 'inch').
            timezone: Timezone for timestamps (default: UTC).
            forecast_days: Number of forecast days (0-35).
            forecast_hours: Number of forecast hours.
            past_days: Number of past days (0-35).
            past_hours: Number of past hours.
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            start_hour: Start hour (YYYY-MM-DDTHH:MM).
            end_hour: End hour (YYYY-MM-DDTHH:MM).
            cell_selection: Grid cell selection ('land', 'sea', 'nearest').
            elevation: Manual elevation in meters.

        Returns:
            WeatherResponse with requested data.
        """
        self._validate_coordinates(latitude, longitude)

        if not models:
            raise ValueError("At least one ensemble model must be specified")

        if start_date:
            self._validate_date(start_date, "start_date")
        if end_date:
            self._validate_date(end_date, "end_date")

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
            models=models,
            hourly=hourly,
            daily=daily,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
            timezone=timezone,
            forecast_days=forecast_days,
            forecast_hours=forecast_hours,
            past_days=past_days,
            past_hours=past_hours,
            start_date=start_date,
            end_date=end_date,
            start_hour=start_hour,
            end_hour=end_hour,
            cell_selection=cell_selection,
            elevation=elevation,
        )

        data = self._make_request(self.BASE_URL, params)
        return WeatherResponse.from_response(data)

    def get_forecast(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        models: Optional[List[str]] = None,
        variables: Optional[List[str]] = None,
        forecast_days: int = 7,
        timezone: str = "UTC",
        temperature_unit: str = "celsius",
    ) -> WeatherResponse:
        """
        Get ensemble forecast data.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            models: List of ensemble models (default: icon_seamless).
            variables: List of hourly variables (default: basic set).
            forecast_days: Number of forecast days (0-35).
            timezone: Timezone for timestamps.
            temperature_unit: Temperature unit.

        Returns:
            WeatherResponse with ensemble forecast.
        """
        if models is None:
            models = ["icon_seamless"]
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            models=models,
            hourly=variables,
            forecast_days=forecast_days,
            timezone=timezone,
            temperature_unit=temperature_unit,
        )

    def compare_models(
        self,
        latitude: float,
        longitude: float,
        variables: Optional[List[str]] = None,
        forecast_days: int = 7,
        timezone: str = "UTC",
    ) -> WeatherResponse:
        """
        Compare multiple ensemble models.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of hourly variables (default: basic set).
            forecast_days: Number of forecast days (0-35).
            timezone: Timezone for timestamps.

        Returns:
            WeatherResponse with data from multiple models.
        """
        if variables is None:
            variables = self.variables.BASIC

        # Use main models for comparison
        models = ["icon_seamless", "gfs_seamless", "ecmwf_ifs04"]

        return self.get(
            latitude=latitude,
            longitude=longitude,
            models=models,
            hourly=variables,
            forecast_days=forecast_days,
            timezone=timezone,
        )
