"""
Flood API client.

Provides access to river discharge forecasts and flood risk data.
"""

from typing import List, Optional, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import FloodResponse


class FloodVariables:
    """Available variables for Flood API."""

    # Daily variables
    DAILY_ALL = [
        "river_discharge",
        "river_discharge_mean",
        "river_discharge_median",
        "river_discharge_max",
        "river_discharge_min",
        "river_discharge_p25",
        "river_discharge_p75",
    ]

    # Common presets
    BASIC = ["river_discharge"]
    STATISTICS = [
        "river_discharge_mean",
        "river_discharge_median",
        "river_discharge_max",
        "river_discharge_min",
    ]
    PERCENTILES = ["river_discharge_p25", "river_discharge_p75"]


class FloodClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Flood API.

    Provides river discharge forecasts for up to 210 days.
    Data available from 1984 onwards.

    Example:
        >>> client = FloodClient()
        >>> response = client.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     daily=["river_discharge", "river_discharge_max"]
        ... )
        >>> df = response.to_dataframe()
    """

    BASE_URL = "https://flood-api.open-meteo.com/v1/flood"

    def __init__(self, config: Optional[APIConfig] = None):
        super().__init__(config)
        self.variables = FloodVariables

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        daily: Optional[List[str]] = None,
        timeformat: str = "iso8601",
        forecast_days: Optional[int] = None,
        past_days: Optional[int] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
        ensemble: bool = False,
        cell_selection: str = "nearest",
    ) -> FloodResponse:
        """
        Get flood/river discharge data.

        Args:
            latitude: Latitude (-90 to 90). Can be a list for multiple locations.
            longitude: Longitude (-180 to 180). Can be a list for multiple locations.
            daily: List of daily variables to fetch.
            timeformat: Time format ('iso8601' or 'unixtime').
            forecast_days: Number of forecast days (0-210).
            past_days: Number of past days to include.
            start_date: Start date (YYYY-MM-DD). Data available from 1984.
            end_date: End date (YYYY-MM-DD).
            ensemble: Return all forecast ensemble members.
            cell_selection: Grid cell selection ('nearest', 'land', 'sea').

        Returns:
            FloodResponse with requested data.
        """
        self._validate_coordinates(latitude, longitude)

        if start_date:
            self._validate_date(start_date, "start_date")
        if end_date:
            self._validate_date(end_date, "end_date")

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
            daily=daily,
            timeformat=timeformat,
            forecast_days=forecast_days,
            past_days=past_days,
            start_date=start_date,
            end_date=end_date,
            ensemble=ensemble,
            cell_selection=cell_selection,
        )

        data = self._make_request(self.BASE_URL, params)
        return FloodResponse.from_response(data)

    def get_forecast(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        forecast_days: int = 92,
    ) -> FloodResponse:
        """
        Get river discharge forecast.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of daily variables (default: basic set).
            forecast_days: Number of forecast days (0-210).

        Returns:
            FloodResponse with forecast data.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            daily=variables,
            forecast_days=forecast_days,
        )

    def get_historical(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        start_date: str,
        end_date: str,
        variables: Optional[List[str]] = None,
    ) -> FloodResponse:
        """
        Get historical river discharge data.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            start_date: Start date (YYYY-MM-DD). Data available from 1984.
            end_date: End date (YYYY-MM-DD).
            variables: List of daily variables (default: basic set).

        Returns:
            FloodResponse with historical data.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            daily=variables,
            start_date=start_date,
            end_date=end_date,
        )

    def get_with_statistics(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        forecast_days: int = 92,
    ) -> FloodResponse:
        """
        Get river discharge with statistical measures.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            forecast_days: Number of forecast days (0-210).

        Returns:
            FloodResponse with discharge and statistics.
        """
        variables = self.variables.BASIC + self.variables.STATISTICS + self.variables.PERCENTILES

        return self.get(
            latitude=latitude,
            longitude=longitude,
            daily=variables,
            forecast_days=forecast_days,
        )
