"""
Weather Forecast API client.

Provides access to weather forecasts with up to 16 days of predictions.
Supports hourly, daily, current, and 15-minutely data.
"""

from typing import List, Optional, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import WeatherResponse
from openmeteo.validators import (
    VariableValidator,
    DateValidator,
    ValidationError,
)


class ForecastVariables:
    """Available variables for Weather Forecast API."""

    # Hourly temperature variables
    HOURLY_TEMPERATURE = [
        "temperature_2m",
        "apparent_temperature",
        "dew_point_2m",
        "wet_bulb_temperature_2m",
    ]

    # Hourly humidity and pressure
    HOURLY_HUMIDITY_PRESSURE = [
        "relative_humidity_2m",
        "surface_pressure",
        "pressure_msl",
        "vapour_pressure_deficit",
    ]

    # Hourly precipitation
    HOURLY_PRECIPITATION = [
        "precipitation",
        "rain",
        "showers",
        "snowfall",
        "snow_depth",
        "precipitation_probability",
        "freezing_level_height",
    ]

    # Hourly wind
    HOURLY_WIND = [
        "wind_speed_10m",
        "wind_speed_80m",
        "wind_speed_120m",
        "wind_speed_180m",
        "wind_direction_10m",
        "wind_direction_80m",
        "wind_direction_120m",
        "wind_direction_180m",
        "wind_gusts_10m",
    ]

    # Hourly clouds and visibility
    HOURLY_CLOUDS = [
        "cloud_cover",
        "cloud_cover_low",
        "cloud_cover_mid",
        "cloud_cover_high",
        "visibility",
    ]

    # Hourly solar radiation
    HOURLY_SOLAR = [
        "shortwave_radiation",
        "direct_radiation",
        "diffuse_radiation",
        "direct_normal_irradiance",
        "global_tilted_irradiance",
        "terrestrial_radiation",
        "shortwave_radiation_instant",
        "direct_radiation_instant",
        "diffuse_radiation_instant",
        "direct_normal_irradiance_instant",
        "global_tilted_irradiance_instant",
        "terrestrial_radiation_instant",
    ]

    # Hourly soil
    HOURLY_SOIL = [
        "soil_temperature_0cm",
        "soil_temperature_6cm",
        "soil_temperature_18cm",
        "soil_temperature_54cm",
        "soil_moisture_0_to_1cm",
        "soil_moisture_1_to_3cm",
        "soil_moisture_3_to_9cm",
        "soil_moisture_9_to_27cm",
        "soil_moisture_27_to_81cm",
    ]

    # Hourly other
    HOURLY_OTHER = [
        "weather_code",
        "cape",
        "lifted_index",
        "convective_inhibition",
        "evapotranspiration",
        "et0_fao_evapotranspiration",
        "uv_index",
        "uv_index_clear_sky",
        "is_day",
        "sunshine_duration",
        "total_column_integrated_water_vapour",
        "boundary_layer_height",
    ]

    # All hourly variables
    HOURLY_ALL = (
        HOURLY_TEMPERATURE
        + HOURLY_HUMIDITY_PRESSURE
        + HOURLY_PRECIPITATION
        + HOURLY_WIND
        + HOURLY_CLOUDS
        + HOURLY_SOLAR
        + HOURLY_SOIL
        + HOURLY_OTHER
    )

    # Daily variables
    DAILY_ALL = [
        "weather_code",
        "temperature_2m_max",
        "temperature_2m_min",
        "temperature_2m_mean",
        "apparent_temperature_max",
        "apparent_temperature_min",
        "apparent_temperature_mean",
        "sunrise",
        "sunset",
        "daylight_duration",
        "sunshine_duration",
        "uv_index_max",
        "uv_index_clear_sky_max",
        "precipitation_sum",
        "rain_sum",
        "showers_sum",
        "snowfall_sum",
        "precipitation_hours",
        "precipitation_probability_max",
        "precipitation_probability_min",
        "precipitation_probability_mean",
        "wind_speed_10m_max",
        "wind_gusts_10m_max",
        "wind_direction_10m_dominant",
        "shortwave_radiation_sum",
        "et0_fao_evapotranspiration",
    ]

    # Current variables (same as hourly)
    CURRENT_ALL = HOURLY_ALL

    # Common variable presets
    BASIC = ["temperature_2m", "relative_humidity_2m", "wind_speed_10m", "weather_code"]
    TEMPERATURE = ["temperature_2m", "apparent_temperature", "dew_point_2m"]
    PRECIPITATION = ["precipitation", "rain", "showers", "snowfall", "precipitation_probability"]
    WIND = ["wind_speed_10m", "wind_direction_10m", "wind_gusts_10m"]
    SOLAR = ["shortwave_radiation", "direct_radiation", "diffuse_radiation", "uv_index"]


class ForecastClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Weather Forecast API.

    Provides weather forecasts with up to 16 days of predictions.

    Example:
        >>> client = ForecastClient()
        >>> response = client.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     hourly=["temperature_2m", "precipitation"]
        ... )
        >>> df = response.to_dataframe()
    """

    BASE_URL = "https://api.open-meteo.com/v1/forecast"

    # Available weather models
    MODELS = [
        "best_match",
        "ecmwf_ifs04",
        "ecmwf_aifs025",
        "gfs_seamless",
        "gfs_global",
        "gfs_hrrr",
        "meteofrance_seamless",
        "meteofrance_arpege_world",
        "meteofrance_arpege_europe",
        "meteofrance_arome_france",
        "meteofrance_arome_france_hd",
        "jma_seamless",
        "jma_msm",
        "jma_gsm",
        "gem_seamless",
        "gem_global",
        "gem_regional",
        "gem_hrdps_continental",
        "icon_seamless",
        "icon_global",
        "icon_eu",
        "icon_d2",
        "dwd_icon_seamless",
        "dwd_icon_global",
        "dwd_icon_eu",
        "dwd_icon_d2",
        "metno_seamless",
        "metno_nordic",
        "knmi_seamless",
        "knmi_harmonie_arome_europe",
        "knmi_harmonie_arome_netherlands",
        "dmi_seamless",
        "dmi_harmonie_arome_europe",
        "ukmo_seamless",
        "ukmo_global_deterministic_10km",
        "ukmo_uk_deterministic_2km",
        "bom_access_global",
        "cma_grapes_global",
    ]

    def __init__(self, config: Optional[APIConfig] = None):
        super().__init__(config)
        self.variables = ForecastVariables

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        hourly: Optional[List[str]] = None,
        daily: Optional[List[str]] = None,
        current: Optional[List[str]] = None,
        minutely_15: Optional[List[str]] = None,
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
        timezone: str = "UTC",
        forecast_days: Optional[int] = None,
        forecast_hours: Optional[int] = None,
        forecast_minutely_15: Optional[int] = None,
        past_days: Optional[int] = None,
        past_hours: Optional[int] = None,
        past_minutely_15: Optional[int] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
        start_hour: Optional[str] = None,
        end_hour: Optional[str] = None,
        start_minutely_15: Optional[str] = None,
        end_minutely_15: Optional[str] = None,
        models: Optional[List[str]] = None,
        cell_selection: str = "land",
        elevation: Optional[float] = None,
        tilt: Optional[float] = None,
        azimuth: Optional[float] = None,
    ) -> WeatherResponse:
        """
        Get weather forecast data.

        Args:
            latitude: Latitude (-90 to 90). Can be a list for multiple locations.
            longitude: Longitude (-180 to 180). Can be a list for multiple locations.
            hourly: List of hourly variables to fetch.
            daily: List of daily variables to fetch.
            current: List of current variables to fetch.
            minutely_15: List of 15-minute variables to fetch.
            temperature_unit: Temperature unit ('celsius' or 'fahrenheit').
            wind_speed_unit: Wind speed unit ('kmh', 'ms', 'mph', 'kn').
            precipitation_unit: Precipitation unit ('mm' or 'inch').
            timezone: Timezone for timestamps (default: UTC).
            forecast_days: Number of forecast days (0-16).
            forecast_hours: Number of forecast hours.
            forecast_minutely_15: Number of 15-minute forecast periods.
            past_days: Number of past days to include (0-92).
            past_hours: Number of past hours to include.
            past_minutely_15: Number of past 15-minute periods.
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            start_hour: Start hour (YYYY-MM-DDTHH:MM).
            end_hour: End hour (YYYY-MM-DDTHH:MM).
            start_minutely_15: Start time for 15-minute data.
            end_minutely_15: End time for 15-minute data.
            models: List of weather models to use.
            cell_selection: Grid cell selection ('land', 'sea', 'nearest').
            elevation: Manual elevation in meters.
            tilt: Solar panel tilt for GTI calculation.
            azimuth: Solar panel azimuth for GTI calculation.

        Returns:
            WeatherResponse with requested data.

        Raises:
            ValidationError: If parameters are invalid.
        """
        # Validate coordinates
        self._validate_coordinates(latitude, longitude)

        # Validate dates if provided
        if start_date:
            self._validate_date(start_date, "start_date")
        if end_date:
            self._validate_date(end_date, "end_date")
        if start_date and end_date:
            self._validate_date_range(start_date, end_date)

        # Validate forecast/past days
        DateValidator.validate_forecast_days(forecast_days, max_days=16)
        DateValidator.validate_past_days(past_days, max_days=92)

        # Validate variables against known valid sets
        VariableValidator.validate_variables(
            hourly, VariableValidator.FORECAST_HOURLY, "hourly"
        )
        VariableValidator.validate_variables(
            daily, VariableValidator.FORECAST_DAILY, "daily"
        )
        VariableValidator.validate_variables(
            current, VariableValidator.FORECAST_HOURLY, "current"
        )

        # Sanitize units
        temperature_unit = self._sanitize_unit(
            temperature_unit, ["celsius", "fahrenheit"], "celsius"
        )
        wind_speed_unit = self._sanitize_unit(
            wind_speed_unit, ["kmh", "ms", "mph", "kn"], "kmh"
        )
        precipitation_unit = self._sanitize_unit(
            precipitation_unit, ["mm", "inch"], "mm"
        )

        # Sanitize timezone
        timezone = self._sanitize_timezone(timezone)

        # Validate cell_selection
        if cell_selection not in ["land", "sea", "nearest"]:
            raise ValidationError(
                f"Invalid cell_selection: {cell_selection}. "
                "Must be 'land', 'sea', or 'nearest'."
            )

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
            hourly=hourly,
            daily=daily,
            current=current,
            minutely_15=minutely_15,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
            timezone=timezone,
            forecast_days=forecast_days,
            forecast_hours=forecast_hours,
            forecast_minutely_15=forecast_minutely_15,
            past_days=past_days,
            past_hours=past_hours,
            past_minutely_15=past_minutely_15,
            start_date=start_date,
            end_date=end_date,
            start_hour=start_hour,
            end_hour=end_hour,
            start_minutely_15=start_minutely_15,
            end_minutely_15=end_minutely_15,
            models=models,
            cell_selection=cell_selection,
            elevation=elevation,
            tilt=tilt,
            azimuth=azimuth,
        )

        data = self._make_request(self.BASE_URL, params)
        return WeatherResponse.from_response(data)

    def get_current(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        timezone: str = "UTC",
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
    ) -> WeatherResponse:
        """
        Get current weather conditions.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of current variables (default: basic set).
            timezone: Timezone for timestamps.
            temperature_unit: Temperature unit.
            wind_speed_unit: Wind speed unit.
            precipitation_unit: Precipitation unit.

        Returns:
            WeatherResponse with current conditions.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            current=variables,
            timezone=timezone,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
        )

    def get_hourly(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        forecast_days: int = 7,
        timezone: str = "UTC",
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
    ) -> WeatherResponse:
        """
        Get hourly forecast data.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of hourly variables (default: basic set).
            forecast_days: Number of forecast days (0-16).
            timezone: Timezone for timestamps.
            temperature_unit: Temperature unit.
            wind_speed_unit: Wind speed unit.
            precipitation_unit: Precipitation unit.

        Returns:
            WeatherResponse with hourly data.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            hourly=variables,
            forecast_days=forecast_days,
            timezone=timezone,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
        )

    def get_daily(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        forecast_days: int = 7,
        timezone: str = "UTC",
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
    ) -> WeatherResponse:
        """
        Get daily forecast data.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of daily variables (default: common set).
            forecast_days: Number of forecast days (0-16).
            timezone: Timezone for timestamps.
            temperature_unit: Temperature unit.
            wind_speed_unit: Wind speed unit.
            precipitation_unit: Precipitation unit.

        Returns:
            WeatherResponse with daily data.
        """
        if variables is None:
            variables = [
                "weather_code",
                "temperature_2m_max",
                "temperature_2m_min",
                "precipitation_sum",
                "wind_speed_10m_max",
            ]

        return self.get(
            latitude=latitude,
            longitude=longitude,
            daily=variables,
            forecast_days=forecast_days,
            timezone=timezone,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
        )
