"""
Geocoding API client.

Provides location search and coordinate lookup functionality.
"""

from typing import List, Optional
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import GeocodingResult


class GeocodingClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Geocoding API.

    Converts location names to coordinates and vice versa.

    Example:
        >>> client = GeocodingClient()
        >>> results = client.search("Berlin")
        >>> print(results[0].latitude, results[0].longitude)
        52.52437 13.41053
    """

    BASE_URL = "https://geocoding-api.open-meteo.com/v1/search"
    GET_URL = "https://geocoding-api.open-meteo.com/v1/get"

    def search(
        self,
        name: str,
        count: int = 10,
        language: str = "en",
        country_code: Optional[str] = None,
        format: str = "json",
    ) -> List[GeocodingResult]:
        """
        Search for locations by name.

        Args:
            name: Location name to search for (minimum 2 characters).
            count: Maximum number of results (1-100).
            language: Language for results (ISO 639-1).
            country_code: Filter by country (ISO 3166-1 alpha-2).
            format: Response format ('json' or 'protobuf').

        Returns:
            List of GeocodingResult objects.

        Raises:
            ValueError: If name is too short.
        """
        if len(name) < 2:
            raise ValueError("Search name must be at least 2 characters")

        params = self._build_params(
            name=name,
            count=count,
            language=language,
            countryCode=country_code,
            format=format,
        )

        data = self._make_request(self.BASE_URL, params)

        results = []
        for item in data.get("results", []):
            results.append(GeocodingResult.from_response(item))

        return results

    def get_by_id(self, location_id: int) -> Optional[GeocodingResult]:
        """
        Get location details by ID.

        Args:
            location_id: Location ID from previous search.

        Returns:
            GeocodingResult or None if not found.
        """
        params = self._build_params(id=location_id)

        data = self._make_request(self.GET_URL, params)

        if data:
            return GeocodingResult.from_response(data)
        return None

    def search_city(
        self,
        name: str,
        country_code: Optional[str] = None,
        language: str = "en",
    ) -> Optional[GeocodingResult]:
        """
        Search for a city and return the best match.

        Args:
            name: City name to search for.
            country_code: Optional country filter.
            language: Language for results.

        Returns:
            Best matching GeocodingResult or None.
        """
        results = self.search(
            name=name,
            count=1,
            language=language,
            country_code=country_code,
        )
        return results[0] if results else None

    def get_coordinates(
        self,
        name: str,
        country_code: Optional[str] = None,
    ) -> Optional[tuple]:
        """
        Get coordinates for a location name.

        Args:
            name: Location name to search for.
            country_code: Optional country filter.

        Returns:
            Tuple of (latitude, longitude) or None.
        """
        result = self.search_city(name, country_code)
        if result:
            return (result.latitude, result.longitude)
        return None
