"""
Historical Weather API client.

Provides access to historical weather data from 1940 onwards.
"""

from typing import List, Optional, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import WeatherResponse
from openmeteo.validators import (
    VariableValidator,
    DateValidator,
    ValidationError,
)


class HistoricalVariables:
    """Available variables for Historical Weather API."""

    # Hourly temperature variables
    HOURLY_TEMPERATURE = [
        "temperature_2m",
        "apparent_temperature",
        "dew_point_2m",
        "wet_bulb_temperature_2m",
    ]

    # Hourly humidity and pressure
    HOURLY_HUMIDITY_PRESSURE = [
        "relative_humidity_2m",
        "surface_pressure",
        "pressure_msl",
        "vapour_pressure_deficit",
    ]

    # Hourly precipitation
    HOURLY_PRECIPITATION = [
        "precipitation",
        "rain",
        "snowfall",
        "snow_depth",
    ]

    # Hourly wind
    HOURLY_WIND = [
        "wind_speed_10m",
        "wind_speed_100m",
        "wind_direction_10m",
        "wind_direction_100m",
        "wind_gusts_10m",
    ]

    # Hourly clouds
    HOURLY_CLOUDS = [
        "cloud_cover",
        "cloud_cover_low",
        "cloud_cover_mid",
        "cloud_cover_high",
    ]

    # Hourly solar radiation
    HOURLY_SOLAR = [
        "shortwave_radiation",
        "direct_radiation",
        "direct_normal_irradiance",
        "diffuse_radiation",
        "global_tilted_irradiance",
        "terrestrial_radiation",
        "sunshine_duration",
    ]

    # Hourly soil
    HOURLY_SOIL = [
        "soil_temperature_0_to_7cm",
        "soil_temperature_7_to_28cm",
        "soil_temperature_28_to_100cm",
        "soil_temperature_100_to_255cm",
        "soil_moisture_0_to_7cm",
        "soil_moisture_7_to_28cm",
        "soil_moisture_28_to_100cm",
        "soil_moisture_100_to_255cm",
    ]

    # Hourly other
    HOURLY_OTHER = [
        "weather_code",
        "et0_fao_evapotranspiration",
        "boundary_layer_height",
        "total_column_integrated_water_vapour",
        "is_day",
    ]

    # All hourly variables
    HOURLY_ALL = (
        HOURLY_TEMPERATURE
        + HOURLY_HUMIDITY_PRESSURE
        + HOURLY_PRECIPITATION
        + HOURLY_WIND
        + HOURLY_CLOUDS
        + HOURLY_SOLAR
        + HOURLY_SOIL
        + HOURLY_OTHER
    )

    # Daily variables
    DAILY_ALL = [
        "weather_code",
        "temperature_2m_max",
        "temperature_2m_min",
        "temperature_2m_mean",
        "apparent_temperature_max",
        "apparent_temperature_min",
        "apparent_temperature_mean",
        "sunrise",
        "sunset",
        "daylight_duration",
        "sunshine_duration",
        "precipitation_sum",
        "rain_sum",
        "snowfall_sum",
        "precipitation_hours",
        "wind_speed_10m_max",
        "wind_gusts_10m_max",
        "wind_direction_10m_dominant",
        "shortwave_radiation_sum",
        "et0_fao_evapotranspiration",
    ]

    # Common variable presets
    BASIC = ["temperature_2m", "relative_humidity_2m", "wind_speed_10m"]
    TEMPERATURE = ["temperature_2m", "apparent_temperature", "dew_point_2m"]
    PRECIPITATION = ["precipitation", "rain", "snowfall"]
    WIND = ["wind_speed_10m", "wind_direction_10m", "wind_gusts_10m"]
    SOLAR = ["shortwave_radiation", "direct_radiation", "diffuse_radiation"]


class HistoricalClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Historical Weather API.

    Provides access to historical weather data from 1940 onwards.

    Example:
        >>> client = HistoricalClient()
        >>> response = client.get(
        ...     latitude=52.52,
        ...     longitude=13.41,
        ...     start_date="2023-01-01",
        ...     end_date="2023-01-31",
        ...     hourly=["temperature_2m", "precipitation"]
        ... )
        >>> df = response.to_dataframe()
    """

    BASE_URL = "https://archive-api.open-meteo.com/v1/archive"

    # Available reanalysis models
    MODELS = [
        "best_match",
        "era5_seamless",
        "era5",
        "era5_land",
        "cerra",
        "ecmwf_ifs",
    ]

    def __init__(self, config: Optional[APIConfig] = None):
        super().__init__(config)
        self.variables = HistoricalVariables

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        start_date: str,
        end_date: str,
        hourly: Optional[List[str]] = None,
        daily: Optional[List[str]] = None,
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
        timezone: str = "UTC",
        models: Optional[List[str]] = None,
        cell_selection: str = "land",
        elevation: Optional[float] = None,
        tilt: Optional[float] = None,
        azimuth: Optional[float] = None,
    ) -> WeatherResponse:
        """
        Get historical weather data.

        Args:
            latitude: Latitude (-90 to 90). Can be a list for multiple locations.
            longitude: Longitude (-180 to 180). Can be a list for multiple locations.
            start_date: Start date (YYYY-MM-DD). Data available from 1940.
            end_date: End date (YYYY-MM-DD).
            hourly: List of hourly variables to fetch.
            daily: List of daily variables to fetch.
            temperature_unit: Temperature unit ('celsius' or 'fahrenheit').
            wind_speed_unit: Wind speed unit ('kmh', 'ms', 'mph', 'kn').
            precipitation_unit: Precipitation unit ('mm' or 'inch').
            timezone: Timezone for timestamps (default: UTC).
            models: List of reanalysis models to use.
            cell_selection: Grid cell selection ('land', 'sea', 'nearest').
            elevation: Manual elevation in meters.
            tilt: Solar panel tilt for GTI calculation.
            azimuth: Solar panel azimuth for GTI calculation.

        Returns:
            WeatherResponse with requested data.

        Raises:
            ValidationError: If parameters are invalid.
        """
        # Validate coordinates
        self._validate_coordinates(latitude, longitude)

        # Validate date range with historical data limits
        self._validate_date_range(
            start_date,
            end_date,
            min_date=DateValidator.HISTORICAL_START,
        )

        # Sanitize units
        temperature_unit = self._sanitize_unit(
            temperature_unit, ["celsius", "fahrenheit"], "celsius"
        )
        wind_speed_unit = self._sanitize_unit(
            wind_speed_unit, ["kmh", "ms", "mph", "kn"], "kmh"
        )
        precipitation_unit = self._sanitize_unit(
            precipitation_unit, ["mm", "inch"], "mm"
        )

        # Sanitize timezone
        timezone = self._sanitize_timezone(timezone)

        # Validate cell_selection
        if cell_selection not in ["land", "sea", "nearest"]:
            raise ValidationError(
                f"Invalid cell_selection: {cell_selection}. "
                "Must be 'land', 'sea', or 'nearest'."
            )

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
            start_date=start_date,
            end_date=end_date,
            hourly=hourly,
            daily=daily,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
            timezone=timezone,
            models=models,
            cell_selection=cell_selection,
            elevation=elevation,
            tilt=tilt,
            azimuth=azimuth,
        )

        data = self._make_request(self.BASE_URL, params)
        return WeatherResponse.from_response(data)

    def get_hourly(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        start_date: str,
        end_date: str,
        variables: Optional[List[str]] = None,
        timezone: str = "UTC",
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
    ) -> WeatherResponse:
        """
        Get hourly historical data.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            variables: List of hourly variables (default: basic set).
            timezone: Timezone for timestamps.
            temperature_unit: Temperature unit.
            wind_speed_unit: Wind speed unit.
            precipitation_unit: Precipitation unit.

        Returns:
            WeatherResponse with hourly data.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            start_date=start_date,
            end_date=end_date,
            hourly=variables,
            timezone=timezone,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
        )

    def get_daily(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        start_date: str,
        end_date: str,
        variables: Optional[List[str]] = None,
        timezone: str = "UTC",
        temperature_unit: str = "celsius",
        wind_speed_unit: str = "kmh",
        precipitation_unit: str = "mm",
    ) -> WeatherResponse:
        """
        Get daily historical data.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            variables: List of daily variables (default: common set).
            timezone: Timezone for timestamps.
            temperature_unit: Temperature unit.
            wind_speed_unit: Wind speed unit.
            precipitation_unit: Precipitation unit.

        Returns:
            WeatherResponse with daily data.
        """
        if variables is None:
            variables = [
                "weather_code",
                "temperature_2m_max",
                "temperature_2m_min",
                "precipitation_sum",
                "wind_speed_10m_max",
            ]

        return self.get(
            latitude=latitude,
            longitude=longitude,
            start_date=start_date,
            end_date=end_date,
            daily=variables,
            timezone=timezone,
            temperature_unit=temperature_unit,
            wind_speed_unit=wind_speed_unit,
            precipitation_unit=precipitation_unit,
        )
