"""
Marine Weather API client.

Provides access to marine weather forecasts including wave heights,
ocean currents, and sea surface temperature.
"""

from typing import List, Optional, Union
from openmeteo.base import OpenMeteoBaseClient, APIConfig
from openmeteo.models import WeatherResponse


class MarineVariables:
    """Available variables for Marine Weather API."""

    # Wave variables
    WAVES = [
        "wave_height",
        "wave_direction",
        "wave_period",
    ]

    # Wind wave variables
    WIND_WAVES = [
        "wind_wave_height",
        "wind_wave_direction",
        "wind_wave_period",
        "wind_wave_peak_period",
    ]

    # Swell variables
    SWELL_PRIMARY = [
        "swell_wave_height",
        "swell_wave_direction",
        "swell_wave_period",
        "swell_wave_peak_period",
    ]

    SWELL_SECONDARY = [
        "secondary_swell_wave_height",
        "secondary_swell_wave_direction",
        "secondary_swell_wave_period",
        "secondary_swell_wave_peak_period",
    ]

    SWELL_TERTIARY = [
        "tertiary_swell_wave_height",
        "tertiary_swell_wave_direction",
        "tertiary_swell_wave_period",
        "tertiary_swell_wave_peak_period",
    ]

    # Ocean variables
    OCEAN = [
        "ocean_current_velocity",
        "ocean_current_direction",
        "sea_surface_temperature",
        "sea_level_height_msl",
        "sea_level_height_absolute",
    ]

    # All hourly variables
    HOURLY_ALL = WAVES + WIND_WAVES + SWELL_PRIMARY + SWELL_SECONDARY + SWELL_TERTIARY + OCEAN

    # Daily variables
    DAILY_ALL = [
        "wave_height_max",
        "wave_direction_dominant",
        "wave_period_max",
        "wind_wave_height_max",
        "wind_wave_direction_dominant",
        "wind_wave_period_max",
        "wind_wave_peak_period_max",
        "swell_wave_height_max",
        "swell_wave_direction_dominant",
        "swell_wave_period_max",
        "swell_wave_peak_period_max",
    ]

    # Current variables
    CURRENT_ALL = HOURLY_ALL

    # Common presets
    BASIC = ["wave_height", "wave_direction", "wave_period", "sea_surface_temperature"]
    WAVES_ALL = WAVES + WIND_WAVES
    SWELL_ALL = SWELL_PRIMARY + SWELL_SECONDARY + SWELL_TERTIARY


class MarineClient(OpenMeteoBaseClient):
    """
    Client for Open-Meteo Marine Weather API.

    Provides marine weather forecasts for up to 8 days.

    Example:
        >>> client = MarineClient()
        >>> response = client.get(
        ...     latitude=54.32,
        ...     longitude=10.13,
        ...     hourly=["wave_height", "wave_direction", "sea_surface_temperature"]
        ... )
        >>> df = response.to_dataframe()
    """

    BASE_URL = "https://marine-api.open-meteo.com/v1/marine"

    def __init__(self, config: Optional[APIConfig] = None):
        super().__init__(config)
        self.variables = MarineVariables

    def get(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        hourly: Optional[List[str]] = None,
        daily: Optional[List[str]] = None,
        current: Optional[List[str]] = None,
        timezone: str = "UTC",
        length_unit: str = "metric",
        forecast_days: Optional[int] = None,
        forecast_hours: Optional[int] = None,
        past_days: Optional[int] = None,
        past_hours: Optional[int] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
        start_hour: Optional[str] = None,
        end_hour: Optional[str] = None,
        cell_selection: str = "sea",
    ) -> WeatherResponse:
        """
        Get marine weather data.

        Args:
            latitude: Latitude (-90 to 90). Can be a list for multiple locations.
            longitude: Longitude (-180 to 180). Can be a list for multiple locations.
            hourly: List of hourly variables to fetch.
            daily: List of daily variables to fetch.
            current: List of current variables to fetch.
            timezone: Timezone for timestamps (default: UTC).
            length_unit: Length unit ('metric' or 'imperial').
            forecast_days: Number of forecast days (0-8).
            forecast_hours: Number of forecast hours.
            past_days: Number of past days (0-92).
            past_hours: Number of past hours.
            start_date: Start date (YYYY-MM-DD).
            end_date: End date (YYYY-MM-DD).
            start_hour: Start hour (YYYY-MM-DDTHH:MM).
            end_hour: End hour (YYYY-MM-DDTHH:MM).
            cell_selection: Grid cell selection ('sea', 'land', 'nearest').

        Returns:
            WeatherResponse with requested data.
        """
        self._validate_coordinates(latitude, longitude)

        if start_date:
            self._validate_date(start_date, "start_date")
        if end_date:
            self._validate_date(end_date, "end_date")

        params = self._build_params(
            latitude=latitude,
            longitude=longitude,
            hourly=hourly,
            daily=daily,
            current=current,
            timezone=timezone,
            length_unit=length_unit,
            forecast_days=forecast_days,
            forecast_hours=forecast_hours,
            past_days=past_days,
            past_hours=past_hours,
            start_date=start_date,
            end_date=end_date,
            start_hour=start_hour,
            end_hour=end_hour,
            cell_selection=cell_selection,
        )

        data = self._make_request(self.BASE_URL, params)
        return WeatherResponse.from_response(data)

    def get_current(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        timezone: str = "UTC",
        length_unit: str = "metric",
    ) -> WeatherResponse:
        """
        Get current marine conditions.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of current variables (default: basic set).
            timezone: Timezone for timestamps.
            length_unit: Length unit.

        Returns:
            WeatherResponse with current conditions.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            current=variables,
            timezone=timezone,
            length_unit=length_unit,
        )

    def get_hourly(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        forecast_days: int = 5,
        timezone: str = "UTC",
        length_unit: str = "metric",
    ) -> WeatherResponse:
        """
        Get hourly marine forecast.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of hourly variables (default: basic set).
            forecast_days: Number of forecast days (0-8).
            timezone: Timezone for timestamps.
            length_unit: Length unit.

        Returns:
            WeatherResponse with hourly data.
        """
        if variables is None:
            variables = self.variables.BASIC

        return self.get(
            latitude=latitude,
            longitude=longitude,
            hourly=variables,
            forecast_days=forecast_days,
            timezone=timezone,
            length_unit=length_unit,
        )

    def get_daily(
        self,
        latitude: Union[float, List[float]],
        longitude: Union[float, List[float]],
        variables: Optional[List[str]] = None,
        forecast_days: int = 5,
        timezone: str = "UTC",
        length_unit: str = "metric",
    ) -> WeatherResponse:
        """
        Get daily marine forecast.

        Args:
            latitude: Latitude (-90 to 90).
            longitude: Longitude (-180 to 180).
            variables: List of daily variables.
            forecast_days: Number of forecast days (0-8).
            timezone: Timezone for timestamps.
            length_unit: Length unit.

        Returns:
            WeatherResponse with daily data.
        """
        if variables is None:
            variables = [
                "wave_height_max",
                "wave_direction_dominant",
                "wave_period_max",
            ]

        return self.get(
            latitude=latitude,
            longitude=longitude,
            daily=variables,
            forecast_days=forecast_days,
            timezone=timezone,
            length_unit=length_unit,
        )
