"""
Open-Meteo FlatBuffers Schema Definitions.

This module contains the schema definitions for parsing Open-Meteo API
FlatBuffers responses. It's a standalone implementation that doesn't
depend on openmeteo-sdk.

Based on: https://github.com/open-meteo/sdk/blob/main/flatbuffers/weather_api.fbs
"""

from enum import IntEnum


class Unit(IntEnum):
    """Measurement units used in Open-Meteo API responses."""
    undefined = 0
    celsius = 1
    centimetre = 2
    cubic_metre_per_cubic_metre = 3
    cubic_metre_per_second = 4
    degree_direction = 5
    dimensionless = 6
    dimensionless_integer = 7
    euro_per_megawatt_hour = 8
    fahrenheit = 9
    feet = 10
    gigajoule_per_kilogram = 11
    grams_per_cubic_metre = 12
    grains_per_cubic_metre = 13
    hectopascal = 14
    hours = 15
    inch = 16
    iso8601 = 17
    joule_per_kilogram = 18
    kelvin = 19
    kilojoule_per_kilogram = 20
    kilogram_per_square_metre = 21
    kilometres_per_hour = 22
    knots = 23
    megajoule_per_square_metre = 24
    metre = 25
    metres_per_second = 26
    micrograms_per_cubic_metre = 27
    miles_per_hour = 28
    millimetre = 29
    minutes = 30
    pascal = 31
    percentage = 32
    seconds = 33
    unix_time = 34
    watt_per_square_metre = 35
    wmo_code = 36


class Aggregation(IntEnum):
    """Aggregation methods for daily/monthly variables."""
    none = 0
    minimum = 1
    maximum = 2
    mean = 3
    p10 = 4
    p25 = 5
    median = 6
    p75 = 7
    p90 = 8
    dominant = 9
    sum = 10
    standard_deviation = 11


class Variable(IntEnum):
    """Weather variables available in Open-Meteo API."""
    undefined = 0
    apparent_temperature = 1
    cape = 2
    cloud_cover = 3
    cloud_cover_high = 4
    cloud_cover_low = 5
    cloud_cover_mid = 6
    daylight_duration = 7
    dew_point = 8
    diffuse_radiation = 9
    diffuse_radiation_instant = 10
    direct_normal_irradiance = 11
    direct_normal_irradiance_instant = 12
    direct_radiation = 13
    direct_radiation_instant = 14
    et0_fao_evapotranspiration = 15
    evapotranspiration = 16
    freezing_level_height = 17
    global_tilted_irradiance = 18
    global_tilted_irradiance_instant = 19
    growing_degree_days = 20
    is_day = 21
    latent_heat_flux = 22
    leaf_wetness_probability = 23
    lifted_index = 24
    lightning_potential = 25
    precipitation = 26
    precipitation_hours = 27
    precipitation_probability = 28
    pressure_msl = 29
    rain = 30
    relative_humidity = 31
    runoff = 32
    sensible_heat_flux = 33
    shortwave_radiation = 34
    shortwave_radiation_instant = 35
    showers = 36
    snow_depth = 37
    snow_height = 38
    snowfall = 39
    snowfall_height = 40
    snowfall_water_equivalent = 41
    soil_moisture = 42
    soil_moisture_index = 43
    soil_temperature = 44
    sunrise = 45
    sunset = 46
    sunshine_duration = 47
    surface_pressure = 48
    surface_temperature = 49
    temperature = 50
    terrestrial_radiation = 51
    terrestrial_radiation_instant = 52
    total_column_integrated_water_vapour = 53
    updraft = 54
    uv_index = 55
    uv_index_clear_sky = 56
    vapour_pressure_deficit = 57
    visibility = 58
    weather_code = 59
    wind_direction = 60
    wind_gusts = 61
    wind_speed = 62
    # Air quality variables
    aerosol_optical_depth = 63
    alder_pollen = 64
    ammonia = 65
    birch_pollen = 66
    carbon_dioxide = 67
    carbon_monoxide = 68
    dust = 69
    european_aqi = 70
    european_aqi_nitrogen_dioxide = 71
    european_aqi_ozone = 72
    european_aqi_pm10 = 73
    european_aqi_pm2_5 = 74
    european_aqi_sulphur_dioxide = 75
    grass_pollen = 76
    methane = 77
    mugwort_pollen = 78
    nitrogen_dioxide = 79
    nitrogen_monoxide = 80
    olive_pollen = 81
    ozone = 82
    pm10 = 83
    pm2_5 = 84
    ragweed_pollen = 85
    sulphur_dioxide = 86
    us_aqi = 87
    us_aqi_carbon_monoxide = 88
    us_aqi_nitrogen_dioxide = 89
    us_aqi_ozone = 90
    us_aqi_pm10 = 91
    us_aqi_pm2_5 = 92
    us_aqi_sulphur_dioxide = 93
    # Marine variables
    ocean_current_direction = 94
    ocean_current_velocity = 95
    sea_surface_temperature = 96
    swell_wave_direction = 97
    swell_wave_height = 98
    swell_wave_peak_period = 99
    swell_wave_period = 100
    wave_direction = 101
    wave_height = 102
    wave_peak_period = 103
    wave_period = 104
    # Flood variables
    river_discharge = 105
    # Additional variables
    convective_inhibition = 106
    geopotential_height = 107
    boundary_layer_height = 108
    skin_temperature = 109
    wet_bulb_temperature = 110


# Mapping from variable enum to string name (for API compatibility)
VARIABLE_NAMES = {v: v.name for v in Variable}

# Mapping from string to variable enum
VARIABLE_FROM_NAME = {v.name: v for v in Variable}

# Unit names mapping
UNIT_NAMES = {u: u.name for u in Unit}
