"""
DAG Template Generator for Open-Meteo API.

Generates standalone Python scripts for fetching weather data,
compatible with Airflow, cron jobs, or direct execution.
"""

from typing import List, Dict, Optional, Any
from datetime import datetime
import json

from openmeteo.validators import (
    CoordinateValidator,
    ValidationError,
)


class DagGenerator:
    """
    Generator for standalone weather data fetching scripts.

    Generates Python scripts that can be used as:
    - Airflow DAGs
    - Standalone scripts
    - Cron jobs
    """

    # API endpoint configurations
    API_CONFIGS = {
        "forecast": {
            "name": "Weather Forecast",
            "client": "forecast",
            "method": "get",
            "default_params": {
                "hourly": ["temperature_2m", "precipitation", "wind_speed_10m", "weather_code"],
                "forecast_days": 7,
            },
            "requires_dates": False,
        },
        "historical": {
            "name": "Historical Weather",
            "client": "historical",
            "method": "get",
            "default_params": {
                "hourly": ["temperature_2m", "precipitation", "wind_speed_10m"],
            },
            "requires_dates": True,
        },
        "air_quality": {
            "name": "Air Quality",
            "client": "air_quality",
            "method": "get",
            "default_params": {
                "hourly": ["pm10", "pm2_5", "european_aqi", "us_aqi"],
                "forecast_days": 5,
            },
            "requires_dates": False,
        },
        "marine": {
            "name": "Marine Weather",
            "client": "marine",
            "method": "get",
            "default_params": {
                "hourly": ["wave_height", "wave_direction", "wave_period", "sea_surface_temperature"],
                "forecast_days": 5,
            },
            "requires_dates": False,
        },
        "flood": {
            "name": "Flood/River Discharge",
            "client": "flood",
            "method": "get",
            "default_params": {
                "daily": ["river_discharge", "river_discharge_max", "river_discharge_min"],
                "forecast_days": 92,
            },
            "requires_dates": False,
        },
        "climate": {
            "name": "Climate Projections",
            "client": "climate",
            "method": "get",
            "default_params": {
                "models": ["EC_Earth3P_HR"],
                "daily": ["temperature_2m_max", "temperature_2m_min", "precipitation_sum"],
            },
            "requires_dates": True,
        },
        "ensemble": {
            "name": "Ensemble Models",
            "client": "ensemble",
            "method": "get",
            "default_params": {
                "models": ["icon_seamless"],
                "hourly": ["temperature_2m", "precipitation"],
                "forecast_days": 7,
            },
            "requires_dates": False,
        },
    }

    def __init__(
        self,
        apis: List[str],
        locations: List[Dict[str, Any]],
        output_format: str = "csv",
        output_dir: str = "./output",
        dag_id: Optional[str] = None,
        schedule: str = "@daily",
        include_airflow: bool = True,
    ):
        """
        Initialize the DAG generator.

        Args:
            apis: List of API names to include (forecast, historical, etc.)
            locations: List of location dicts with 'name', 'lat', 'lon'
            output_format: Output format ('csv' or 'json')
            output_dir: Directory to save output files
            dag_id: Airflow DAG ID (auto-generated if not provided)
            schedule: Cron schedule or Airflow preset (@daily, @hourly, etc.)
            include_airflow: Include Airflow DAG wrapper
        """
        self.apis = [api.lower() for api in apis]
        self.locations = locations
        self.output_format = output_format
        self.output_dir = output_dir
        self.dag_id = dag_id or f"openmeteo_{'_'.join(self.apis)}"
        self.schedule = schedule
        self.include_airflow = include_airflow

        # Validate APIs
        invalid_apis = set(self.apis) - set(self.API_CONFIGS.keys())
        if invalid_apis:
            raise ValueError(f"Invalid APIs: {invalid_apis}. Valid options: {list(self.API_CONFIGS.keys())}")

        # Validate locations
        self._validate_locations(locations)

        # Validate output format
        if output_format not in ["csv", "json"]:
            raise ValueError(f"Invalid output_format: {output_format}. Must be 'csv' or 'json'.")

    def _validate_locations(self, locations: List[Dict[str, Any]]) -> None:
        """Validate location configurations."""
        if not locations:
            raise ValueError("At least one location is required")

        for i, loc in enumerate(locations):
            # Check required fields
            if "name" not in loc:
                raise ValueError(f"Location {i} missing 'name' field")
            if "lat" not in loc:
                raise ValueError(f"Location '{loc.get('name', i)}' missing 'lat' field")
            if "lon" not in loc:
                raise ValueError(f"Location '{loc.get('name', i)}' missing 'lon' field")

            # Validate coordinates using the validator
            try:
                CoordinateValidator.validate_latitude(loc["lat"], f"Location '{loc['name']}' latitude")
                CoordinateValidator.validate_longitude(loc["lon"], f"Location '{loc['name']}' longitude")
            except ValidationError as e:
                raise ValueError(str(e))

    def generate(self) -> str:
        """Generate the complete Python script."""
        sections = [
            self._generate_header(),
            self._generate_imports(),
            self._generate_config(),
            self._generate_fetch_functions(),
            self._generate_main_function(),
        ]

        if self.include_airflow:
            sections.append(self._generate_airflow_dag())

        sections.append(self._generate_script_entry())

        return "\n\n".join(sections)

    def _generate_header(self) -> str:
        """Generate file header with documentation."""
        api_names = [self.API_CONFIGS[api]["name"] for api in self.apis]

        return f'''"""
Auto-generated Open-Meteo Data Fetcher

APIs: {", ".join(api_names)}
Generated: {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}

This script fetches weather data from Open-Meteo API and saves it to {self.output_format.upper()} files.
Can be run standalone or as an Airflow DAG.

Usage:
    python {self.dag_id}.py                    # Run standalone
    python {self.dag_id}.py --help             # Show options

Airflow:
    Copy to your DAGs folder and it will be auto-detected.
"""'''

    def _generate_imports(self) -> str:
        """Generate import statements."""
        imports = [
            "import os",
            "import sys",
            "import argparse",
            "import logging",
            "from datetime import datetime, timedelta",
            "from typing import List, Dict, Optional",
            "",
            "import requests",
            "import pandas as pd",
        ]

        if self.include_airflow:
            imports.extend([
                "",
                "# Airflow imports (optional - script works without Airflow)",
                "try:",
                "    from airflow import DAG",
                "    from airflow.operators.python import PythonOperator",
                "    AIRFLOW_AVAILABLE = True",
                "except ImportError:",
                "    AIRFLOW_AVAILABLE = False",
            ])

        return "\n".join(imports)

    def _generate_config(self) -> str:
        """Generate configuration section."""
        locations_str = json.dumps(self.locations, indent=4)

        config = f'''
# =============================================================================
# CONFIGURATION - Modify these settings as needed
# =============================================================================

# Locations to fetch data for
LOCATIONS = {locations_str}

# Output settings
OUTPUT_DIR = "{self.output_dir}"
OUTPUT_FORMAT = "{self.output_format}"  # 'csv' or 'json'

# API settings
API_TIMEOUT = 30
API_RETRIES = 3

# Logging
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)
'''

        # Add API-specific configs
        for api in self.apis:
            config_name = api.upper()
            params = self.API_CONFIGS[api]["default_params"]
            config += f"\n# {self.API_CONFIGS[api]['name']} settings\n"
            config += f"{config_name}_PARAMS = {json.dumps(params, indent=4)}\n"

        return config

    def _generate_fetch_functions(self) -> str:
        """Generate fetch functions for each API."""
        functions = [
            self._generate_validation_functions(),
            self._generate_base_request_function(),
        ]

        for api in self.apis:
            functions.append(self._generate_api_function(api))

        return "\n\n".join(functions)

    def _generate_validation_functions(self) -> str:
        """Generate validation functions to embed in the script."""
        return '''
# =============================================================================
# VALIDATION FUNCTIONS
# =============================================================================

class ValidationError(Exception):
    """Custom exception for validation errors."""
    pass


def validate_latitude(lat: float, param_name: str = "latitude") -> None:
    """Validate latitude value (-90 to 90)."""
    if lat is None:
        raise ValidationError(f"{param_name} is required")
    if not isinstance(lat, (int, float)):
        raise ValidationError(f"{param_name} must be a number, got {type(lat).__name__}")
    if not (-90 <= lat <= 90):
        raise ValidationError(f"Invalid {param_name}: {lat}. Must be between -90 and 90.")


def validate_longitude(lon: float, param_name: str = "longitude") -> None:
    """Validate longitude value (-180 to 180)."""
    if lon is None:
        raise ValidationError(f"{param_name} is required")
    if not isinstance(lon, (int, float)):
        raise ValidationError(f"{param_name} must be a number, got {type(lon).__name__}")
    if not (-180 <= lon <= 180):
        raise ValidationError(f"Invalid {param_name}: {lon}. Must be between -180 and 180.")


def validate_location(loc: dict, index: int = 0) -> None:
    """Validate a location dictionary."""
    if "name" not in loc:
        raise ValidationError(f"Location {index} missing 'name' field")
    if "lat" not in loc:
        raise ValidationError(f"Location '{loc.get('name', index)}' missing 'lat' field")
    if "lon" not in loc:
        raise ValidationError(f"Location '{loc.get('name', index)}' missing 'lon' field")

    validate_latitude(loc["lat"], f"Location '{loc['name']}' latitude")
    validate_longitude(loc["lon"], f"Location '{loc['name']}' longitude")


def validate_locations(locations: list) -> None:
    """Validate all locations."""
    if not locations:
        raise ValidationError("At least one location is required")
    for i, loc in enumerate(locations):
        validate_location(loc, i)


def validate_date_format(date_str: str, param_name: str = "date") -> None:
    """Validate date format (YYYY-MM-DD)."""
    if not date_str:
        return
    import re
    if not re.match(r"^\\d{{4}}-\\d{{2}}-\\d{{2}}$", date_str):
        raise ValidationError(
            f"Invalid {param_name} format: '{date_str}'. Expected format: YYYY-MM-DD"
        )
    # Check if it's a valid date
    try:
        datetime.strptime(date_str, "%Y-%m-%d")
    except ValueError as e:
        raise ValidationError(f"Invalid {param_name}: {e}")


def validate_date_range(start_date: str, end_date: str) -> None:
    """Validate that start_date is before end_date."""
    if not start_date or not end_date:
        return
    start = datetime.strptime(start_date, "%Y-%m-%d")
    end = datetime.strptime(end_date, "%Y-%m-%d")
    if start > end:
        raise ValidationError(
            f"start_date ({start_date}) must be before end_date ({end_date})"
        )


def validate_response(data: dict) -> None:
    """Validate API response structure."""
    if not isinstance(data, dict):
        raise ValidationError("API response must be a dictionary")

    required_fields = ["latitude", "longitude"]
    missing = [f for f in required_fields if f not in data]
    if missing:
        raise ValidationError(f"API response missing required fields: {missing}")'''

    def _generate_base_request_function(self) -> str:
        """Generate the base API request function."""
        return '''
# =============================================================================
# API FUNCTIONS
# =============================================================================

def make_api_request(url: str, params: dict, retries: int = API_RETRIES) -> dict:
    """Make API request with retry logic."""
    last_error = None

    for attempt in range(retries):
        try:
            response = requests.get(url, params=params, timeout=API_TIMEOUT)
            response.raise_for_status()
            data = response.json()

            if "error" in data and data.get("error"):
                raise ValueError(f"API Error: {data.get('reason', 'Unknown error')}")

            return data

        except Exception as e:
            last_error = e
            logger.warning(f"Request failed (attempt {attempt + 1}/{retries}): {e}")
            if attempt < retries - 1:
                import time
                time.sleep(1 * (attempt + 1))

    raise last_error


def save_dataframe(df: pd.DataFrame, filename: str):
    """Save DataFrame to file."""
    os.makedirs(OUTPUT_DIR, exist_ok=True)
    filepath = os.path.join(OUTPUT_DIR, filename)

    if OUTPUT_FORMAT == "csv":
        df.to_csv(filepath, index=False)
    else:
        df.to_json(filepath, orient="records", date_format="iso")

    logger.info(f"Saved {len(df)} records to {filepath}")
    return filepath'''

    def _generate_api_function(self, api: str) -> str:
        """Generate function for specific API."""
        config = self.API_CONFIGS[api]
        func_name = f"fetch_{api}_data"
        config_name = api.upper()

        # API-specific URL and parameters
        api_urls = {
            "forecast": "https://api.open-meteo.com/v1/forecast",
            "historical": "https://archive-api.open-meteo.com/v1/archive",
            "air_quality": "https://air-quality-api.open-meteo.com/v1/air-quality",
            "marine": "https://marine-api.open-meteo.com/v1/marine",
            "flood": "https://flood-api.open-meteo.com/v1/flood",
            "climate": "https://climate-api.open-meteo.com/v1/climate",
            "ensemble": "https://ensemble-api.open-meteo.com/v1/ensemble",
        }

        url = api_urls[api]

        # Build function based on API type
        if config["requires_dates"]:
            if api == "historical":
                return f'''
def {func_name}(
    locations: List[Dict] = LOCATIONS,
    start_date: str = None,
    end_date: str = None,
    **kwargs
) -> pd.DataFrame:
    """
    Fetch {config['name']} data for all locations.

    Args:
        locations: List of location dicts with 'name', 'lat', 'lon'
        start_date: Start date (YYYY-MM-DD). Defaults to 30 days ago.
        end_date: End date (YYYY-MM-DD). Defaults to yesterday.

    Returns:
        DataFrame with weather data
    """
    # Validate inputs
    validate_locations(locations)

    if start_date is None:
        start_date = (datetime.now() - timedelta(days=30)).strftime("%Y-%m-%d")
    if end_date is None:
        end_date = (datetime.now() - timedelta(days=1)).strftime("%Y-%m-%d")

    # Validate dates
    validate_date_format(start_date, "start_date")
    validate_date_format(end_date, "end_date")
    validate_date_range(start_date, end_date)

    logger.info(f"Fetching {config['name']} data for {{len(locations)}} locations")
    logger.info(f"Date range: {{start_date}} to {{end_date}}")

    all_data = []

    for loc in locations:
        logger.info(f"  Fetching {{loc['name']}}...")

        params = {{
            "latitude": loc["lat"],
            "longitude": loc["lon"],
            "start_date": start_date,
            "end_date": end_date,
            "timezone": "UTC",
            **{config_name}_PARAMS,
            **kwargs,
        }}

        # Convert lists to comma-separated strings
        for key, value in params.items():
            if isinstance(value, list):
                params[key] = ",".join(str(v) for v in value)

        try:
            data = make_api_request("{url}", params)

            # Validate response
            validate_response(data)

            # Parse hourly or daily data
            if "hourly" in data:
                time_data = data["hourly"].pop("time", [])
                df = pd.DataFrame({{"time": time_data, **data["hourly"]}})
            elif "daily" in data:
                time_data = data["daily"].pop("time", [])
                df = pd.DataFrame({{"time": time_data, **data["daily"]}})
            else:
                logger.warning(f"No data returned for {{loc['name']}}")
                continue

            df["location_name"] = loc["name"]
            df["latitude"] = loc["lat"]
            df["longitude"] = loc["lon"]
            all_data.append(df)

        except ValidationError as e:
            logger.error(f"Validation error for {{loc['name']}}: {{e}}")
        except Exception as e:
            logger.error(f"Failed to fetch {{loc['name']}}: {{e}}")

    if not all_data:
        return pd.DataFrame()

    combined = pd.concat(all_data, ignore_index=True)
    logger.info(f"Total records: {{len(combined)}}")

    return combined'''
            else:  # climate
                return f'''
def {func_name}(
    locations: List[Dict] = LOCATIONS,
    start_date: str = None,
    end_date: str = None,
    **kwargs
) -> pd.DataFrame:
    """
    Fetch {config['name']} data for all locations.

    Args:
        locations: List of location dicts with 'name', 'lat', 'lon'
        start_date: Start date (YYYY-MM-DD). Defaults to 2030-01-01.
        end_date: End date (YYYY-MM-DD). Defaults to 2030-12-31.

    Returns:
        DataFrame with climate data
    """
    # Validate inputs
    validate_locations(locations)

    if start_date is None:
        start_date = "2030-01-01"
    if end_date is None:
        end_date = "2030-12-31"

    # Validate dates
    validate_date_format(start_date, "start_date")
    validate_date_format(end_date, "end_date")
    validate_date_range(start_date, end_date)

    logger.info(f"Fetching {config['name']} data for {{len(locations)}} locations")
    logger.info(f"Date range: {{start_date}} to {{end_date}}")

    all_data = []

    for loc in locations:
        logger.info(f"  Fetching {{loc['name']}}...")

        params = {{
            "latitude": loc["lat"],
            "longitude": loc["lon"],
            "start_date": start_date,
            "end_date": end_date,
            **{config_name}_PARAMS,
            **kwargs,
        }}

        # Convert lists to comma-separated strings
        for key, value in params.items():
            if isinstance(value, list):
                params[key] = ",".join(str(v) for v in value)

        try:
            data = make_api_request("{url}", params)

            # Validate response
            validate_response(data)

            if "daily" in data:
                time_data = data["daily"].pop("time", [])
                df = pd.DataFrame({{"time": time_data, **data["daily"]}})
            else:
                logger.warning(f"No data returned for {{loc['name']}}")
                continue

            df["location_name"] = loc["name"]
            df["latitude"] = loc["lat"]
            df["longitude"] = loc["lon"]
            all_data.append(df)

        except ValidationError as e:
            logger.error(f"Validation error for {{loc['name']}}: {{e}}")
        except Exception as e:
            logger.error(f"Failed to fetch {{loc['name']}}: {{e}}")

    if not all_data:
        return pd.DataFrame()

    combined = pd.concat(all_data, ignore_index=True)
    logger.info(f"Total records: {{len(combined)}}")

    return combined'''
        else:
            # Forecast-type APIs (no date range required)
            return f'''
def {func_name}(
    locations: List[Dict] = LOCATIONS,
    **kwargs
) -> pd.DataFrame:
    """
    Fetch {config['name']} data for all locations.

    Args:
        locations: List of location dicts with 'name', 'lat', 'lon'

    Returns:
        DataFrame with weather data
    """
    # Validate inputs
    validate_locations(locations)

    logger.info(f"Fetching {config['name']} data for {{len(locations)}} locations")

    all_data = []

    for loc in locations:
        logger.info(f"  Fetching {{loc['name']}}...")

        params = {{
            "latitude": loc["lat"],
            "longitude": loc["lon"],
            "timezone": "UTC",
            **{config_name}_PARAMS,
            **kwargs,
        }}

        # Convert lists to comma-separated strings
        for key, value in params.items():
            if isinstance(value, list):
                params[key] = ",".join(str(v) for v in value)

        try:
            data = make_api_request("{url}", params)

            # Validate response
            validate_response(data)

            # Parse hourly or daily data
            if "hourly" in data:
                time_data = data["hourly"].pop("time", [])
                df = pd.DataFrame({{"time": time_data, **data["hourly"]}})
            elif "daily" in data:
                time_data = data["daily"].pop("time", [])
                df = pd.DataFrame({{"time": time_data, **data["daily"]}})
            else:
                logger.warning(f"No data returned for {{loc['name']}}")
                continue

            df["location_name"] = loc["name"]
            df["latitude"] = loc["lat"]
            df["longitude"] = loc["lon"]
            all_data.append(df)

        except ValidationError as e:
            logger.error(f"Validation error for {{loc['name']}}: {{e}}")
        except Exception as e:
            logger.error(f"Failed to fetch {{loc['name']}}: {{e}}")

    if not all_data:
        return pd.DataFrame()

    combined = pd.concat(all_data, ignore_index=True)
    logger.info(f"Total records: {{len(combined)}}")

    return combined'''

    def _generate_main_function(self) -> str:
        """Generate main execution function."""
        fetch_calls = []
        for api in self.apis:
            func_name = f"fetch_{api}_data"
            # Generate f-string for filename in the output code
            fetch_calls.append(f'''
    # Fetch {self.API_CONFIGS[api]['name']}
    logger.info("=" * 60)
    df_{api} = {func_name}()
    if not df_{api}.empty:
        save_dataframe(df_{api}, f"{api}_{{datetime.now().strftime('%Y%m%d_%H%M%S')}}.{self.output_format}")
        results["{api}"] = df_{api}''')

        fetch_code = "\n".join(fetch_calls)

        return f'''
# =============================================================================
# MAIN EXECUTION
# =============================================================================

def run_pipeline(**context) -> Dict[str, pd.DataFrame]:
    """
    Run the complete data fetching pipeline.

    Returns:
        Dictionary of DataFrames for each API
    """
    logger.info("Starting Open-Meteo data pipeline")
    logger.info(f"Output directory: {{OUTPUT_DIR}}")
    logger.info(f"Output format: {{OUTPUT_FORMAT}}")

    results = {{}}{fetch_code}

    logger.info("=" * 60)
    logger.info("Pipeline completed!")

    return results'''

    def _generate_airflow_dag(self) -> str:
        """Generate Airflow DAG definition."""
        task_definitions = []
        task_names = []

        for api in self.apis:
            task_id = f"fetch_{api}"
            func_name = f"fetch_{api}_data"
            task_names.append(task_id)
            task_definitions.append(f'''
        {task_id} = PythonOperator(
            task_id="{task_id}",
            python_callable=lambda: save_dataframe(
                {func_name}(),
                "{api}_{{{{ ds_nodash }}}}.{self.output_format}"
            ),
        )''')

        tasks_code = "\n".join(task_definitions)

        # Create task dependencies (parallel execution)
        if len(task_names) > 1:
            deps = f"\n        [{', '.join(task_names)}]  # Run in parallel"
        else:
            deps = ""

        return f'''
# =============================================================================
# AIRFLOW DAG DEFINITION
# =============================================================================

if AIRFLOW_AVAILABLE:
    default_args = {{
        "owner": "airflow",
        "depends_on_past": False,
        "email_on_failure": False,
        "email_on_retry": False,
        "retries": 2,
        "retry_delay": timedelta(minutes=5),
    }}

    with DAG(
        dag_id="{self.dag_id}",
        default_args=default_args,
        description="Auto-generated Open-Meteo data pipeline",
        schedule_interval="{self.schedule}",
        start_date=datetime(2024, 1, 1),
        catchup=False,
        tags=["weather", "open-meteo", "auto-generated"],
    ) as dag:{tasks_code}{deps}'''

    def _generate_script_entry(self) -> str:
        """Generate script entry point."""
        return '''
# =============================================================================
# SCRIPT ENTRY POINT
# =============================================================================

def parse_args():
    """Parse command line arguments."""
    parser = argparse.ArgumentParser(
        description="Fetch weather data from Open-Meteo API"
    )
    parser.add_argument(
        "--output-dir", "-o",
        type=str,
        default=OUTPUT_DIR,
        help=f"Output directory (default: {OUTPUT_DIR})"
    )
    parser.add_argument(
        "--format", "-f",
        type=str,
        choices=["csv", "json"],
        default=OUTPUT_FORMAT,
        help=f"Output format (default: {OUTPUT_FORMAT})"
    )
    return parser.parse_args()


if __name__ == "__main__":
    args = parse_args()

    # Override config from command line
    OUTPUT_DIR = args.output_dir
    OUTPUT_FORMAT = args.format

    # Run the pipeline
    run_pipeline()'''

    @classmethod
    def get_available_apis(cls) -> List[str]:
        """Get list of available API names."""
        return list(cls.API_CONFIGS.keys())

    @classmethod
    def get_api_description(cls, api: str) -> str:
        """Get description for an API."""
        if api in cls.API_CONFIGS:
            return cls.API_CONFIGS[api]["name"]
        return "Unknown API"


class StandaloneScriptGenerator:
    """
    Generator for simple standalone scripts without Airflow.

    Creates minimal, focused Python scripts for a single API
    that can be run directly or via cron.
    """

    # API endpoint URLs
    API_URLS = {
        "forecast": "https://api.open-meteo.com/v1/forecast",
        "historical": "https://archive-api.open-meteo.com/v1/archive",
        "air_quality": "https://air-quality-api.open-meteo.com/v1/air-quality",
        "marine": "https://marine-api.open-meteo.com/v1/marine",
        "flood": "https://flood-api.open-meteo.com/v1/flood",
        "climate": "https://climate-api.open-meteo.com/v1/climate",
        "ensemble": "https://ensemble-api.open-meteo.com/v1/ensemble",
    }

    # Default parameters for each API
    API_DEFAULTS = {
        "forecast": {
            "hourly": ["temperature_2m", "precipitation", "wind_speed_10m", "weather_code"],
            "forecast_days": 7,
        },
        "historical": {
            "hourly": ["temperature_2m", "precipitation", "wind_speed_10m"],
        },
        "air_quality": {
            "hourly": ["pm10", "pm2_5", "european_aqi", "us_aqi"],
            "forecast_days": 5,
        },
        "marine": {
            "hourly": ["wave_height", "wave_direction", "wave_period", "sea_surface_temperature"],
            "forecast_days": 5,
        },
        "flood": {
            "daily": ["river_discharge", "river_discharge_max", "river_discharge_min"],
            "forecast_days": 92,
        },
        "climate": {
            "models": ["EC_Earth3P_HR"],
            "daily": ["temperature_2m_max", "temperature_2m_min", "precipitation_sum"],
        },
        "ensemble": {
            "models": ["icon_seamless"],
            "hourly": ["temperature_2m", "precipitation"],
            "forecast_days": 7,
        },
    }

    # API descriptions
    API_NAMES = {
        "forecast": "Weather Forecast",
        "historical": "Historical Weather",
        "air_quality": "Air Quality",
        "marine": "Marine Weather",
        "flood": "Flood/River Discharge",
        "climate": "Climate Projections",
        "ensemble": "Ensemble Models",
    }

    def __init__(
        self,
        api: str,
        locations: List[Dict[str, Any]],
        output_format: str = "csv",
        output_dir: str = "./output",
    ):
        """
        Initialize the standalone script generator.

        Args:
            api: API name (forecast, historical, etc.)
            locations: List of location dicts with 'name', 'lat', 'lon'
            output_format: Output format ('csv' or 'json')
            output_dir: Directory to save output files
        """
        self.api = api.lower()
        self.locations = locations
        self.output_format = output_format
        self.output_dir = output_dir

        # Validate API
        if self.api not in self.API_URLS:
            raise ValueError(
                f"Invalid API: {api}. Valid options: {list(self.API_URLS.keys())}"
            )

        # Validate locations
        self._validate_locations(locations)

        # Validate output format
        if output_format not in ["csv", "json"]:
            raise ValueError(f"Invalid output_format: {output_format}. Must be 'csv' or 'json'.")

    def _validate_locations(self, locations: List[Dict[str, Any]]) -> None:
        """Validate location configurations."""
        if not locations:
            raise ValueError("At least one location is required")

        for i, loc in enumerate(locations):
            if "name" not in loc:
                raise ValueError(f"Location {i} missing 'name' field")
            if "lat" not in loc:
                raise ValueError(f"Location '{loc.get('name', i)}' missing 'lat' field")
            if "lon" not in loc:
                raise ValueError(f"Location '{loc.get('name', i)}' missing 'lon' field")

            try:
                CoordinateValidator.validate_latitude(loc["lat"], f"Location '{loc['name']}' latitude")
                CoordinateValidator.validate_longitude(loc["lon"], f"Location '{loc['name']}' longitude")
            except ValidationError as e:
                raise ValueError(str(e))

    def generate(self) -> str:
        """Generate the complete standalone Python script."""
        sections = [
            self._generate_header(),
            self._generate_imports(),
            self._generate_config(),
            self._generate_validation_functions(),
            self._generate_api_functions(),
            self._generate_main(),
        ]
        return "\n\n".join(sections)

    def _generate_header(self) -> str:
        """Generate file header."""
        api_name = self.API_NAMES[self.api]
        return f'''#!/usr/bin/env python3
"""
Open-Meteo {api_name} Data Fetcher

Generated: {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}

This standalone script fetches {api_name.lower()} data from Open-Meteo API
and saves it to {self.output_format.upper()} files.

Usage:
    python <script>.py                      # Run with defaults
    python <script>.py --help               # Show all options
    python <script>.py -o ./data            # Custom output directory
"""'''

    def _generate_imports(self) -> str:
        """Generate import statements."""
        return '''import os
import sys
import argparse
import logging
import re
from datetime import datetime, timedelta
from typing import List, Dict, Optional

import requests
import pandas as pd

# Logging setup
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)'''

    def _generate_config(self) -> str:
        """Generate configuration section."""
        locations_str = json.dumps(self.locations, indent=4)
        defaults_str = json.dumps(self.API_DEFAULTS[self.api], indent=4)

        return f'''
# =============================================================================
# CONFIGURATION
# =============================================================================

# API endpoint
API_URL = "{self.API_URLS[self.api]}"

# Locations to fetch data for
LOCATIONS = {locations_str}

# Default API parameters
API_PARAMS = {defaults_str}

# Output settings
OUTPUT_DIR = "{self.output_dir}"
OUTPUT_FORMAT = "{self.output_format}"

# Request settings
TIMEOUT = 30
RETRIES = 3'''

    def _generate_validation_functions(self) -> str:
        """Generate validation functions."""
        return '''
# =============================================================================
# VALIDATION
# =============================================================================

class ValidationError(Exception):
    """Validation error."""
    pass


def validate_latitude(lat: float, name: str = "latitude") -> None:
    """Validate latitude (-90 to 90)."""
    if lat is None:
        raise ValidationError(f"{name} is required")
    if not isinstance(lat, (int, float)):
        raise ValidationError(f"{name} must be a number")
    if not (-90 <= lat <= 90):
        raise ValidationError(f"Invalid {name}: {lat}. Must be between -90 and 90.")


def validate_longitude(lon: float, name: str = "longitude") -> None:
    """Validate longitude (-180 to 180)."""
    if lon is None:
        raise ValidationError(f"{name} is required")
    if not isinstance(lon, (int, float)):
        raise ValidationError(f"{name} must be a number")
    if not (-180 <= lon <= 180):
        raise ValidationError(f"Invalid {name}: {lon}. Must be between -180 and 180.")


def validate_locations(locations: list) -> None:
    """Validate all locations."""
    if not locations:
        raise ValidationError("At least one location is required")
    for i, loc in enumerate(locations):
        if "name" not in loc:
            raise ValidationError(f"Location {i} missing 'name'")
        if "lat" not in loc or "lon" not in loc:
            raise ValidationError(f"Location '{loc.get('name', i)}' missing coordinates")
        validate_latitude(loc["lat"], f"Location '{loc['name']}' latitude")
        validate_longitude(loc["lon"], f"Location '{loc['name']}' longitude")


def validate_date(date_str: str, name: str = "date") -> None:
    """Validate date format (YYYY-MM-DD)."""
    if not date_str:
        return
    if not re.match(r"^\\d{4}-\\d{2}-\\d{2}$", date_str):
        raise ValidationError(f"Invalid {name}: '{date_str}'. Use YYYY-MM-DD format.")
    try:
        datetime.strptime(date_str, "%Y-%m-%d")
    except ValueError as e:
        raise ValidationError(f"Invalid {name}: {e}")


def validate_date_range(start: str, end: str) -> None:
    """Validate date range."""
    if not start or not end:
        return
    s = datetime.strptime(start, "%Y-%m-%d")
    e = datetime.strptime(end, "%Y-%m-%d")
    if s > e:
        raise ValidationError(f"start_date ({start}) must be before end_date ({end})")'''

    def _generate_api_functions(self) -> str:
        """Generate API request and data processing functions."""
        requires_dates = self.api in ["historical", "climate"]

        base_functions = '''
# =============================================================================
# API FUNCTIONS
# =============================================================================

def make_request(url: str, params: dict) -> dict:
    """Make API request with retries."""
    last_error = None

    for attempt in range(RETRIES):
        try:
            response = requests.get(url, params=params, timeout=TIMEOUT)
            response.raise_for_status()
            data = response.json()

            if data.get("error"):
                raise ValueError(f"API Error: {data.get('reason', 'Unknown')}")

            # Validate response
            if "latitude" not in data or "longitude" not in data:
                raise ValidationError("Invalid API response: missing coordinates")

            return data

        except Exception as e:
            last_error = e
            logger.warning(f"Request failed (attempt {attempt + 1}/{RETRIES}): {e}")
            if attempt < RETRIES - 1:
                import time
                time.sleep(1 * (attempt + 1))

    raise last_error


def save_data(df: pd.DataFrame, filename: str) -> str:
    """Save DataFrame to file."""
    os.makedirs(OUTPUT_DIR, exist_ok=True)
    filepath = os.path.join(OUTPUT_DIR, filename)

    if OUTPUT_FORMAT == "csv":
        df.to_csv(filepath, index=False)
    else:
        df.to_json(filepath, orient="records", date_format="iso")

    logger.info(f"Saved {len(df)} records to {filepath}")
    return filepath'''

        # Generate the fetch function based on API type
        if requires_dates:
            if self.api == "historical":
                fetch_func = self._generate_historical_fetch()
            else:  # climate
                fetch_func = self._generate_climate_fetch()
        else:
            fetch_func = self._generate_forecast_fetch()

        return base_functions + "\n\n" + fetch_func

    def _generate_forecast_fetch(self) -> str:
        """Generate fetch function for forecast-type APIs."""
        api_name = self.API_NAMES[self.api]
        return f'''
def fetch_data(locations: List[Dict] = LOCATIONS) -> pd.DataFrame:
    """
    Fetch {api_name} data for all locations.

    Args:
        locations: List of location dicts with 'name', 'lat', 'lon'

    Returns:
        DataFrame with weather data
    """
    validate_locations(locations)
    logger.info(f"Fetching {api_name} data for {{len(locations)}} locations")

    all_data = []

    for loc in locations:
        logger.info(f"  Fetching {{loc['name']}}...")

        params = {{
            "latitude": loc["lat"],
            "longitude": loc["lon"],
            "timezone": "UTC",
            **API_PARAMS,
        }}

        # Convert lists to comma-separated strings
        for key, value in list(params.items()):
            if isinstance(value, list):
                params[key] = ",".join(str(v) for v in value)

        try:
            data = make_request(API_URL, params)

            # Parse response
            if "hourly" in data:
                time_col = data["hourly"].pop("time", [])
                df = pd.DataFrame({{"time": time_col, **data["hourly"]}})
            elif "daily" in data:
                time_col = data["daily"].pop("time", [])
                df = pd.DataFrame({{"time": time_col, **data["daily"]}})
            else:
                logger.warning(f"No data for {{loc['name']}}")
                continue

            df["location"] = loc["name"]
            df["latitude"] = loc["lat"]
            df["longitude"] = loc["lon"]
            all_data.append(df)

        except Exception as e:
            logger.error(f"Failed for {{loc['name']}}: {{e}}")

    if not all_data:
        return pd.DataFrame()

    result = pd.concat(all_data, ignore_index=True)
    logger.info(f"Total records: {{len(result)}}")
    return result'''

    def _generate_historical_fetch(self) -> str:
        """Generate fetch function for historical API."""
        return '''
def fetch_data(
    locations: List[Dict] = LOCATIONS,
    start_date: str = None,
    end_date: str = None,
) -> pd.DataFrame:
    """
    Fetch Historical Weather data for all locations.

    Args:
        locations: List of location dicts with 'name', 'lat', 'lon'
        start_date: Start date (YYYY-MM-DD). Defaults to 30 days ago.
        end_date: End date (YYYY-MM-DD). Defaults to yesterday.

    Returns:
        DataFrame with weather data
    """
    validate_locations(locations)

    # Default date range
    if start_date is None:
        start_date = (datetime.now() - timedelta(days=30)).strftime("%Y-%m-%d")
    if end_date is None:
        end_date = (datetime.now() - timedelta(days=1)).strftime("%Y-%m-%d")

    validate_date(start_date, "start_date")
    validate_date(end_date, "end_date")
    validate_date_range(start_date, end_date)

    logger.info(f"Fetching Historical data for {len(locations)} locations")
    logger.info(f"Date range: {start_date} to {end_date}")

    all_data = []

    for loc in locations:
        logger.info(f"  Fetching {loc['name']}...")

        params = {
            "latitude": loc["lat"],
            "longitude": loc["lon"],
            "start_date": start_date,
            "end_date": end_date,
            "timezone": "UTC",
            **API_PARAMS,
        }

        # Convert lists to comma-separated strings
        for key, value in list(params.items()):
            if isinstance(value, list):
                params[key] = ",".join(str(v) for v in value)

        try:
            data = make_request(API_URL, params)

            if "hourly" in data:
                time_col = data["hourly"].pop("time", [])
                df = pd.DataFrame({"time": time_col, **data["hourly"]})
            elif "daily" in data:
                time_col = data["daily"].pop("time", [])
                df = pd.DataFrame({"time": time_col, **data["daily"]})
            else:
                logger.warning(f"No data for {loc['name']}")
                continue

            df["location"] = loc["name"]
            df["latitude"] = loc["lat"]
            df["longitude"] = loc["lon"]
            all_data.append(df)

        except Exception as e:
            logger.error(f"Failed for {loc['name']}: {e}")

    if not all_data:
        return pd.DataFrame()

    result = pd.concat(all_data, ignore_index=True)
    logger.info(f"Total records: {len(result)}")
    return result'''

    def _generate_climate_fetch(self) -> str:
        """Generate fetch function for climate API."""
        return '''
def fetch_data(
    locations: List[Dict] = LOCATIONS,
    start_date: str = None,
    end_date: str = None,
) -> pd.DataFrame:
    """
    Fetch Climate Projection data for all locations.

    Args:
        locations: List of location dicts with 'name', 'lat', 'lon'
        start_date: Start date (YYYY-MM-DD). Defaults to 2030-01-01.
        end_date: End date (YYYY-MM-DD). Defaults to 2030-12-31.

    Returns:
        DataFrame with climate data
    """
    validate_locations(locations)

    # Default date range (future projections)
    if start_date is None:
        start_date = "2030-01-01"
    if end_date is None:
        end_date = "2030-12-31"

    validate_date(start_date, "start_date")
    validate_date(end_date, "end_date")
    validate_date_range(start_date, end_date)

    logger.info(f"Fetching Climate data for {len(locations)} locations")
    logger.info(f"Date range: {start_date} to {end_date}")

    all_data = []

    for loc in locations:
        logger.info(f"  Fetching {loc['name']}...")

        params = {
            "latitude": loc["lat"],
            "longitude": loc["lon"],
            "start_date": start_date,
            "end_date": end_date,
            **API_PARAMS,
        }

        # Convert lists to comma-separated strings
        for key, value in list(params.items()):
            if isinstance(value, list):
                params[key] = ",".join(str(v) for v in value)

        try:
            data = make_request(API_URL, params)

            if "daily" in data:
                time_col = data["daily"].pop("time", [])
                df = pd.DataFrame({"time": time_col, **data["daily"]})
            else:
                logger.warning(f"No data for {loc['name']}")
                continue

            df["location"] = loc["name"]
            df["latitude"] = loc["lat"]
            df["longitude"] = loc["lon"]
            all_data.append(df)

        except Exception as e:
            logger.error(f"Failed for {loc['name']}: {e}")

    if not all_data:
        return pd.DataFrame()

    result = pd.concat(all_data, ignore_index=True)
    logger.info(f"Total records: {len(result)}")
    return result'''

    def _generate_main(self) -> str:
        """Generate main execution code."""
        api_name = self.API_NAMES[self.api]
        requires_dates = self.api in ["historical", "climate"]

        if requires_dates:
            return f'''
# =============================================================================
# MAIN
# =============================================================================

def main():
    """Main entry point."""
    global OUTPUT_DIR, OUTPUT_FORMAT

    parser = argparse.ArgumentParser(
        description="Fetch {api_name} data from Open-Meteo API"
    )
    parser.add_argument(
        "-o", "--output-dir",
        type=str,
        default=OUTPUT_DIR,
        help=f"Output directory (default: {{OUTPUT_DIR}})"
    )
    parser.add_argument(
        "-f", "--format",
        type=str,
        choices=["csv", "json"],
        default=OUTPUT_FORMAT,
        help=f"Output format (default: {{OUTPUT_FORMAT}})"
    )
    parser.add_argument(
        "--start-date",
        type=str,
        help="Start date (YYYY-MM-DD)"
    )
    parser.add_argument(
        "--end-date",
        type=str,
        help="End date (YYYY-MM-DD)"
    )

    args = parser.parse_args()

    # Update globals from args
    OUTPUT_DIR = args.output_dir
    OUTPUT_FORMAT = args.format

    # Fetch data
    logger.info("=" * 60)
    logger.info("{api_name} Data Fetcher")
    logger.info("=" * 60)

    df = fetch_data(start_date=args.start_date, end_date=args.end_date)

    if not df.empty:
        filename = f"{self.api}_{{datetime.now().strftime('%Y%m%d_%H%M%S')}}.{{OUTPUT_FORMAT}}"
        save_data(df, filename)
        logger.info("Done!")
    else:
        logger.warning("No data fetched")
        sys.exit(1)


if __name__ == "__main__":
    main()'''
        else:
            return f'''
# =============================================================================
# MAIN
# =============================================================================

def main():
    """Main entry point."""
    global OUTPUT_DIR, OUTPUT_FORMAT

    parser = argparse.ArgumentParser(
        description="Fetch {api_name} data from Open-Meteo API"
    )
    parser.add_argument(
        "-o", "--output-dir",
        type=str,
        default=OUTPUT_DIR,
        help=f"Output directory (default: {{OUTPUT_DIR}})"
    )
    parser.add_argument(
        "-f", "--format",
        type=str,
        choices=["csv", "json"],
        default=OUTPUT_FORMAT,
        help=f"Output format (default: {{OUTPUT_FORMAT}})"
    )

    args = parser.parse_args()

    # Update globals from args
    OUTPUT_DIR = args.output_dir
    OUTPUT_FORMAT = args.format

    # Fetch data
    logger.info("=" * 60)
    logger.info("{api_name} Data Fetcher")
    logger.info("=" * 60)

    df = fetch_data()

    if not df.empty:
        filename = f"{self.api}_{{datetime.now().strftime('%Y%m%d_%H%M%S')}}.{{OUTPUT_FORMAT}}"
        save_data(df, filename)
        logger.info("Done!")
    else:
        logger.warning("No data fetched")
        sys.exit(1)


if __name__ == "__main__":
    main()'''
