"""
Data models for Open-Meteo API responses.

Supports both JSON and FlatBuffers response formats with zero-copy
DataFrame conversion when using FlatBuffers with numpy arrays.
"""

from dataclasses import dataclass, field
from typing import Dict, List, Optional, Any, Union
from datetime import datetime
import pandas as pd

# Optional numpy import for zero-copy support
try:
    import numpy as np
    NUMPY_AVAILABLE = True
except ImportError:
    np = None
    NUMPY_AVAILABLE = False


@dataclass
class Location:
    """Geographic location with coordinates."""
    latitude: float
    longitude: float
    elevation: Optional[float] = None
    timezone: Optional[str] = None
    timezone_abbreviation: Optional[str] = None
    utc_offset_seconds: Optional[int] = None

    @classmethod
    def from_response(cls, data: Dict[str, Any]) -> "Location":
        """Create Location from API response."""
        return cls(
            latitude=data.get("latitude"),
            longitude=data.get("longitude"),
            elevation=data.get("elevation"),
            timezone=data.get("timezone"),
            timezone_abbreviation=data.get("timezone_abbreviation"),
            utc_offset_seconds=data.get("utc_offset_seconds"),
        )


@dataclass
class WeatherUnits:
    """Units for weather variables."""
    time: str = "iso8601"
    data: Dict[str, str] = field(default_factory=dict)

    @classmethod
    def from_response(cls, data: Dict[str, Any], prefix: str = "") -> "WeatherUnits":
        """Create WeatherUnits from API response."""
        units_key = f"{prefix}_units" if prefix else "hourly_units"
        units_data = data.get(units_key, {})
        return cls(
            time=units_data.get("time", "iso8601"),
            data=units_data
        )


@dataclass
class TimeSeriesData:
    """
    Container for time series data (hourly, daily, etc.).

    Supports both JSON (list) and FlatBuffers (numpy array) data formats.
    When data contains numpy arrays, zero-copy DataFrame conversion is used.
    """
    time: Union[List[str], Any]  # List[str] for JSON, numpy array for FlatBuffers
    data: Dict[str, Union[List[Any], Any]]  # Values can be lists or numpy arrays
    units: Dict[str, str]

    def to_dataframe(self, zero_copy: bool = True) -> pd.DataFrame:
        """
        Convert to pandas DataFrame.

        Args:
            zero_copy: If True and data contains numpy arrays, use zero-copy
                       conversion for better performance. Default is True.

        Returns:
            DataFrame with time series data.
        """
        df_data = {}

        # Handle time column - check if it's a numpy array (FlatBuffers) or list (JSON)
        if NUMPY_AVAILABLE and hasattr(self.time, 'dtype'):
            # FlatBuffers: time is unix timestamps as int64
            if zero_copy:
                # Zero-copy: create datetime index directly from numpy array
                df_data["time"] = pd.to_datetime(self.time, unit='s', utc=True)
            else:
                df_data["time"] = pd.to_datetime(self.time.tolist(), unit='s', utc=True)
        else:
            # JSON: time is ISO8601 strings
            df_data["time"] = pd.to_datetime(self.time)

        # Handle data columns
        for key, values in self.data.items():
            if NUMPY_AVAILABLE and hasattr(values, 'dtype') and zero_copy:
                # Zero-copy: numpy array passed directly to pandas
                df_data[key] = values
            else:
                # Regular list or copy requested
                df_data[key] = values

        return pd.DataFrame(df_data)

    def to_numpy(self) -> Dict[str, Any]:
        """
        Get data as numpy arrays (zero-copy when using FlatBuffers).

        Returns:
            Dictionary with variable names as keys and numpy arrays as values.
            Returns original arrays for FlatBuffers data (zero-copy).

        Raises:
            ImportError: If numpy is not available.
        """
        if not NUMPY_AVAILABLE:
            raise ImportError("numpy is required for to_numpy(). Install with: pip install numpy")

        result = {}

        # Handle time
        if hasattr(self.time, 'dtype'):
            result["time"] = self.time  # Already numpy array
        else:
            result["time"] = np.array(self.time)

        # Handle data columns
        for key, values in self.data.items():
            if hasattr(values, 'dtype'):
                result[key] = values  # Already numpy array (zero-copy)
            else:
                result[key] = np.array(values)

        return result

    def __getattr__(self, name: str) -> Union[List[Any], Any]:
        """Access data columns as attributes."""
        if name in self.data:
            return self.data[name]
        raise AttributeError(f"No variable named '{name}'")


@dataclass
class CurrentData:
    """Container for current weather data."""
    time: str
    interval: int
    data: Dict[str, Any]
    units: Dict[str, str]

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        result = {"time": self.time, "interval": self.interval}
        result.update(self.data)
        return result

    def __getattr__(self, name: str) -> Any:
        """Access data values as attributes."""
        if name in self.data:
            return self.data[name]
        raise AttributeError(f"No variable named '{name}'")


@dataclass
class WeatherResponse:
    """
    Generic response container for weather API responses.

    Supports hourly, daily, current, and minutely_15 data.
    """
    location: Location
    generation_time_ms: float
    hourly: Optional[TimeSeriesData] = None
    daily: Optional[TimeSeriesData] = None
    current: Optional[CurrentData] = None
    minutely_15: Optional[TimeSeriesData] = None
    raw_response: Dict[str, Any] = field(default_factory=dict)

    @classmethod
    def from_response(cls, data: Dict[str, Any]) -> "WeatherResponse":
        """Create WeatherResponse from API response."""
        location = Location.from_response(data)

        hourly = None
        if "hourly" in data:
            hourly_data = data["hourly"]
            time_data = hourly_data.pop("time", [])
            hourly = TimeSeriesData(
                time=time_data,
                data=hourly_data,
                units=data.get("hourly_units", {})
            )

        daily = None
        if "daily" in data:
            daily_data = data["daily"]
            time_data = daily_data.pop("time", [])
            daily = TimeSeriesData(
                time=time_data,
                data=daily_data,
                units=data.get("daily_units", {})
            )

        current = None
        if "current" in data:
            current_data = data["current"].copy()
            time_val = current_data.pop("time", "")
            interval = current_data.pop("interval", 0)
            current = CurrentData(
                time=time_val,
                interval=interval,
                data=current_data,
                units=data.get("current_units", {})
            )

        minutely_15 = None
        if "minutely_15" in data:
            minutely_data = data["minutely_15"]
            time_data = minutely_data.pop("time", [])
            minutely_15 = TimeSeriesData(
                time=time_data,
                data=minutely_data,
                units=data.get("minutely_15_units", {})
            )

        return cls(
            location=location,
            generation_time_ms=data.get("generationtime_ms", 0),
            hourly=hourly,
            daily=daily,
            current=current,
            minutely_15=minutely_15,
            raw_response=data,
        )

    def to_dataframe(self, data_type: str = "hourly") -> pd.DataFrame:
        """
        Convert response data to pandas DataFrame.

        Args:
            data_type: Type of data to convert ('hourly', 'daily', 'minutely_15')

        Returns:
            DataFrame with time series data
        """
        data_map = {
            "hourly": self.hourly,
            "daily": self.daily,
            "minutely_15": self.minutely_15,
        }

        data = data_map.get(data_type)
        if data is None:
            raise ValueError(f"No {data_type} data available in response")

        df = data.to_dataframe()
        df["latitude"] = self.location.latitude
        df["longitude"] = self.location.longitude
        return df


@dataclass
class GeocodingResult:
    """Result from geocoding search."""
    id: int
    name: str
    latitude: float
    longitude: float
    elevation: Optional[float] = None
    timezone: Optional[str] = None
    feature_code: Optional[str] = None
    country_code: Optional[str] = None
    country: Optional[str] = None
    admin1: Optional[str] = None
    admin2: Optional[str] = None
    admin3: Optional[str] = None
    admin4: Optional[str] = None
    population: Optional[int] = None
    postcodes: Optional[List[str]] = None

    @classmethod
    def from_response(cls, data: Dict[str, Any]) -> "GeocodingResult":
        """Create GeocodingResult from API response."""
        return cls(
            id=data.get("id"),
            name=data.get("name"),
            latitude=data.get("latitude"),
            longitude=data.get("longitude"),
            elevation=data.get("elevation"),
            timezone=data.get("timezone"),
            feature_code=data.get("feature_code"),
            country_code=data.get("country_code"),
            country=data.get("country"),
            admin1=data.get("admin1"),
            admin2=data.get("admin2"),
            admin3=data.get("admin3"),
            admin4=data.get("admin4"),
            population=data.get("population"),
            postcodes=data.get("postcodes"),
        )


@dataclass
class ElevationResponse:
    """Response from elevation API."""
    elevations: List[float]

    @classmethod
    def from_response(cls, data: Dict[str, Any]) -> "ElevationResponse":
        """Create ElevationResponse from API response."""
        return cls(elevations=data.get("elevation", []))


@dataclass
class FloodResponse:
    """Response from flood API."""
    location: Location
    generation_time_ms: float
    daily: Optional[TimeSeriesData] = None
    raw_response: Dict[str, Any] = field(default_factory=dict)

    @classmethod
    def from_response(cls, data: Dict[str, Any]) -> "FloodResponse":
        """Create FloodResponse from API response."""
        location = Location.from_response(data)

        daily = None
        if "daily" in data:
            daily_data = data["daily"].copy()
            time_data = daily_data.pop("time", [])
            daily = TimeSeriesData(
                time=time_data,
                data=daily_data,
                units=data.get("daily_units", {})
            )

        return cls(
            location=location,
            generation_time_ms=data.get("generationtime_ms", 0),
            daily=daily,
            raw_response=data,
        )

    def to_dataframe(self) -> pd.DataFrame:
        """Convert to pandas DataFrame."""
        if self.daily is None:
            raise ValueError("No daily data available")
        df = self.daily.to_dataframe()
        df["latitude"] = self.location.latitude
        df["longitude"] = self.location.longitude
        return df
