"""Tests for base client functionality including retry logic, error handling, and request methods."""

import pytest
import responses
import requests
from unittest.mock import patch, MagicMock
import time

from openmeteo.base import OpenMeteoBaseClient, APIConfig, is_flatbuffers_available


class ConcreteTestClient(OpenMeteoBaseClient):
    """Concrete implementation for testing abstract base class."""
    BASE_URL = "https://test-api.open-meteo.com/v1/test"


class TestBaseClientRetryLogic:
    """Tests for retry logic in base client."""

    @responses.activate
    def test_successful_request_no_retry(self):
        """Test that successful requests don't trigger retries."""
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            json={"latitude": 52.52, "longitude": 13.41, "data": "test"},
            status=200,
        )

        client = ConcreteTestClient()
        result = client._make_request(client.BASE_URL, {"latitude": 52.52})

        assert result["latitude"] == 52.52
        assert len(responses.calls) == 1

    @responses.activate
    def test_retry_on_server_error(self):
        """Test retry behavior on 5xx server errors."""
        # First two requests fail with 503, third succeeds
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            json={"error": "Service unavailable"},
            status=503,
        )
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            json={"error": "Service unavailable"},
            status=503,
        )
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            json={"latitude": 52.52, "longitude": 13.41},
            status=200,
        )

        config = APIConfig(retry_attempts=3, retry_delay=0.01)
        client = ConcreteTestClient(config)
        result = client._make_request(client.BASE_URL, {})

        assert result["latitude"] == 52.52
        assert len(responses.calls) == 3

    @responses.activate
    def test_max_retries_exceeded(self):
        """Test that exception is raised after max retries exceeded."""
        for _ in range(3):
            responses.add(
                responses.GET,
                "https://test-api.open-meteo.com/v1/test",
                json={"error": "Service unavailable"},
                status=503,
            )

        config = APIConfig(retry_attempts=3, retry_delay=0.01)
        client = ConcreteTestClient(config)

        with pytest.raises(requests.RequestException):
            client._make_request(client.BASE_URL, {})

        assert len(responses.calls) == 3

    @responses.activate
    def test_api_error_response(self):
        """Test handling of API error responses."""
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            json={"error": True, "reason": "Invalid coordinates"},
            status=200,
        )

        client = ConcreteTestClient()

        with pytest.raises(ValueError, match="API Error: Invalid coordinates"):
            client._make_request(client.BASE_URL, {})

    @responses.activate
    def test_api_error_unknown_reason(self):
        """Test API error with missing reason field."""
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            json={"error": True},
            status=200,
        )

        client = ConcreteTestClient()

        with pytest.raises(ValueError, match="API Error: Unknown API error"):
            client._make_request(client.BASE_URL, {})


class TestBaseClientPostRequest:
    """Tests for POST request functionality."""

    @responses.activate
    def test_post_request(self):
        """Test POST request method."""
        responses.add(
            responses.POST,
            "https://test-api.open-meteo.com/v1/test",
            json={"latitude": 52.52, "longitude": 13.41},
            status=200,
        )

        client = ConcreteTestClient()
        result = client._make_request(client.BASE_URL, {"latitude": 52.52}, method="POST")

        assert result["latitude"] == 52.52
        assert responses.calls[0].request.method == "POST"

    @responses.activate
    def test_post_request_retry(self):
        """Test POST request with retry."""
        responses.add(
            responses.POST,
            "https://test-api.open-meteo.com/v1/test",
            json={"error": "Server error"},
            status=500,
        )
        responses.add(
            responses.POST,
            "https://test-api.open-meteo.com/v1/test",
            json={"latitude": 52.52},
            status=200,
        )

        config = APIConfig(retry_attempts=2, retry_delay=0.01)
        client = ConcreteTestClient(config)
        result = client._make_request(client.BASE_URL, {}, method="POST")

        assert result["latitude"] == 52.52
        assert len(responses.calls) == 2


class TestBaseClientRawRequest:
    """Tests for raw bytes request functionality."""

    @responses.activate
    def test_raw_request_success(self):
        """Test raw bytes request."""
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            body=b"\x00\x01\x02\x03",
            status=200,
        )

        client = ConcreteTestClient()
        result = client._make_request_raw(client.BASE_URL, {})

        assert result == b"\x00\x01\x02\x03"

    @responses.activate
    def test_raw_request_post(self):
        """Test raw POST request."""
        responses.add(
            responses.POST,
            "https://test-api.open-meteo.com/v1/test",
            body=b"binary data",
            status=200,
        )

        client = ConcreteTestClient()
        result = client._make_request_raw(client.BASE_URL, {}, method="POST")

        assert result == b"binary data"

    @responses.activate
    def test_raw_request_retry(self):
        """Test raw request with retry."""
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            body=b"",
            status=502,
        )
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            body=b"success",
            status=200,
        )

        config = APIConfig(retry_attempts=2, retry_delay=0.01)
        client = ConcreteTestClient(config)
        result = client._make_request_raw(client.BASE_URL, {})

        assert result == b"success"
        assert len(responses.calls) == 2

    @responses.activate
    def test_raw_request_max_retries(self):
        """Test raw request max retries exceeded."""
        for _ in range(3):
            responses.add(
                responses.GET,
                "https://test-api.open-meteo.com/v1/test",
                body=b"",
                status=500,
            )

        config = APIConfig(retry_attempts=3, retry_delay=0.01)
        client = ConcreteTestClient(config)

        with pytest.raises(requests.RequestException):
            client._make_request_raw(client.BASE_URL, {})


class TestBaseClientBuildParams:
    """Tests for parameter building functionality."""

    def test_build_params_basic(self):
        """Test basic parameter building."""
        client = ConcreteTestClient()
        params = client._build_params(latitude=52.52, longitude=13.41)

        assert params["latitude"] == 52.52
        assert params["longitude"] == 13.41

    def test_build_params_filters_none(self):
        """Test that None values are filtered out."""
        client = ConcreteTestClient()
        params = client._build_params(latitude=52.52, longitude=None, hourly=None)

        assert "latitude" in params
        assert "longitude" not in params
        assert "hourly" not in params

    def test_build_params_list_to_csv(self):
        """Test that lists are converted to comma-separated strings."""
        client = ConcreteTestClient()
        params = client._build_params(hourly=["temperature_2m", "precipitation"])

        assert params["hourly"] == "temperature_2m,precipitation"

    def test_build_params_tuple_to_csv(self):
        """Test that tuples are converted to comma-separated strings."""
        client = ConcreteTestClient()
        params = client._build_params(hourly=("temperature_2m", "wind_speed"))

        assert params["hourly"] == "temperature_2m,wind_speed"

    def test_build_params_boolean(self):
        """Test that booleans are converted to lowercase strings."""
        client = ConcreteTestClient()
        params = client._build_params(past_days=True, forecast_days=False)

        assert params["past_days"] == "true"
        assert params["forecast_days"] == "false"

    def test_build_params_with_api_key(self):
        """Test that API key is added to params."""
        config = APIConfig(api_key="test-api-key-123")
        client = ConcreteTestClient(config)
        params = client._build_params(latitude=52.52)

        assert params["apikey"] == "test-api-key-123"

    def test_build_params_flatbuffers_format(self):
        """Test that FlatBuffers format is added to params."""
        config = APIConfig(format="flatbuffers")
        with patch.object(ConcreteTestClient, '_init_flatbuffers_parser'):
            client = ConcreteTestClient(config)
            params = client._build_params(latitude=52.52)

        assert params["format"] == "flatbuffers"


class TestBaseClientValidation:
    """Tests for validation methods."""

    def test_validate_coordinates_valid(self):
        """Test valid coordinates."""
        client = ConcreteTestClient()
        # Should not raise
        client._validate_coordinates(52.52, 13.41)
        client._validate_coordinates(-90, -180)
        client._validate_coordinates(90, 180)
        client._validate_coordinates(0, 0)

    def test_validate_coordinates_invalid_latitude(self):
        """Test invalid latitude."""
        client = ConcreteTestClient()
        with pytest.raises(ValueError, match="Invalid latitude"):
            client._validate_coordinates(91, 0)
        with pytest.raises(ValueError, match="Invalid latitude"):
            client._validate_coordinates(-91, 0)

    def test_validate_coordinates_invalid_longitude(self):
        """Test invalid longitude."""
        client = ConcreteTestClient()
        with pytest.raises(ValueError, match="Invalid longitude"):
            client._validate_coordinates(0, 181)
        with pytest.raises(ValueError, match="Invalid longitude"):
            client._validate_coordinates(0, -181)

    def test_validate_coordinates_list(self):
        """Test validating lists of coordinates."""
        client = ConcreteTestClient()
        # Should not raise
        client._validate_coordinates([52.52, 48.85], [13.41, 2.35])

    def test_validate_date_valid(self):
        """Test valid date."""
        client = ConcreteTestClient()
        # Should not raise
        client._validate_date("2024-01-15", "test_date")
        client._validate_date("2023-12-31", "start_date")

    def test_validate_date_invalid_format(self):
        """Test invalid date format."""
        client = ConcreteTestClient()
        with pytest.raises(ValueError):
            client._validate_date("01-15-2024", "test_date")
        with pytest.raises(ValueError):
            client._validate_date("2024/01/15", "test_date")

    def test_validate_date_range_valid(self):
        """Test valid date range."""
        client = ConcreteTestClient()
        # Should not raise
        client._validate_date_range("2024-01-01", "2024-01-31")

    def test_validate_date_range_invalid_order(self):
        """Test date range where end is before start."""
        client = ConcreteTestClient()
        with pytest.raises(ValueError):
            client._validate_date_range("2024-01-31", "2024-01-01")

    def test_validate_response_valid(self):
        """Test valid response validation."""
        client = ConcreteTestClient()
        # Should not raise
        client._validate_response({"latitude": 52.52, "longitude": 13.41})

    def test_validate_response_missing_field(self):
        """Test response validation with missing field."""
        client = ConcreteTestClient()
        with pytest.raises(ValueError):
            client._validate_response({"latitude": 52.52})


class TestBaseClientSanitization:
    """Tests for input sanitization methods."""

    def test_sanitize_timezone(self):
        """Test timezone sanitization."""
        client = ConcreteTestClient()
        result = client._sanitize_timezone("Europe/Berlin")
        assert result == "Europe/Berlin"

    def test_sanitize_unit(self):
        """Test unit sanitization."""
        client = ConcreteTestClient()
        result = client._sanitize_unit("celsius", ["celsius", "fahrenheit"], "celsius")
        assert result == "celsius"


class TestBaseClientContextManager:
    """Tests for context manager functionality."""

    def test_context_manager_enter_exit(self):
        """Test context manager enter and exit."""
        with ConcreteTestClient() as client:
            assert client is not None

    def test_close_session(self):
        """Test that close properly closes the session."""
        client = ConcreteTestClient()
        client.close()
        # Session should be closed (this is somewhat implementation-specific)


class TestIsFlatBuffersAvailable:
    """Tests for is_flatbuffers_available function."""

    def test_returns_boolean(self):
        """Test that function returns a boolean."""
        result = is_flatbuffers_available()
        assert isinstance(result, bool)


class TestFlatBuffersIntegration:
    """Tests for FlatBuffers integration in base client."""

    def test_init_flatbuffers_unavailable(self):
        """Test error when FlatBuffers not available."""
        config = APIConfig(format="flatbuffers")

        with patch('openmeteo.base.OpenMeteoBaseClient._init_flatbuffers_parser') as mock_init:
            mock_init.side_effect = ImportError("Test error")
            with pytest.raises(ImportError):
                client = ConcreteTestClient(config)
                client._init_flatbuffers_parser()

    @responses.activate
    def test_flatbuffers_response_empty(self):
        """Test handling of empty FlatBuffers response."""
        responses.add(
            responses.GET,
            "https://test-api.open-meteo.com/v1/test",
            body=b"",
            status=200,
        )

        config = APIConfig(format="flatbuffers")

        with patch.object(ConcreteTestClient, '_init_flatbuffers_parser'):
            client = ConcreteTestClient(config)
            client._flatbuffers_parser = MagicMock()
            client._flatbuffers_parser.parse.return_value = []

            with pytest.raises(ValueError, match="Empty FlatBuffers response"):
                client._make_request(client.BASE_URL, {})
