"""Tests for CLI interface."""

import pytest
import sys
from unittest.mock import patch, MagicMock
import pandas as pd

from openmeteo.cli import (
    create_parser,
    parse_variables,
    main,
)


class TestCLIParser:
    """Tests for CLI argument parser."""

    def test_create_parser(self):
        """Test parser creation."""
        parser = create_parser()
        assert parser is not None

    def test_forecast_command(self):
        """Test forecast command parsing."""
        parser = create_parser()
        args = parser.parse_args([
            "forecast",
            "--lat", "52.52",
            "--lon", "13.41",
            "--hourly", "temperature_2m,precipitation"
        ])

        assert args.command == "forecast"
        assert args.lat == 52.52
        assert args.lon == 13.41
        assert args.hourly == "temperature_2m,precipitation"

    def test_historical_command(self):
        """Test historical command parsing."""
        parser = create_parser()
        args = parser.parse_args([
            "historical",
            "--lat", "52.52",
            "--lon", "13.41",
            "--start", "2023-01-01",
            "--end", "2023-01-31",
            "--hourly", "temperature_2m"
        ])

        assert args.command == "historical"
        assert args.start == "2023-01-01"
        assert args.end == "2023-01-31"

    def test_geocoding_command(self):
        """Test geocoding command parsing."""
        parser = create_parser()
        args = parser.parse_args([
            "geocoding",
            "--search", "Berlin",
            "--count", "5"
        ])

        assert args.command == "geocoding"
        assert args.search == "Berlin"
        assert args.count == 5

    def test_elevation_command(self):
        """Test elevation command parsing."""
        parser = create_parser()
        args = parser.parse_args([
            "elevation",
            "--lat", "52.52",
            "--lon", "13.41"
        ])

        assert args.command == "elevation"
        assert args.lat == 52.52
        assert args.lon == 13.41

    def test_output_options(self):
        """Test output options parsing."""
        parser = create_parser()
        args = parser.parse_args([
            "forecast",
            "--lat", "52.52",
            "--lon", "13.41",
            "-o", "output.csv",
            "--format", "csv"
        ])

        assert args.output == "output.csv"
        assert args.format == "csv"


class TestParseVariables:
    """Tests for variable parsing."""

    def test_parse_single_variable(self):
        """Test parsing single variable."""
        result = parse_variables("temperature_2m")
        assert result == ["temperature_2m"]

    def test_parse_multiple_variables(self):
        """Test parsing multiple variables."""
        result = parse_variables("temperature_2m,precipitation,wind_speed_10m")
        assert result == ["temperature_2m", "precipitation", "wind_speed_10m"]

    def test_parse_with_spaces(self):
        """Test parsing with spaces."""
        result = parse_variables("temperature_2m, precipitation, wind_speed_10m")
        assert result == ["temperature_2m", "precipitation", "wind_speed_10m"]

    def test_parse_none(self):
        """Test parsing None."""
        result = parse_variables(None)
        assert result is None


class TestCLICommands:
    """Integration tests for CLI commands."""

    @patch('openmeteo.cli.OpenMeteo')
    def test_main_no_command(self, mock_client):
        """Test main with no command shows help."""
        with pytest.raises(SystemExit) as exc_info:
            with patch.object(sys, 'argv', ['openmeteo']):
                main()
        assert exc_info.value.code == 1

    @patch('openmeteo.cli.OpenMeteo')
    def test_forecast_with_location_name(self, mock_client_class):
        """Test forecast with location name resolution."""
        mock_client = MagicMock()
        mock_client_class.return_value = mock_client
        mock_client.geocoding.get_coordinates.return_value = (52.52, 13.41)

        mock_response = MagicMock()
        mock_response.current = None
        mock_response.hourly = MagicMock()
        mock_response.hourly.time = ["2024-01-01T00:00"]
        mock_response.hourly.data = {"temperature_2m": [5.0]}
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01T00:00"],
            "temperature_2m": [5.0],
        })
        mock_client.forecast.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'open-meteo', 'forecast',
            '--location', 'Berlin',
            '--hourly', 'temperature_2m'
        ]):
            with patch('builtins.print'):
                main()

        mock_client.geocoding.get_coordinates.assert_called_once_with("Berlin")


class TestCLICommandsWithMockedHTTP:
    """Tests for CLI commands with mocked HTTP responses."""

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_air_quality(self, mock_client_class, capsys):
        """Test air-quality command."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.current = None
        mock_response.hourly = MagicMock()
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01T00:00"],
            "pm10": [15.0],
        })
        mock_client.air_quality.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'air-quality',
            '--lat', '52.52', '--lon', '13.41',
            '--hourly', 'pm10'
        ]):
            main()

        mock_client.air_quality.get.assert_called_once()

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_marine(self, mock_client_class, capsys):
        """Test marine command."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.current = None
        mock_response.hourly = MagicMock()
        mock_response.daily = None
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01T00:00"],
            "wave_height": [1.5],
        })
        mock_client.marine.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'marine',
            '--lat', '54.32', '--lon', '10.13',
            '--hourly', 'wave_height'
        ]):
            main()

        mock_client.marine.get.assert_called_once()

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_flood(self, mock_client_class, capsys):
        """Test flood command."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.daily = MagicMock()
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01"],
            "river_discharge": [100.5],
        })
        mock_client.flood.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'flood',
            '--lat', '52.52', '--lon', '13.41'
        ]):
            main()

        mock_client.flood.get.assert_called_once()

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_climate(self, mock_client_class, capsys):
        """Test climate command."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.daily = MagicMock()
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2050-01-01"],
            "temperature_2m_max": [15.0],
        })
        mock_client.climate.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'climate',
            '--lat', '52.52', '--lon', '13.41',
            '--start', '2050-01-01', '--end', '2050-12-31'
        ]):
            main()

        mock_client.climate.get.assert_called_once()

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_ensemble(self, mock_client_class, capsys):
        """Test ensemble command."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.hourly = MagicMock()
        mock_response.daily = None
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01T00:00"],
            "temperature_2m": [5.0],
        })
        mock_client.ensemble.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'ensemble',
            '--lat', '52.52', '--lon', '13.41',
            '--hourly', 'temperature_2m'
        ]):
            main()

        mock_client.ensemble.get.assert_called_once()

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_geocoding(self, mock_client_class, capsys):
        """Test geocoding command."""
        mock_client = mock_client_class.return_value

        mock_result = MagicMock()
        mock_result.id = 2950159
        mock_result.name = "Berlin"
        mock_result.latitude = 52.52437
        mock_result.longitude = 13.41053
        mock_result.elevation = 34.0
        mock_result.country = "Germany"
        mock_result.country_code = "DE"
        mock_result.timezone = "Europe/Berlin"
        mock_result.admin1 = "Berlin"
        mock_result.population = 3426354
        mock_client.geocoding.search.return_value = [mock_result]

        with patch.object(sys, 'argv', [
            'openmeteo', 'geocoding',
            '--search', 'Berlin'
        ]):
            main()

        mock_client.geocoding.search.assert_called_once()
        captured = capsys.readouterr()
        assert "Berlin" in captured.out

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_geocoding_no_results(self, mock_client_class, capsys):
        """Test geocoding command with no results."""
        mock_client = mock_client_class.return_value
        mock_client.geocoding.search.return_value = []

        with patch.object(sys, 'argv', [
            'openmeteo', 'geocoding',
            '--search', 'NonexistentPlace12345'
        ]):
            main()

        captured = capsys.readouterr()
        assert "No results found" in captured.out

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_elevation(self, mock_client_class, capsys):
        """Test elevation command."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.elevations = [34.0]
        mock_client.elevation.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'elevation',
            '--lat', '52.52', '--lon', '13.41'
        ]):
            main()

        mock_client.elevation.get.assert_called_once()

    @patch('openmeteo.cli.OpenMeteo')
    def test_cmd_historical(self, mock_client_class, capsys):
        """Test historical command."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.hourly = MagicMock()
        mock_response.daily = None
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2023-01-01T00:00"],
            "temperature_2m": [2.0],
        })
        mock_client.historical.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'historical',
            '--lat', '52.52', '--lon', '13.41',
            '--start', '2023-01-01', '--end', '2023-01-31',
            '--hourly', 'temperature_2m'
        ]):
            main()

        mock_client.historical.get.assert_called_once()


class TestCLIGeneratorCommands:
    """Tests for CLI generator commands."""

    def test_cmd_list_apis(self, capsys):
        """Test list-apis command."""
        with patch.object(sys, 'argv', ['openmeteo', 'list-apis']):
            main()

        captured = capsys.readouterr()
        assert "Available APIs" in captured.out
        assert "forecast" in captured.out

    def test_cmd_generate_dag(self, tmp_path, capsys):
        """Test generate-dag command."""
        output_file = tmp_path / "test_dag.py"

        with patch.object(sys, 'argv', [
            'openmeteo', 'generate-dag',
            '--apis', 'forecast',
            '--locations', 'Berlin:52.52:13.41',
            '-o', str(output_file)
        ]):
            main()

        captured = capsys.readouterr()
        assert "Generated DAG script" in captured.out
        assert output_file.exists()
        content = output_file.read_text()
        assert "def " in content

    def test_cmd_generate_dag_multiple_apis(self, tmp_path, capsys):
        """Test generate-dag with multiple APIs."""
        output_file = tmp_path / "test_dag.py"

        with patch.object(sys, 'argv', [
            'openmeteo', 'generate-dag',
            '--apis', 'forecast,air_quality',
            '--locations', 'Berlin:52.52:13.41,Paris:48.85:2.35',
            '-o', str(output_file),
            '--schedule', '@hourly'
        ]):
            main()

        captured = capsys.readouterr()
        assert "Generated DAG script" in captured.out

    def test_cmd_generate_dag_invalid_location(self, capsys):
        """Test generate-dag with invalid location format."""
        with patch.object(sys, 'argv', [
            'openmeteo', 'generate-dag',
            '--apis', 'forecast',
            '--locations', 'Berlin:52.52',  # Missing longitude
            '-o', 'test.py'
        ]):
            with pytest.raises(SystemExit) as exc_info:
                main()
            assert exc_info.value.code == 1

        captured = capsys.readouterr()
        assert "Invalid location format" in captured.err

    def test_cmd_generate_dag_invalid_coordinates(self, capsys):
        """Test generate-dag with invalid coordinates."""
        with patch.object(sys, 'argv', [
            'openmeteo', 'generate-dag',
            '--apis', 'forecast',
            '--locations', 'Berlin:abc:13.41',  # Invalid latitude
            '-o', 'test.py'
        ]):
            with pytest.raises(SystemExit) as exc_info:
                main()
            assert exc_info.value.code == 1

        captured = capsys.readouterr()
        assert "Invalid coordinates" in captured.err

    def test_cmd_generate_script(self, tmp_path, capsys):
        """Test generate-script command."""
        output_file = tmp_path / "test_script.py"

        with patch.object(sys, 'argv', [
            'openmeteo', 'generate-script',
            '--api', 'forecast',
            '--locations', 'Berlin:52.52:13.41',
            '-o', str(output_file)
        ]):
            main()

        captured = capsys.readouterr()
        assert "Generated standalone script" in captured.out
        assert output_file.exists()
        content = output_file.read_text()
        assert "def main" in content

    def test_cmd_generate_script_historical(self, tmp_path, capsys):
        """Test generate-script for historical API."""
        output_file = tmp_path / "test_historical.py"

        with patch.object(sys, 'argv', [
            'openmeteo', 'generate-script',
            '--api', 'historical',
            '--locations', 'Berlin:52.52:13.41',
            '-o', str(output_file),
            '--format', 'json'
        ]):
            main()

        captured = capsys.readouterr()
        assert "Generated standalone script" in captured.out
        assert "--start-date" in captured.out

    def test_cmd_generate_script_invalid_location(self, capsys):
        """Test generate-script with invalid location."""
        with patch.object(sys, 'argv', [
            'openmeteo', 'generate-script',
            '--api', 'forecast',
            '--locations', 'Berlin',  # Missing coordinates
            '-o', 'test.py'
        ]):
            with pytest.raises(SystemExit) as exc_info:
                main()
            assert exc_info.value.code == 1


class TestCLIOutputFormats:
    """Tests for CLI output format handling."""

    @patch('openmeteo.cli.OpenMeteo')
    def test_output_csv(self, mock_client_class, tmp_path, capsys):
        """Test CSV output format."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.current = None
        mock_response.hourly = MagicMock()
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01T00:00"],
            "temperature_2m": [5.0],
        })
        mock_client.forecast.get.return_value = mock_response

        output_file = tmp_path / "output.csv"

        with patch.object(sys, 'argv', [
            'openmeteo', 'forecast',
            '--lat', '52.52', '--lon', '13.41',
            '-o', str(output_file)
        ]):
            main()

        assert output_file.exists()
        content = output_file.read_text()
        assert "time" in content
        assert "temperature_2m" in content

    @patch('openmeteo.cli.OpenMeteo')
    def test_output_json(self, mock_client_class, tmp_path, capsys):
        """Test JSON output format."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.current = None
        mock_response.hourly = MagicMock()
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01T00:00"],
            "temperature_2m": [5.0],
        })
        mock_client.forecast.get.return_value = mock_response

        output_file = tmp_path / "output.json"

        with patch.object(sys, 'argv', [
            'openmeteo', 'forecast',
            '--lat', '52.52', '--lon', '13.41',
            '-o', str(output_file)
        ]):
            main()

        assert output_file.exists()


class TestCLIEdgeCases:
    """Tests for CLI edge cases and error handling."""

    @patch('openmeteo.cli.OpenMeteo')
    def test_forecast_default_variables(self, mock_client_class, capsys):
        """Test forecast with default variables when none specified."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.current = None
        mock_response.hourly = MagicMock()
        mock_response.daily = None
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2024-01-01T00:00"],
            "temperature_2m": [5.0],
        })
        mock_client.forecast.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'forecast',
            '--lat', '52.52', '--lon', '13.41'
        ]):
            main()

        # Should have been called with default hourly variables
        mock_client.forecast.get.assert_called_once()

    @patch('openmeteo.cli.OpenMeteo')
    def test_forecast_with_current_weather(self, mock_client_class, capsys):
        """Test forecast with current weather data."""
        mock_client = mock_client_class.return_value

        mock_current = MagicMock()
        mock_current.time = "2024-01-01T12:00"
        mock_current.data = {"temperature_2m": 5.5}
        mock_current.units = {"temperature_2m": "°C"}

        mock_response = MagicMock()
        mock_response.current = mock_current
        mock_response.hourly = None
        mock_response.daily = None
        mock_client.forecast.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'forecast',
            '--lat', '52.52', '--lon', '13.41',
            '--current', 'temperature_2m'
        ]):
            main()

        captured = capsys.readouterr()
        assert "Current Weather" in captured.out

    @patch('openmeteo.cli.OpenMeteo')
    def test_historical_no_data_returned(self, mock_client_class, capsys):
        """Test historical command when no data returned."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.hourly = None
        mock_response.daily = None
        mock_client.historical.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'historical',
            '--lat', '52.52', '--lon', '13.41',
            '--start', '2023-01-01', '--end', '2023-01-31'
        ]):
            main()

        captured = capsys.readouterr()
        assert "No data returned" in captured.out

    @patch('openmeteo.cli.OpenMeteo')
    def test_flood_no_data_returned(self, mock_client_class, capsys):
        """Test flood command when no data returned."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.daily = None
        mock_client.flood.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'flood',
            '--lat', '52.52', '--lon', '13.41'
        ]):
            main()

        captured = capsys.readouterr()
        assert "No data returned" in captured.out

    @patch('openmeteo.cli.OpenMeteo')
    def test_climate_no_data_returned(self, mock_client_class, capsys):
        """Test climate command when no data returned."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.daily = None
        mock_client.climate.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'climate',
            '--lat', '52.52', '--lon', '13.41',
            '--start', '2050-01-01', '--end', '2050-12-31'
        ]):
            main()

        captured = capsys.readouterr()
        assert "No data returned" in captured.out

    @patch('openmeteo.cli.OpenMeteo')
    def test_flood_with_date_range(self, mock_client_class, capsys):
        """Test flood command with historical date range."""
        mock_client = mock_client_class.return_value

        mock_response = MagicMock()
        mock_response.daily = MagicMock()
        mock_response.to_dataframe.return_value = pd.DataFrame({
            "time": ["2023-01-01"],
            "river_discharge": [100.5],
        })
        mock_client.flood.get.return_value = mock_response

        with patch.object(sys, 'argv', [
            'openmeteo', 'flood',
            '--lat', '52.52', '--lon', '13.41',
            '--start', '2023-01-01', '--end', '2023-01-31'
        ]):
            main()

        mock_client.flood.get.assert_called_once()
        call_kwargs = mock_client.flood.get.call_args[1]
        assert call_kwargs.get('start_date') == '2023-01-01'
        assert call_kwargs.get('end_date') == '2023-01-31'

    @patch('openmeteo.cli.OpenMeteo')
    def test_api_error_handling(self, mock_client_class, capsys):
        """Test error handling for API errors."""
        mock_client = mock_client_class.return_value
        mock_client.forecast.get.side_effect = ValueError("API Error: Invalid coordinates")

        with pytest.raises(SystemExit) as exc_info:
            with patch.object(sys, 'argv', [
                'openmeteo', 'forecast',
                '--lat', '52.52', '--lon', '13.41'
            ]):
                main()

        assert exc_info.value.code == 1
        captured = capsys.readouterr()
        assert "Error" in captured.err
