"""Tests for Open-Meteo API clients."""

import pytest
import responses
from datetime import datetime, timedelta

from openmeteo import OpenMeteo
from openmeteo.base import APIConfig
from openmeteo.clients.forecast import ForecastClient, ForecastVariables
from openmeteo.clients.historical import HistoricalClient
from openmeteo.clients.air_quality import AirQualityClient
from openmeteo.clients.marine import MarineClient
from openmeteo.clients.flood import FloodClient
from openmeteo.clients.climate import ClimateClient
from openmeteo.clients.ensemble import EnsembleClient
from openmeteo.clients.geocoding import GeocodingClient
from openmeteo.clients.elevation import ElevationClient
from openmeteo.models import WeatherResponse, GeocodingResult, ElevationResponse


class TestOpenMeteoClient:
    """Tests for main OpenMeteo client."""

    def test_init_default_config(self):
        """Test default initialization."""
        client = OpenMeteo()
        assert client.config.timeout == 30
        assert client.config.retry_attempts == 3
        assert client.config.api_key is None

    def test_init_custom_config(self):
        """Test custom configuration."""
        config = APIConfig(timeout=60, retry_attempts=5, api_key="test-key")
        client = OpenMeteo(config=config)
        assert client.config.timeout == 60
        assert client.config.retry_attempts == 5
        assert client.config.api_key == "test-key"

    def test_lazy_client_initialization(self):
        """Test that sub-clients are lazily initialized."""
        client = OpenMeteo()
        assert client._forecast is None
        _ = client.forecast
        assert client._forecast is not None

    def test_all_clients_accessible(self):
        """Test all sub-clients are accessible."""
        client = OpenMeteo()
        assert isinstance(client.forecast, ForecastClient)
        assert isinstance(client.historical, HistoricalClient)
        assert isinstance(client.air_quality, AirQualityClient)
        assert isinstance(client.marine, MarineClient)
        assert isinstance(client.flood, FloodClient)
        assert isinstance(client.climate, ClimateClient)
        assert isinstance(client.ensemble, EnsembleClient)
        assert isinstance(client.geocoding, GeocodingClient)
        assert isinstance(client.elevation, ElevationClient)

    def test_context_manager(self):
        """Test context manager functionality."""
        with OpenMeteo() as client:
            assert client.forecast is not None


class TestForecastClient:
    """Tests for ForecastClient."""

    def test_variables_available(self):
        """Test that variables are defined."""
        assert len(ForecastVariables.HOURLY_ALL) > 0
        assert len(ForecastVariables.DAILY_ALL) > 0
        assert "temperature_2m" in ForecastVariables.HOURLY_TEMPERATURE
        assert "precipitation" in ForecastVariables.HOURLY_PRECIPITATION

    def test_validate_coordinates(self):
        """Test coordinate validation."""
        client = ForecastClient()

        # Valid coordinates
        client._validate_coordinates(52.52, 13.41)
        client._validate_coordinates(-90, -180)
        client._validate_coordinates(90, 180)

        # Invalid latitude
        with pytest.raises(ValueError, match="Invalid latitude"):
            client._validate_coordinates(91, 0)

        # Invalid longitude
        with pytest.raises(ValueError, match="Invalid longitude"):
            client._validate_coordinates(0, 181)

    def test_validate_date(self):
        """Test date validation."""
        client = ForecastClient()

        # Valid date
        client._validate_date("2024-01-15", "start_date")

        # Invalid date format
        with pytest.raises(ValueError, match="Invalid start_date"):
            client._validate_date("01-15-2024", "start_date")

    @responses.activate
    def test_get_forecast(self):
        """Test getting forecast data."""
        responses.add(
            responses.GET,
            "https://api.open-meteo.com/v1/forecast",
            json={
                "latitude": 52.52,
                "longitude": 13.41,
                "generationtime_ms": 0.5,
                "timezone": "UTC",
                "hourly": {
                    "time": ["2024-01-01T00:00", "2024-01-01T01:00"],
                    "temperature_2m": [5.0, 4.5],
                },
                "hourly_units": {
                    "time": "iso8601",
                    "temperature_2m": "°C",
                },
            },
            status=200,
        )

        client = ForecastClient()
        response = client.get(
            latitude=52.52,
            longitude=13.41,
            hourly=["temperature_2m"],
        )

        assert isinstance(response, WeatherResponse)
        assert response.location.latitude == 52.52
        assert response.hourly is not None
        assert len(response.hourly.time) == 2


class TestHistoricalClient:
    """Tests for HistoricalClient."""

    @responses.activate
    def test_get_historical(self):
        """Test getting historical data."""
        responses.add(
            responses.GET,
            "https://archive-api.open-meteo.com/v1/archive",
            json={
                "latitude": 52.52,
                "longitude": 13.41,
                "generationtime_ms": 0.5,
                "timezone": "UTC",
                "hourly": {
                    "time": ["2023-01-01T00:00", "2023-01-01T01:00"],
                    "temperature_2m": [2.0, 1.5],
                },
                "hourly_units": {
                    "time": "iso8601",
                    "temperature_2m": "°C",
                },
            },
            status=200,
        )

        client = HistoricalClient()
        response = client.get(
            latitude=52.52,
            longitude=13.41,
            start_date="2023-01-01",
            end_date="2023-01-31",
            hourly=["temperature_2m"],
        )

        assert isinstance(response, WeatherResponse)
        assert response.hourly is not None


class TestAirQualityClient:
    """Tests for AirQualityClient."""

    @responses.activate
    def test_get_air_quality(self):
        """Test getting air quality data."""
        responses.add(
            responses.GET,
            "https://air-quality-api.open-meteo.com/v1/air-quality",
            json={
                "latitude": 52.52,
                "longitude": 13.41,
                "generationtime_ms": 0.5,
                "timezone": "UTC",
                "hourly": {
                    "time": ["2024-01-01T00:00"],
                    "pm10": [15.0],
                    "pm2_5": [8.0],
                },
                "hourly_units": {
                    "time": "iso8601",
                    "pm10": "μg/m³",
                    "pm2_5": "μg/m³",
                },
            },
            status=200,
        )

        client = AirQualityClient()
        response = client.get(
            latitude=52.52,
            longitude=13.41,
            hourly=["pm10", "pm2_5"],
        )

        assert isinstance(response, WeatherResponse)
        assert response.hourly is not None


class TestMarineClient:
    """Tests for MarineClient."""

    @responses.activate
    def test_get_marine(self):
        """Test getting marine data."""
        responses.add(
            responses.GET,
            "https://marine-api.open-meteo.com/v1/marine",
            json={
                "latitude": 54.32,
                "longitude": 10.13,
                "generationtime_ms": 0.5,
                "timezone": "UTC",
                "hourly": {
                    "time": ["2024-01-01T00:00"],
                    "wave_height": [1.5],
                },
                "hourly_units": {
                    "time": "iso8601",
                    "wave_height": "m",
                },
            },
            status=200,
        )

        client = MarineClient()
        response = client.get(
            latitude=54.32,
            longitude=10.13,
            hourly=["wave_height"],
        )

        assert isinstance(response, WeatherResponse)
        assert response.hourly is not None


class TestFloodClient:
    """Tests for FloodClient."""

    @responses.activate
    def test_get_flood(self):
        """Test getting flood data."""
        responses.add(
            responses.GET,
            "https://flood-api.open-meteo.com/v1/flood",
            json={
                "latitude": 52.52,
                "longitude": 13.41,
                "generationtime_ms": 0.5,
                "daily": {
                    "time": ["2024-01-01"],
                    "river_discharge": [100.5],
                },
                "daily_units": {
                    "time": "iso8601",
                    "river_discharge": "m³/s",
                },
            },
            status=200,
        )

        client = FloodClient()
        response = client.get(
            latitude=52.52,
            longitude=13.41,
            daily=["river_discharge"],
        )

        assert response.daily is not None


class TestGeocodingClient:
    """Tests for GeocodingClient."""

    def test_search_name_too_short(self):
        """Test that short names are rejected."""
        client = GeocodingClient()
        with pytest.raises(ValueError, match="at least 2 characters"):
            client.search("B")

    @responses.activate
    def test_search(self):
        """Test location search."""
        responses.add(
            responses.GET,
            "https://geocoding-api.open-meteo.com/v1/search",
            json={
                "results": [
                    {
                        "id": 2950159,
                        "name": "Berlin",
                        "latitude": 52.52437,
                        "longitude": 13.41053,
                        "elevation": 34.0,
                        "country": "Germany",
                        "country_code": "DE",
                        "timezone": "Europe/Berlin",
                    }
                ]
            },
            status=200,
        )

        client = GeocodingClient()
        results = client.search("Berlin")

        assert len(results) == 1
        assert isinstance(results[0], GeocodingResult)
        assert results[0].name == "Berlin"
        assert results[0].latitude == 52.52437


class TestElevationClient:
    """Tests for ElevationClient."""

    def test_batch_limit(self):
        """Test batch size limit."""
        client = ElevationClient()
        # Use valid latitude values (0 to 89) repeated to get 101 coordinates
        latitudes = [i % 90 for i in range(101)]  # Values 0-89, then 0-10
        longitudes = list(range(101))  # Longitudes 0-100 are valid
        with pytest.raises(ValueError, match="Maximum 100 coordinates"):
            client.get(
                latitude=latitudes,
                longitude=longitudes,
            )

    @responses.activate
    def test_get_elevation(self):
        """Test getting elevation."""
        responses.add(
            responses.GET,
            "https://api.open-meteo.com/v1/elevation",
            json={"elevation": [34.0]},
            status=200,
        )

        client = ElevationClient()
        elevation = client.get_elevation(latitude=52.52, longitude=13.41)

        assert elevation == 34.0


class TestWeatherResponse:
    """Tests for WeatherResponse model."""

    def test_from_response(self):
        """Test creating response from API data."""
        data = {
            "latitude": 52.52,
            "longitude": 13.41,
            "generationtime_ms": 0.5,
            "timezone": "UTC",
            "hourly": {
                "time": ["2024-01-01T00:00", "2024-01-01T01:00"],
                "temperature_2m": [5.0, 4.5],
            },
            "hourly_units": {
                "time": "iso8601",
                "temperature_2m": "°C",
            },
        }

        response = WeatherResponse.from_response(data)

        assert response.location.latitude == 52.52
        assert response.location.longitude == 13.41
        assert response.hourly is not None
        assert len(response.hourly.time) == 2

    def test_to_dataframe(self):
        """Test converting response to DataFrame."""
        data = {
            "latitude": 52.52,
            "longitude": 13.41,
            "generationtime_ms": 0.5,
            "hourly": {
                "time": ["2024-01-01T00:00", "2024-01-01T01:00"],
                "temperature_2m": [5.0, 4.5],
            },
            "hourly_units": {"time": "iso8601", "temperature_2m": "°C"},
        }

        response = WeatherResponse.from_response(data)
        df = response.to_dataframe("hourly")

        assert len(df) == 2
        assert "time" in df.columns
        assert "temperature_2m" in df.columns
        assert "latitude" in df.columns
        assert "longitude" in df.columns


class TestAPIConfig:
    """Tests for APIConfig."""

    def test_default_values(self):
        """Test default configuration values."""
        config = APIConfig()
        assert config.timeout == 30
        assert config.retry_attempts == 3
        assert config.retry_delay == 1.0
        assert config.api_key is None

    def test_custom_values(self):
        """Test custom configuration values."""
        config = APIConfig(
            timeout=60,
            retry_attempts=5,
            retry_delay=2.0,
            api_key="test-key"
        )
        assert config.timeout == 60
        assert config.retry_attempts == 5
        assert config.retry_delay == 2.0
        assert config.api_key == "test-key"
