"""Tests for FlatBuffers support."""

import pytest
import struct
from unittest.mock import patch, MagicMock

# Check if FlatBuffers dependencies are available
try:
    import numpy as np
    import flatbuffers
    FLATBUFFERS_AVAILABLE = True
except ImportError:
    FLATBUFFERS_AVAILABLE = False
    np = None

from openmeteo.base import APIConfig, is_flatbuffers_available


class TestAPIConfig:
    """Tests for APIConfig with format option."""

    def test_default_format_is_json(self):
        """Test that default format is JSON."""
        config = APIConfig()
        assert config.format == "json"

    def test_flatbuffers_format(self):
        """Test FlatBuffers format configuration."""
        config = APIConfig(format="flatbuffers")
        assert config.format == "flatbuffers"

    def test_config_with_api_key(self):
        """Test configuration with API key."""
        config = APIConfig(api_key="test-key", format="flatbuffers")
        assert config.api_key == "test-key"
        assert config.format == "flatbuffers"


class TestIsFlatBuffersAvailable:
    """Tests for is_flatbuffers_available function."""

    def test_returns_boolean(self):
        """Test that is_flatbuffers_available returns a boolean."""
        result = is_flatbuffers_available()
        assert isinstance(result, bool)

    @pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="FlatBuffers not installed")
    def test_returns_true_when_installed(self):
        """Test returns True when FlatBuffers is installed."""
        assert is_flatbuffers_available() is True


@pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="FlatBuffers not installed")
class TestFlatBuffersSchema:
    """Tests for FlatBuffers schema definitions."""

    def test_variable_enum(self):
        """Test Variable enum values."""
        from openmeteo.flatbuffers_schema import Variable

        assert Variable.undefined == 0
        assert Variable.temperature == 50
        assert Variable.precipitation == 26

    def test_unit_enum(self):
        """Test Unit enum values."""
        from openmeteo.flatbuffers_schema import Unit

        assert Unit.undefined == 0
        assert Unit.celsius == 1
        assert Unit.fahrenheit == 9

    def test_aggregation_enum(self):
        """Test Aggregation enum values."""
        from openmeteo.flatbuffers_schema import Aggregation

        assert Aggregation.none == 0
        assert Aggregation.minimum == 1
        assert Aggregation.maximum == 2
        assert Aggregation.mean == 3

    def test_variable_names_mapping(self):
        """Test VARIABLE_NAMES mapping."""
        from openmeteo.flatbuffers_schema import Variable, VARIABLE_NAMES

        assert VARIABLE_NAMES[Variable.temperature] == "temperature"
        assert VARIABLE_NAMES[Variable.precipitation] == "precipitation"
        assert VARIABLE_NAMES[Variable.wind_speed] == "wind_speed"


@pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="FlatBuffers not installed")
class TestFlatBuffersParser:
    """Tests for FlatBuffers parser."""

    def test_parser_initialization(self):
        """Test parser can be initialized."""
        from openmeteo.flatbuffers_parser import FlatBuffersParser

        parser = FlatBuffersParser()
        assert parser is not None

    def test_check_flatbuffers_available(self):
        """Test check_flatbuffers_available doesn't raise when installed."""
        from openmeteo.flatbuffers_parser import check_flatbuffers_available

        # Should not raise
        check_flatbuffers_available()


@pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="FlatBuffers not installed")
class TestVariableData:
    """Tests for VariableData dataclass."""

    def test_variable_data_name_basic(self):
        """Test basic variable name generation."""
        from openmeteo.flatbuffers_parser import VariableData
        from openmeteo.flatbuffers_schema import Variable, Unit, Aggregation

        var = VariableData(
            variable=Variable.temperature,
            unit=Unit.celsius,
            values=np.array([20.0, 21.0, 22.0]),
        )
        assert var.name == "temperature"

    def test_variable_data_name_with_altitude(self):
        """Test variable name with altitude suffix."""
        from openmeteo.flatbuffers_parser import VariableData
        from openmeteo.flatbuffers_schema import Variable, Unit, Aggregation

        var = VariableData(
            variable=Variable.temperature,
            unit=Unit.celsius,
            values=np.array([20.0]),
            altitude=2,
        )
        assert var.name == "temperature_2m"

    def test_variable_data_name_with_pressure_level(self):
        """Test variable name with pressure level suffix."""
        from openmeteo.flatbuffers_parser import VariableData
        from openmeteo.flatbuffers_schema import Variable, Unit, Aggregation

        var = VariableData(
            variable=Variable.temperature,
            unit=Unit.celsius,
            values=np.array([20.0]),
            pressure_level=850,
        )
        assert var.name == "temperature_850hPa"

    def test_variable_data_name_with_aggregation(self):
        """Test variable name with aggregation suffix."""
        from openmeteo.flatbuffers_parser import VariableData
        from openmeteo.flatbuffers_schema import Variable, Unit, Aggregation

        var = VariableData(
            variable=Variable.temperature,
            unit=Unit.celsius,
            values=np.array([25.0]),
            altitude=2,
            aggregation=Aggregation.maximum,
        )
        assert var.name == "temperature_2m_maximum"


@pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="FlatBuffers not installed")
class TestTimeSeriesData:
    """Tests for TimeSeriesData from FlatBuffers parser."""

    def test_time_series_to_dict(self):
        """Test TimeSeriesData to_dict conversion."""
        from openmeteo.flatbuffers_parser import TimeSeriesData, VariableData
        from openmeteo.flatbuffers_schema import Variable, Unit, Aggregation

        ts = TimeSeriesData(
            time=np.array([1704067200, 1704070800]),  # Unix timestamps
            interval=3600,
            variables=[
                VariableData(
                    variable=Variable.temperature,
                    unit=Unit.celsius,
                    values=np.array([20.0, 21.0]),
                    altitude=2,
                ),
            ],
        )

        result = ts.to_dict()
        assert "time" in result
        assert "temperature_2m" in result
        assert len(result["time"]) == 2

    def test_time_series_get_variable(self):
        """Test TimeSeriesData get_variable method."""
        from openmeteo.flatbuffers_parser import TimeSeriesData, VariableData
        from openmeteo.flatbuffers_schema import Variable, Unit, Aggregation

        var = VariableData(
            variable=Variable.temperature,
            unit=Unit.celsius,
            values=np.array([20.0]),
            altitude=2,
        )

        ts = TimeSeriesData(
            time=np.array([1704067200]),
            interval=3600,
            variables=[var],
        )

        found = ts.get_variable(Variable.temperature, altitude=2)
        assert found is not None
        assert found.altitude == 2

        not_found = ts.get_variable(Variable.precipitation)
        assert not_found is None


@pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="FlatBuffers not installed")
class TestWeatherApiResponse:
    """Tests for WeatherApiResponse dataclass."""

    def test_response_creation(self):
        """Test WeatherApiResponse creation."""
        from openmeteo.flatbuffers_parser import WeatherApiResponse

        response = WeatherApiResponse(
            latitude=52.52,
            longitude=13.41,
            elevation=38.0,
            timezone="Europe/Berlin",
        )

        assert response.latitude == 52.52
        assert response.longitude == 13.41
        assert response.elevation == 38.0
        assert response.timezone == "Europe/Berlin"


@pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="FlatBuffers not installed")
class TestResponseToDict:
    """Tests for response_to_dict function."""

    def test_basic_response_conversion(self):
        """Test basic response to dict conversion."""
        from openmeteo.flatbuffers_parser import (
            WeatherApiResponse,
            TimeSeriesData,
            VariableData,
            response_to_dict,
        )
        from openmeteo.flatbuffers_schema import Variable, Unit, Aggregation

        response = WeatherApiResponse(
            latitude=52.52,
            longitude=13.41,
            elevation=38.0,
            utc_offset_seconds=3600,
            timezone="Europe/Berlin",
            timezone_abbreviation="CET",
            hourly=TimeSeriesData(
                time=np.array([1704067200, 1704070800]),
                interval=3600,
                variables=[
                    VariableData(
                        variable=Variable.temperature,
                        unit=Unit.celsius,
                        values=np.array([5.0, 6.0]),
                        altitude=2,
                    ),
                ],
            ),
        )

        result = response_to_dict(response)

        assert result["latitude"] == 52.52
        assert result["longitude"] == 13.41
        assert result["elevation"] == 38.0
        assert result["timezone"] == "Europe/Berlin"
        assert "hourly" in result
        assert "temperature_2m" in result["hourly"]


class TestModelsZeroCopy:
    """Tests for zero-copy DataFrame conversion in models."""

    def test_time_series_data_with_lists(self):
        """Test TimeSeriesData with regular lists (JSON format)."""
        from openmeteo.models import TimeSeriesData

        ts = TimeSeriesData(
            time=["2024-01-01T00:00", "2024-01-01T01:00"],
            data={"temperature_2m": [20.0, 21.0]},
            units={"temperature_2m": "°C"},
        )

        df = ts.to_dataframe()
        assert len(df) == 2
        assert "temperature_2m" in df.columns
        assert "time" in df.columns

    @pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="numpy not installed")
    def test_time_series_data_with_numpy_arrays(self):
        """Test TimeSeriesData with numpy arrays (FlatBuffers format)."""
        from openmeteo.models import TimeSeriesData

        ts = TimeSeriesData(
            time=np.array([1704067200, 1704070800], dtype=np.int64),
            data={"temperature_2m": np.array([20.0, 21.0], dtype=np.float32)},
            units={"temperature_2m": "°C"},
        )

        # Zero-copy conversion
        df = ts.to_dataframe(zero_copy=True)
        assert len(df) == 2
        assert "temperature_2m" in df.columns

        # Non zero-copy conversion
        df_copy = ts.to_dataframe(zero_copy=False)
        assert len(df_copy) == 2

    @pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="numpy not installed")
    def test_time_series_to_numpy(self):
        """Test TimeSeriesData to_numpy method."""
        from openmeteo.models import TimeSeriesData

        # With numpy arrays
        ts = TimeSeriesData(
            time=np.array([1704067200, 1704070800], dtype=np.int64),
            data={"temperature_2m": np.array([20.0, 21.0], dtype=np.float32)},
            units={},
        )

        result = ts.to_numpy()
        assert isinstance(result["time"], np.ndarray)
        assert isinstance(result["temperature_2m"], np.ndarray)

    @pytest.mark.skipif(not FLATBUFFERS_AVAILABLE, reason="numpy not installed")
    def test_time_series_to_numpy_from_lists(self):
        """Test TimeSeriesData to_numpy with list data."""
        from openmeteo.models import TimeSeriesData

        ts = TimeSeriesData(
            time=["2024-01-01T00:00", "2024-01-01T01:00"],
            data={"temperature_2m": [20.0, 21.0]},
            units={},
        )

        result = ts.to_numpy()
        assert isinstance(result["time"], np.ndarray)
        assert isinstance(result["temperature_2m"], np.ndarray)


class TestBaseClientFlatBuffers:
    """Tests for base client FlatBuffers support."""

    def test_build_params_json_format(self):
        """Test _build_params without FlatBuffers format."""
        from openmeteo.base import OpenMeteoBaseClient, APIConfig

        class TestClient(OpenMeteoBaseClient):
            BASE_URL = "https://test.api"

        config = APIConfig(format="json")
        client = TestClient(config)

        params = client._build_params(latitude=52.52, longitude=13.41)

        assert "format" not in params
        assert params["latitude"] == 52.52

    def test_build_params_flatbuffers_format(self):
        """Test _build_params with FlatBuffers format."""
        from openmeteo.base import OpenMeteoBaseClient, APIConfig

        class TestClient(OpenMeteoBaseClient):
            BASE_URL = "https://test.api"

        config = APIConfig(format="flatbuffers")

        # Mock the FlatBuffers parser initialization
        with patch.object(TestClient, '_init_flatbuffers_parser'):
            client = TestClient(config)
            params = client._build_params(latitude=52.52, longitude=13.41)

        assert params.get("format") == "flatbuffers"
        assert params["latitude"] == 52.52
