"""Tests for DAG and script generators."""

import pytest
import sys
import os
from io import StringIO

from openmeteo.generator import DagGenerator, StandaloneScriptGenerator


class TestDagGenerator:
    """Tests for DagGenerator class."""

    def test_init_valid_apis(self):
        """Test initialization with valid APIs."""
        gen = DagGenerator(
            apis=["forecast", "air_quality"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        assert gen.apis == ["forecast", "air_quality"]
        assert len(gen.locations) == 1

    def test_init_invalid_api(self):
        """Test initialization with invalid API raises error."""
        with pytest.raises(ValueError, match="Invalid APIs"):
            DagGenerator(
                apis=["forecast", "invalid_api"],
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            )

    def test_init_empty_locations(self):
        """Test initialization with empty locations raises error."""
        with pytest.raises(ValueError, match="At least one location"):
            DagGenerator(
                apis=["forecast"],
                locations=[],
            )

    def test_init_missing_location_name(self):
        """Test initialization with missing location name raises error."""
        with pytest.raises(ValueError, match="missing 'name'"):
            DagGenerator(
                apis=["forecast"],
                locations=[{"lat": 52.52, "lon": 13.41}],
            )

    def test_init_missing_location_lat(self):
        """Test initialization with missing latitude raises error."""
        with pytest.raises(ValueError, match="missing 'lat'"):
            DagGenerator(
                apis=["forecast"],
                locations=[{"name": "Berlin", "lon": 13.41}],
            )

    def test_init_missing_location_lon(self):
        """Test initialization with missing longitude raises error."""
        with pytest.raises(ValueError, match="missing 'lon'"):
            DagGenerator(
                apis=["forecast"],
                locations=[{"name": "Berlin", "lat": 52.52}],
            )

    def test_init_invalid_latitude(self):
        """Test initialization with invalid latitude raises error."""
        with pytest.raises(ValueError, match="latitude"):
            DagGenerator(
                apis=["forecast"],
                locations=[{"name": "Berlin", "lat": 91.0, "lon": 13.41}],
            )

    def test_init_invalid_longitude(self):
        """Test initialization with invalid longitude raises error."""
        with pytest.raises(ValueError, match="longitude"):
            DagGenerator(
                apis=["forecast"],
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 181.0}],
            )

    def test_init_invalid_output_format(self):
        """Test initialization with invalid output format raises error."""
        with pytest.raises(ValueError, match="Invalid output_format"):
            DagGenerator(
                apis=["forecast"],
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
                output_format="xml",
            )

    def test_generate_produces_valid_python(self):
        """Test generated script is valid Python."""
        gen = DagGenerator(
            apis=["forecast"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        script = gen.generate()

        # Check it compiles without syntax errors
        compile(script, "<string>", "exec")

    def test_generate_with_airflow(self):
        """Test generated script with Airflow DAG."""
        gen = DagGenerator(
            apis=["forecast", "air_quality"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=True,
            dag_id="test_dag",
            schedule="@hourly",
        )
        script = gen.generate()

        # Check it compiles
        compile(script, "<string>", "exec")

        # Check Airflow components are present
        assert "AIRFLOW_AVAILABLE" in script
        assert "test_dag" in script
        assert "@hourly" in script
        assert "PythonOperator" in script

    def test_generate_without_airflow(self):
        """Test generated script without Airflow DAG."""
        gen = DagGenerator(
            apis=["forecast"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        script = gen.generate()

        # Check it compiles
        compile(script, "<string>", "exec")

        # Should not have DAG definition
        assert "with DAG(" not in script

    def test_generate_contains_validation(self):
        """Test generated script contains validation functions."""
        gen = DagGenerator(
            apis=["forecast"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        script = gen.generate()

        assert "validate_latitude" in script
        assert "validate_longitude" in script
        assert "validate_locations" in script
        assert "ValidationError" in script

    def test_generate_contains_api_functions(self):
        """Test generated script contains API fetch functions."""
        gen = DagGenerator(
            apis=["forecast", "historical"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        script = gen.generate()

        assert "fetch_forecast_data" in script
        assert "fetch_historical_data" in script
        assert "api.open-meteo.com" in script
        assert "archive-api.open-meteo.com" in script

    def test_generate_multiple_locations(self):
        """Test generated script handles multiple locations."""
        gen = DagGenerator(
            apis=["forecast"],
            locations=[
                {"name": "Berlin", "lat": 52.52, "lon": 13.41},
                {"name": "Paris", "lat": 48.85, "lon": 2.35},
            ],
            include_airflow=False,
        )
        script = gen.generate()

        compile(script, "<string>", "exec")
        assert "Berlin" in script
        assert "Paris" in script
        assert "52.52" in script
        assert "48.85" in script

    def test_generate_output_formats(self):
        """Test generated script respects output format."""
        for fmt in ["csv", "json"]:
            gen = DagGenerator(
                apis=["forecast"],
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
                output_format=fmt,
                include_airflow=False,
            )
            script = gen.generate()
            assert f'OUTPUT_FORMAT = "{fmt}"' in script

    def test_generate_contains_retry_logic(self):
        """Test generated script contains retry logic."""
        gen = DagGenerator(
            apis=["forecast"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        script = gen.generate()

        assert "API_RETRIES" in script
        assert "for attempt in range" in script

    def test_generate_all_apis(self):
        """Test generating with all supported APIs."""
        all_apis = DagGenerator.get_available_apis()
        gen = DagGenerator(
            apis=all_apis,
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        script = gen.generate()

        # Should compile
        compile(script, "<string>", "exec")

        # Should have all API functions
        for api in all_apis:
            assert f"fetch_{api}_data" in script

    def test_get_available_apis(self):
        """Test get_available_apis returns correct list."""
        apis = DagGenerator.get_available_apis()
        expected = ["forecast", "historical", "air_quality", "marine", "flood", "climate", "ensemble"]
        assert sorted(apis) == sorted(expected)

    def test_get_api_description(self):
        """Test get_api_description returns correct descriptions."""
        assert DagGenerator.get_api_description("forecast") == "Weather Forecast"
        assert DagGenerator.get_api_description("historical") == "Historical Weather"
        assert DagGenerator.get_api_description("invalid") == "Unknown API"


class TestStandaloneScriptGenerator:
    """Tests for StandaloneScriptGenerator class."""

    def test_init_valid_api(self):
        """Test initialization with valid API."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        assert gen.api == "forecast"

    def test_init_invalid_api(self):
        """Test initialization with invalid API raises error."""
        with pytest.raises(ValueError, match="Invalid API"):
            StandaloneScriptGenerator(
                api="invalid_api",
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            )

    def test_init_empty_locations(self):
        """Test initialization with empty locations raises error."""
        with pytest.raises(ValueError, match="At least one location"):
            StandaloneScriptGenerator(
                api="forecast",
                locations=[],
            )

    def test_init_invalid_output_format(self):
        """Test initialization with invalid output format raises error."""
        with pytest.raises(ValueError, match="Invalid output_format"):
            StandaloneScriptGenerator(
                api="forecast",
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
                output_format="xml",
            )

    def test_generate_forecast_produces_valid_python(self):
        """Test generated forecast script is valid Python."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()
        compile(script, "<string>", "exec")

    def test_generate_historical_produces_valid_python(self):
        """Test generated historical script is valid Python."""
        gen = StandaloneScriptGenerator(
            api="historical",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()
        compile(script, "<string>", "exec")

        # Should have date arguments
        assert "--start-date" in script
        assert "--end-date" in script

    def test_generate_climate_produces_valid_python(self):
        """Test generated climate script is valid Python."""
        gen = StandaloneScriptGenerator(
            api="climate",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()
        compile(script, "<string>", "exec")

        # Should have date arguments
        assert "--start-date" in script
        assert "--end-date" in script

    def test_generate_air_quality_produces_valid_python(self):
        """Test generated air quality script is valid Python."""
        gen = StandaloneScriptGenerator(
            api="air_quality",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()
        compile(script, "<string>", "exec")
        assert "air-quality-api.open-meteo.com" in script

    def test_generate_marine_produces_valid_python(self):
        """Test generated marine script is valid Python."""
        gen = StandaloneScriptGenerator(
            api="marine",
            locations=[{"name": "Ocean", "lat": 54.32, "lon": 10.13}],
        )
        script = gen.generate()
        compile(script, "<string>", "exec")
        assert "marine-api.open-meteo.com" in script

    def test_generate_flood_produces_valid_python(self):
        """Test generated flood script is valid Python."""
        gen = StandaloneScriptGenerator(
            api="flood",
            locations=[{"name": "River", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()
        compile(script, "<string>", "exec")
        assert "flood-api.open-meteo.com" in script

    def test_generate_ensemble_produces_valid_python(self):
        """Test generated ensemble script is valid Python."""
        gen = StandaloneScriptGenerator(
            api="ensemble",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()
        compile(script, "<string>", "exec")
        assert "ensemble-api.open-meteo.com" in script

    def test_generate_all_apis(self):
        """Test generating scripts for all APIs."""
        for api in StandaloneScriptGenerator.API_URLS.keys():
            gen = StandaloneScriptGenerator(
                api=api,
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            )
            script = gen.generate()
            # Should compile without errors
            compile(script, "<string>", "exec")

    def test_generate_contains_validation(self):
        """Test generated script contains validation."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        assert "validate_latitude" in script
        assert "validate_longitude" in script
        assert "ValidationError" in script

    def test_generate_contains_argparse(self):
        """Test generated script contains argparse."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        assert "argparse" in script
        assert "--output-dir" in script
        assert "--format" in script

    def test_generate_contains_logging(self):
        """Test generated script contains logging."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        assert "logging" in script
        assert "logger" in script

    def test_generate_output_dir_config(self):
        """Test generated script respects output directory."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            output_dir="/custom/path",
        )
        script = gen.generate()
        assert 'OUTPUT_DIR = "/custom/path"' in script

    def test_generate_multiple_locations(self):
        """Test generated script handles multiple locations."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[
                {"name": "Berlin", "lat": 52.52, "lon": 13.41},
                {"name": "Paris", "lat": 48.85, "lon": 2.35},
                {"name": "London", "lat": 51.51, "lon": -0.13},
            ],
        )
        script = gen.generate()

        compile(script, "<string>", "exec")
        assert "Berlin" in script
        assert "Paris" in script
        assert "London" in script

    def test_generate_contains_retry_logic(self):
        """Test generated script contains retry logic."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        assert "RETRIES" in script
        assert "for attempt in range" in script

    def test_generate_contains_main_entry(self):
        """Test generated script contains main entry point."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        assert 'if __name__ == "__main__"' in script
        assert "def main():" in script

    def test_generate_contains_shebang(self):
        """Test generated script contains shebang."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        assert script.startswith("#!/usr/bin/env python3")

    def test_generate_filename_uses_fstring(self):
        """Test generated script uses f-string for filename."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        # Should have an f-string for the filename
        assert "filename = f\"" in script or 'filename = f"' in script


class TestGeneratorIntegration:
    """Integration tests for generators."""

    def test_dag_generator_script_can_be_executed(self, tmp_path):
        """Test that generated DAG script can be imported."""
        gen = DagGenerator(
            apis=["forecast"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        script = gen.generate()

        # Write to temp file
        script_path = tmp_path / "test_dag.py"
        script_path.write_text(script)

        # Try to compile and check for errors
        with open(script_path, "r") as f:
            code = f.read()
        compile(code, str(script_path), "exec")

    def test_standalone_generator_script_can_be_executed(self, tmp_path):
        """Test that generated standalone script can be imported."""
        gen = StandaloneScriptGenerator(
            api="forecast",
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
        )
        script = gen.generate()

        # Write to temp file
        script_path = tmp_path / "test_script.py"
        script_path.write_text(script)

        # Try to compile and check for errors
        with open(script_path, "r") as f:
            code = f.read()
        compile(code, str(script_path), "exec")

    def test_generated_scripts_have_consistent_structure(self):
        """Test that all generated scripts have consistent structure."""
        common_elements = [
            "import requests",
            "import pandas as pd",
            "logging",
            "ValidationError",
        ]

        # Test DAG generator (uses run_pipeline instead of main)
        dag_gen = DagGenerator(
            apis=["forecast"],
            locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            include_airflow=False,
        )
        dag_script = dag_gen.generate()

        for element in common_elements:
            assert element in dag_script, f"DAG script missing: {element}"
        assert "def run_pipeline(" in dag_script, "DAG script missing run_pipeline"
        assert 'if __name__ == "__main__"' in dag_script

        # Test standalone generator for each API (uses main)
        standalone_elements = common_elements + ["def main():"]
        for api in StandaloneScriptGenerator.API_URLS.keys():
            gen = StandaloneScriptGenerator(
                api=api,
                locations=[{"name": "Berlin", "lat": 52.52, "lon": 13.41}],
            )
            script = gen.generate()

            for element in standalone_elements:
                assert element in script, f"{api} script missing: {element}"
