"""Tests for validation module."""

import pytest
from datetime import date

from openmeteo.validators import (
    ValidationError,
    CoordinateValidator,
    DateValidator,
    VariableValidator,
    ResponseValidator,
    InputSanitizer,
)


class TestCoordinateValidator:
    """Tests for CoordinateValidator."""

    def test_valid_latitude(self):
        """Test valid latitude values."""
        CoordinateValidator.validate_latitude(0)
        CoordinateValidator.validate_latitude(90)
        CoordinateValidator.validate_latitude(-90)
        CoordinateValidator.validate_latitude(52.52)

    def test_invalid_latitude(self):
        """Test invalid latitude values."""
        with pytest.raises(ValidationError, match="Invalid latitude"):
            CoordinateValidator.validate_latitude(91)

        with pytest.raises(ValidationError, match="Invalid latitude"):
            CoordinateValidator.validate_latitude(-91)

    def test_latitude_type_error(self):
        """Test latitude type validation."""
        with pytest.raises(ValidationError, match="must be a number"):
            CoordinateValidator.validate_latitude("52.52")

    def test_valid_longitude(self):
        """Test valid longitude values."""
        CoordinateValidator.validate_longitude(0)
        CoordinateValidator.validate_longitude(180)
        CoordinateValidator.validate_longitude(-180)
        CoordinateValidator.validate_longitude(13.41)

    def test_invalid_longitude(self):
        """Test invalid longitude values."""
        with pytest.raises(ValidationError, match="Invalid longitude"):
            CoordinateValidator.validate_longitude(181)

        with pytest.raises(ValidationError, match="Invalid longitude"):
            CoordinateValidator.validate_longitude(-181)

    def test_valid_coordinate_pairs(self):
        """Test valid coordinate pairs."""
        CoordinateValidator.validate_coordinates(52.52, 13.41)
        CoordinateValidator.validate_coordinates([52.52, 48.85], [13.41, 2.35])

    def test_mismatched_coordinate_lengths(self):
        """Test mismatched coordinate list lengths."""
        with pytest.raises(ValidationError, match="same length"):
            CoordinateValidator.validate_coordinates([52.52, 48.85], [13.41])

    def test_empty_coordinates(self):
        """Test empty coordinate lists."""
        with pytest.raises(ValidationError, match="At least one coordinate"):
            CoordinateValidator.validate_coordinates([], [])


class TestDateValidator:
    """Tests for DateValidator."""

    def test_valid_date_format(self):
        """Test valid date format."""
        result = DateValidator.parse_date("2024-01-15")
        assert result == date(2024, 1, 15)

    def test_invalid_date_format(self):
        """Test invalid date format."""
        with pytest.raises(ValidationError, match="Expected format: YYYY-MM-DD"):
            DateValidator.parse_date("01-15-2024")

        with pytest.raises(ValidationError, match="Expected format: YYYY-MM-DD"):
            DateValidator.parse_date("2024/01/15")

    def test_valid_date_range(self):
        """Test valid date range."""
        DateValidator.validate_date_range("2024-01-01", "2024-01-31")

    def test_invalid_date_range_reversed(self):
        """Test reversed date range."""
        with pytest.raises(ValidationError, match="must be before"):
            DateValidator.validate_date_range("2024-01-31", "2024-01-01")

    def test_date_range_with_min_date(self):
        """Test date range with minimum date constraint."""
        min_date = date(2020, 1, 1)

        # Valid - within range
        DateValidator.validate_date_range(
            "2020-01-01", "2020-12-31", min_date=min_date
        )

        # Invalid - before min date
        with pytest.raises(ValidationError, match="before available data"):
            DateValidator.validate_date_range(
                "2019-01-01", "2019-12-31", min_date=min_date
            )

    def test_date_range_with_max_date(self):
        """Test date range with maximum date constraint."""
        max_date = date(2050, 12, 31)

        # Valid - within range
        DateValidator.validate_date_range(
            "2040-01-01", "2040-12-31", max_date=max_date
        )

        # Invalid - after max date
        with pytest.raises(ValidationError, match="after available data"):
            DateValidator.validate_date_range(
                "2051-01-01", "2051-12-31", max_date=max_date
            )

    def test_validate_forecast_days(self):
        """Test forecast days validation."""
        DateValidator.validate_forecast_days(7)
        DateValidator.validate_forecast_days(16)
        DateValidator.validate_forecast_days(0)

        with pytest.raises(ValidationError, match="exceeds maximum"):
            DateValidator.validate_forecast_days(17)

        with pytest.raises(ValidationError, match="cannot be negative"):
            DateValidator.validate_forecast_days(-1)

    def test_validate_past_days(self):
        """Test past days validation."""
        DateValidator.validate_past_days(30)
        DateValidator.validate_past_days(92)

        with pytest.raises(ValidationError, match="exceeds maximum"):
            DateValidator.validate_past_days(93)


class TestVariableValidator:
    """Tests for VariableValidator."""

    def test_valid_forecast_hourly_variables(self):
        """Test valid forecast hourly variables."""
        VariableValidator.validate_variables(
            ["temperature_2m", "precipitation", "wind_speed_10m"],
            VariableValidator.FORECAST_HOURLY,
            "hourly"
        )

    def test_invalid_variable(self):
        """Test invalid variable names."""
        with pytest.raises(ValidationError, match="Invalid hourly"):
            VariableValidator.validate_variables(
                ["temperature_2m", "invalid_variable"],
                VariableValidator.FORECAST_HOURLY,
                "hourly"
            )

    def test_none_variables(self):
        """Test None variables (should pass)."""
        VariableValidator.validate_variables(
            None,
            VariableValidator.FORECAST_HOURLY,
            "hourly"
        )

    def test_valid_air_quality_variables(self):
        """Test valid air quality variables."""
        VariableValidator.validate_variables(
            ["pm10", "pm2_5", "european_aqi"],
            VariableValidator.AIR_QUALITY_HOURLY,
            "hourly"
        )

    def test_valid_marine_variables(self):
        """Test valid marine variables."""
        VariableValidator.validate_variables(
            ["wave_height", "sea_surface_temperature"],
            VariableValidator.MARINE_HOURLY,
            "hourly"
        )

    def test_valid_flood_variables(self):
        """Test valid flood variables."""
        VariableValidator.validate_variables(
            ["river_discharge", "river_discharge_max"],
            VariableValidator.FLOOD_DAILY,
            "daily"
        )

    def test_valid_models(self):
        """Test valid model names."""
        VariableValidator.validate_models(
            ["EC_Earth3P_HR", "MPI_ESM1_2_XR"],
            VariableValidator.CLIMATE_MODELS,
            "models"
        )

    def test_invalid_model(self):
        """Test invalid model names."""
        with pytest.raises(ValidationError, match="Invalid models"):
            VariableValidator.validate_models(
                ["invalid_model"],
                VariableValidator.CLIMATE_MODELS,
                "models"
            )

    def test_empty_models(self):
        """Test empty models list."""
        with pytest.raises(ValidationError, match="At least one model"):
            VariableValidator.validate_models(
                [],
                VariableValidator.CLIMATE_MODELS,
                "models"
            )


class TestResponseValidator:
    """Tests for ResponseValidator."""

    def test_valid_response_structure(self):
        """Test valid response structure."""
        data = {
            "latitude": 52.52,
            "longitude": 13.41,
            "hourly": {"time": [], "temperature_2m": []},
        }
        ResponseValidator.validate_response_structure(
            data, required_fields=["latitude", "longitude"]
        )

    def test_missing_required_fields(self):
        """Test missing required fields."""
        data = {"latitude": 52.52}

        with pytest.raises(ValidationError, match="missing required fields"):
            ResponseValidator.validate_response_structure(
                data, required_fields=["latitude", "longitude"]
            )

    def test_valid_time_series_data(self):
        """Test valid time series data."""
        data = {
            "time": ["2024-01-01T00:00", "2024-01-01T01:00"],
            "temperature_2m": [5.0, 4.5],
            "precipitation": [0.0, 0.1],
        }
        ResponseValidator.validate_time_series_data(data)

    def test_inconsistent_time_series_length(self):
        """Test inconsistent time series data length."""
        data = {
            "time": ["2024-01-01T00:00", "2024-01-01T01:00"],
            "temperature_2m": [5.0],  # Wrong length
        }

        with pytest.raises(ValidationError, match="Inconsistent data length"):
            ResponseValidator.validate_time_series_data(data)

    def test_data_quality_check(self):
        """Test data quality check."""
        data = {
            "time": ["2024-01-01T00:00", "2024-01-01T01:00", "2024-01-01T02:00"],
            "temperature_2m": [5.0, None, 4.5],
            "precipitation": [0.0, 0.1, 0.2],
        }

        quality = ResponseValidator.check_data_quality(data)

        assert quality["total_records"] == 3
        assert quality["null_counts"]["temperature_2m"] == 1
        assert "precipitation" not in quality["null_counts"]
        assert quality["completeness"] < 100


class TestInputSanitizer:
    """Tests for InputSanitizer."""

    def test_sanitize_string(self):
        """Test string sanitization."""
        assert InputSanitizer.sanitize_string("  test  ") == "test"
        assert InputSanitizer.sanitize_string("a" * 2000, max_length=100) == "a" * 100

    def test_sanitize_timezone(self):
        """Test timezone sanitization."""
        assert InputSanitizer.sanitize_timezone("UTC") == "UTC"
        assert InputSanitizer.sanitize_timezone("auto") == "auto"
        assert InputSanitizer.sanitize_timezone("Europe/Berlin") == "Europe/Berlin"
        assert InputSanitizer.sanitize_timezone("  UTC  ") == "UTC"
        assert InputSanitizer.sanitize_timezone("") == "UTC"

    def test_sanitize_unit_valid(self):
        """Test unit sanitization with valid values."""
        assert InputSanitizer.sanitize_unit(
            "celsius", ["celsius", "fahrenheit"], "celsius"
        ) == "celsius"
        assert InputSanitizer.sanitize_unit(
            "CELSIUS", ["celsius", "fahrenheit"], "celsius"
        ) == "celsius"

    def test_sanitize_unit_invalid(self):
        """Test unit sanitization with invalid values."""
        # Should return default and log warning
        assert InputSanitizer.sanitize_unit(
            "kelvin", ["celsius", "fahrenheit"], "celsius"
        ) == "celsius"

    def test_sanitize_unit_empty(self):
        """Test unit sanitization with empty value."""
        assert InputSanitizer.sanitize_unit(
            "", ["celsius", "fahrenheit"], "celsius"
        ) == "celsius"
