from __future__ import annotations
from typing import List, Optional, Any, Dict

from ocp_resources.resource import NamespacedResource, MissingRequiredArgumentError


class ServingRuntime(NamespacedResource):
    """
    https://github.com/kserve/kserve/blob/master/pkg/apis/serving/v1alpha1/servingruntime_types.go
    """

    api_group: str = NamespacedResource.ApiGroup.SERVING_KSERVE_IO

    def __init__(
        self,
        containers: Optional[List[Dict[str, Any]]] = None,
        supported_model_formats: Optional[List[Dict[str, Any]]] = None,
        multi_model: Optional[bool] = None,
        disabled: Optional[bool] = None,
        protocol_versions: Optional[List[str]] = None,
        image_pull_secrets: Optional[List[Dict[str, Any]]] = None,
        grpc_endpoint: Optional[int] = None,
        grpc_data_endpoint: Optional[int] = None,
        http_data_endpoint: Optional[int] = None,
        replicas: Optional[int] = None,
        storage_helper: Optional[Dict[str, Any]] = None,
        built_in_adapter: Optional[Dict[str, Any]] = None,
        **kwargs: Any,
    ) -> None:
        """
        ServingRuntime object

        Args:
            containers (Optional[List[Dict[str, Any]]], mandatory if yaml file is not provided): Containers of the serving runtime.
            supported_model_formats (Optional[List[Dict[str, Any]]]): Model formats supported by the serving runtime.
            multi_model (Optional[bool]): Specifies if the model server can serve multiple models.
            disabled (Optional[bool]):  When set to true it disables the serving runtime.
            protocol_versions (Optional[List[str]]): List of protocols versions used by the serving runtime.
            image_pull_secrets (Optional[List[Dict[str, Any]]]): List of references to secrets used for pulling images used by this SR.
            grpc_endpoint (Optional[str]): Port of the gRPC endpoint.
            grpc_data_endpoint (Optional[str]): Port of the gRPC data endpoint.
            http_data_endpoint (Optional[str]): HTTP endpoint for inferencing.
            replicas (Optional[int]): Number of replicas of the deploymnent generated by this SR.
            storage_helper (Optional[Dict[str, Any]]): Configuration for this runtime's use of the storage helper.
            built_in_adapter Optional[Dict[str, Any]]: Configuration for the built-in adapter.
        """
        super().__init__(**kwargs)
        self.containers = containers
        self.supported_model_formats = supported_model_formats
        self.multi_model = multi_model
        self.disabled = disabled
        self.protocol_versions = protocol_versions
        self.image_pull_secrets = image_pull_secrets
        self.grpc_endpoint = grpc_endpoint
        self.grpc_data_endpoint = grpc_data_endpoint
        self.http_data_endpoint = http_data_endpoint
        self.replicas = replicas
        self.storage_helper = storage_helper
        self.built_in_adapter = built_in_adapter

    def to_dict(self) -> None:
        super().to_dict()
        if not self.kind_dict and not self.yaml_file:
            if not self.containers:
                raise MissingRequiredArgumentError(argument="'containers'")

            self.res["spec"] = {}
            _spec = self.res["spec"]

            _spec["containers"] = self.containers

            if self.supported_model_formats:
                _spec["supportedModelFormats"] = self.supported_model_formats

            if self.multi_model is not None:
                _spec["multiModel"] = self.multi_model

            if self.disabled is not None:
                _spec["disabled"] = self.disabled

            if self.protocol_versions:
                _spec["protocolVersions"] = self.protocol_versions

            if self.image_pull_secrets:
                _spec["imagePullSecrets"] = self.image_pull_secrets

            if self.grpc_endpoint:
                _spec["grpcEndpoint"] = self.grpc_endpoint

            if self.grpc_data_endpoint:
                _spec["grpcDataEndpoint"] = self.grpc_data_endpoint

            if self.http_data_endpoint:
                _spec["httpDataEndpoint"] = self.http_data_endpoint

            if self.replicas:
                _spec["replicas"] = self.replicas

            if self.storage_helper:
                _spec["storageHelper"] = self.storage_helper

            if self.built_in_adapter:
                _spec["builtInAdapter"] = self.built_in_adapter
