import ast
import os
from concurrent.futures import Future, ThreadPoolExecutor, as_completed

import click

from class_generator.class_generator import class_generator


def get_generated_files() -> dict[str, dict[str, str]]:
    """
    This function returns a dictionary containing three keys:
        - `with_end_comment`: A dictionary containing the kind of resources that have the end comment in their file.
        - `without_end_comment`: A dictionary containing the kind of resources that don't have the end comment in their file.
        - `not_generated`: A dictionary containing the kind of resources that are not generated by `class_generator`.
    """
    start_comment: str = (
        "# Generated using https://github.com/RedHatQE/openshift-python-wrapper/blob/main/scripts/resource/README.md"
    )
    end_comment: str = "# End of generated code"
    resources_with_end_comment: dict[str, str] = {}
    resources_without_end_comment: dict[str, str] = {}
    resources_not_generated: dict[str, str] = {}
    exclude_files: list[str] = ["resource.py", "__init__.py", "__pycache__"]
    for root, _, files in os.walk("ocp_resources"):
        for _file in files:
            _kind: str = ""

            if _file in exclude_files:
                continue

            if _file.endswith(".py"):
                file_path = os.path.join(root, _file)
                with open(file_path) as fd:
                    content = fd.read()
                    tree = ast.parse(content)
                    for _cls in (node for node in tree.body if isinstance(node, ast.ClassDef)):
                        bases = [_base.id for _base in _cls.bases if isinstance(_base, ast.Name)]
                        if "Resource" in bases or "NamespacedResource" in bases:
                            _kind = _cls.name

                    if not _kind:
                        click.echo(f"Can't find kind in file {file_path}")
                        continue

                    if start_comment in content and end_comment in content:
                        resources_with_end_comment[_kind] = file_path

                    elif start_comment in content:
                        resources_without_end_comment[_kind] = file_path

                    else:
                        resources_not_generated[_kind] = file_path

    return {
        "with_end_comment": resources_with_end_comment,
        "without_end_comment": resources_without_end_comment,
        "not_generated": resources_not_generated,
    }


def generate_resource(kinds: list[str], yes: bool) -> None:
    for _kind in kinds:
        _generate = "y" if yes else click.prompt(f"Do you want to generate {_kind} resource? (y/n) ")

        if _generate.lower() == "y":
            try:
                class_generator(kind=_kind, called_from_cli=False)
            except Exception:
                continue


@click.command("class-generator tools")
@click.option(
    "--list-generated_file",
    is_flag=True,
    help="list all generated files by `class_generator` under `ocp_resources` directory",
)
@click.option("--yes", is_flag=True, help="Answer yes to all prompts")
@click.option(
    "--generated-missing-end-comment",
    is_flag=True,
    help="Generate missing end comment for all resources under `ocp_resources` directory",
)
@click.option("--regenerate-generated-files", is_flag=True, help="Regenerate all generated files")
def main(
    list_generated_file: bool, generated_missing_end_comment: bool, yes: bool, regenerate_generated_files: bool
) -> None:
    res = get_generated_files()

    if regenerate_generated_files:
        click.echo("Regenerating files...")
        futures: list[Future] = []

        with ThreadPoolExecutor() as executor:
            for kind in res["with_end_comment"].keys():
                futures.append(executor.submit(class_generator, kind=kind, called_from_cli=False, overwrite=True))

        for _ in as_completed(futures):
            ...

    if generated_missing_end_comment:
        generate_resource(kinds=list(res["without_end_comment"].keys()), yes=yes)

    if list_generated_file:
        click.echo("\nResources generated - valid:")
        for k, v in res["with_end_comment"].items():
            click.echo(f"\t{k}: {v}")

        click.echo("\nResources generated - missing end comment:")
        for k, v in res["without_end_comment"].items():
            click.echo(f"\t{k}: {v}")

        click.echo("\nResources not generated:")
        for k, v in res["not_generated"].items():
            click.echo(f"\t{k}: {v}")


if __name__ == "__main__":
    main()
