from __future__ import annotations
import abc
from typing import Annotated, Any, ClassVar, Iterable, Type, Optional, TYPE_CHECKING, get_type_hints, overload, get_origin, get_args
from ormlambda.sql.types import TableType, ComparerType, ColumnType
from ormlambda import ConditionType
from ormlambda.sql.type_api import TypeEngine

if TYPE_CHECKING:
    import re
    from ormlambda import Table
    from ormlambda.sql.comparer import Comparer, Regex, Like


from ormlambda.types import (
    # metadata
    PrimaryKey,
    AutoGenerated,
    AutoIncrement,
    Unique,
    CheckTypes,
    Default,
    NotNull,
)


class Column[TProp]:
    PRIVATE_CHAR: ClassVar[str] = "_"

    _dbtype: Optional[TypeEngine]

    __slots__ = (
        "_dtype",
        "_dbtype",
        "column_name",
        "table",
        "is_primary_key",
        "is_auto_generated",
        "is_auto_increment",
        "is_unique",
        "is_not_null",
        "default_value",
        "sql_type",
        "__private_name",
        "_check",
    )

    @overload
    def __init__(self, *, column_name: str): ...

    @overload
    def __init__[T: Table](
        self,
        dtype: Type[TProp],
        *,
        is_primary_key: bool = False,
        is_auto_generated: bool = False,
        is_auto_increment: bool = False,
        is_unique: bool = False,
        is_not_null: bool = False,
        check_types: bool = True,
        default: Optional[Any] = None,
    ) -> None: ...

    def __init__[T: Table](
        self,
        dtype: Optional[Type[TProp]] = None,
        *,
        is_primary_key: bool = False,
        is_auto_generated: bool = False,
        is_auto_increment: bool = False,
        is_unique: bool = False,
        is_not_null: bool = False,
        check_types: bool = True,
        column_name: Optional[str] = None,
        default: Optional[Any] = None,
    ) -> None:
        if dtype is None and column_name is None:
            raise AttributeError("You must specify either the 'dtype' or 'column_name' attribute.")

        if column_name is not None:
            dtype = type(column_name)

        self.dtype: Type[TProp] = dtype
        self.table: Optional[TableType[T]] = None
        self.column_name: Optional[str] = column_name
        self.__private_name: Optional[str] = None
        self._check = check_types
        self.default_value: TProp = default
        self.sql_type: TypeEngine[TProp] = None

        self.is_primary_key: bool = is_primary_key
        self.is_auto_generated: bool = is_auto_generated
        self.is_auto_increment: bool = is_auto_increment
        self.is_unique: bool = is_unique
        self.is_not_null: bool = is_not_null

    def __repr__(self) -> str:
        return f"{type(self).__name__}[{self.dtype.__name__}] => {self.column_name}"

    def __str__(self) -> str:
        return self.table.__table_name__ + "." + self.column_name

    def __set_name__[T: Table](self, owner: TableType[T], name: str) -> None:
        self.table: TableType[T] = owner
        self.column_name = name
        self.__private_name = self.PRIVATE_CHAR + name

        self._fill_from_annotations(owner, name)

    def __get__(self, obj, objtype=None) -> ColumnType[TProp]:
        if not obj:
            return self
        return getattr(obj, self.__private_name)

    def __set__(self, obj, value):
        if self._check and value is not None:
            type_ = self.dtype if not isinstance(self.dtype, TypeEngine) else self.dtype.python_type
            if not isinstance(value, type_):
                raise ValueError(f"The '{self.column_name}' Column from '{self.table.__table_name__}' table expected '{str(self.dtype)}' type. You passed '{type(value).__name__}' type")
        setattr(obj, self.__private_name, value)

    def __hash__(self) -> int:
        return hash(
            (
                self.dtype,
                self.column_name,
                self.table,
                self.is_primary_key,
                self.is_auto_generated,
                self.is_auto_increment,
                self.is_unique,
                self.is_not_null,
            )
        )

    def _fill_from_annotations[T: Table](self, obj: Type[T], name: str) -> None:
        """Read the metada when using Annotated typing class, and set the attributes accordingly"""

        annotations = get_type_hints(obj, include_extras=True)
        if name in annotations:
            annotation = annotations[name]
            if get_origin(annotation) is Annotated:
                dtype, *metadata = get_args(annotation)

                if not self.dtype:
                    self.dtype = dtype

                for meta in metadata:
                    if isinstance(meta, TypeEngine):
                        self.sql_type = meta
                    elif isinstance(meta, PrimaryKey):
                        self.is_primary_key = True
                    elif isinstance(meta, AutoGenerated):
                        self.is_auto_generated = True
                    elif isinstance(meta, AutoIncrement):
                        self.is_auto_increment = True
                    elif isinstance(meta, Unique):
                        self.is_unique = True
                    elif isinstance(meta, CheckTypes):
                        self._check = True
                    elif isinstance(meta, Default):
                        self.default_value = meta.value
                        self.is_auto_generated = True
                    elif isinstance(meta, NotNull):
                        self.is_not_null = True
        return None

    @property
    def dtype(self) -> Type[TProp]:
        return self._dtype

    @dtype.setter
    def dtype(self, value: Type[TProp] | TypeEngine) -> None:
        if isinstance(value, TypeEngine):
            self._dtype = value.python_type
            self._dbtype = value
        else:
            self._dtype = value
            self._dbtype = None

    @property
    def dbtype(self) -> TypeEngine[TProp]:
        return self._dbtype if self._dbtype else self.dtype

    @abc.abstractmethod
    def __comparer_creator(self, other: ColumnType, compare: ComparerType) -> Comparer:
        from ormlambda.sql.comparer import Comparer

        return Comparer(self, other, compare)

    def __eq__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.EQUAL.value)

    def __ne__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.NOT_EQUAL.value)

    def __lt__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.LESS_THAN.value)

    def __le__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.LESS_THAN_OR_EQUAL.value)

    def __gt__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.GREATER_THAN.value)

    def __ge__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.GREATER_THAN_OR_EQUAL.value)

    def contains(self, other: ColumnType) -> Comparer:
        if not isinstance(other, tuple) and isinstance(other, Iterable):
            other = tuple(other)

        return self.__comparer_creator(other, ConditionType.IN.value)

    def not_contains(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.NOT_IN.value)

    def regex(self, pattern: str, flags: Optional[re.RegexFlag | Iterable[re.RegexFlag]] = None) -> Regex:
        from ormlambda.sql.comparer import Regex

        if not isinstance(flags, Iterable):
            flags = (flags,)
        return Regex(
            left_condition=self,
            right_condition=pattern,
            context=None,
            flags=flags,
        )

    def like(self, pattern: str) -> Like:
        from ormlambda.sql.comparer import Like

        return Like(self, pattern)

    @property
    def __name__(self) -> str:
        return self.column_name
