from __future__ import annotations
from typing import Concatenate, Iterable, override, Type, TYPE_CHECKING, Any, Callable, Optional

from ormlambda import ForeignKey


if TYPE_CHECKING:
    from ormlambda.engine.base import Engine
    from ormlambda.sql.types import AliasType
    from ormlambda import Table
    from ormlambda.statements.types import OrderTypes
    from ormlambda.sql.types import ColumnType
    from ormlambda.statements.types import SelectCols
    from ormlambda.statements.interfaces import IStatements_two_generic
    from ormlambda.statements.types import TypeExists
    from ormlambda.sql.clause_info import IAggregate
    from ormlambda.statements.types import WhereTypes


from ormlambda.sql.clause_info import ClauseInfo
from ormlambda.statements import BaseStatement

from ormlambda import Table, Column
from ormlambda.common.enums import JoinType
from ormlambda.sql.clauses.join import JoinContext, TupleJoinType

from ormlambda.common.global_checker import GlobalChecker
from .query_builder import QueryBuilder

from ormlambda.sql import clauses
from ormlambda.sql import functions as func


# COMMENT: It's so important to prevent information generated by other tests from being retained in the class.
@staticmethod
def clear_list[T, **P](f: Callable[Concatenate[Statements, P], T]) -> Callable[P, T]:
    def wrapper(self: Statements, *args: P.args, **kwargs: P.kwargs) -> T:
        try:
            return f(self, *args, **kwargs)
        except Exception as err:
            raise err
        finally:
            ForeignKey.stored_calls.clear()
            self._query_builder.clear()

    return wrapper


class Statements[T: Table, TRepo](BaseStatement[T, None]):
    def __init__(self, model: T, engine: Engine) -> None:
        super().__init__(model, engine)
        self._query_builder = QueryBuilder(self.dialect)

    @override
    def create_table(self, if_exists: TypeExists = "fail") -> None:
        name: str = self._model.__table_name__
        if self._repository.table_exists(name):
            if if_exists == "replace":
                self._repository.drop_table(name)

            elif if_exists == "fail":
                raise ValueError(f"Table '{self._model.__table_name__}' already exists")

            elif if_exists == "append":
                counter: int = 0
                char: str = ""
                while self._repository.table_exists(name + char):
                    counter += 1
                    char = f"_{counter}"
                name += char

                new_model = self._model
                new_model.__table_name__ = name
                return new_model.create_table(self.dialect)

        query = self.model.create_table(self.dialect)
        self._repository.execute(query)
        return None

    @override
    @clear_list
    def insert(self, instances: T | list[T]) -> None:
        insert = clauses.Insert(self._model, self.repository, self._dialect)
        insert.insert(instances)
        insert.execute()
        return None

    @override
    def delete(self, instances: Optional[T | list[T]] = None) -> None:
        if instances is None:
            response = self.select()
            if len(response) == 0:
                return None
            # [0] because if we do not select anything, we retrieve all columns of the unic model, stored in tuple[tuple[model]] structure.
            # We always going to have a tuple of one element
            return self.delete(response)

        delete = clauses.Delete(self._model, self._repository, engine=self._engine)
        delete.delete(instances)
        delete.execute()
        # not necessary to call self._query_builder.clear() because select() method already call it
        return None

    @override
    @clear_list
    def upsert(self, instances: T | list[T]) -> None:
        upsert = clauses.Upsert(self._model, self._repository, engine=self._engine)
        upsert.upsert(instances)
        upsert.execute()
        return None

    @override
    @clear_list
    def update(self, dicc: dict[str, Any] | list[dict[str, Any]]) -> None:
        update = clauses.Update(self._model, self._repository, self._query_builder.WHERE, engine=self._engine)
        update.update(dicc)
        update.execute()

        return None

    @override
    def limit(self, number: int) -> IStatements_two_generic[T, TRepo]:
        limit = clauses.Limit(number, dialect=self._dialect)
        # Only can be one LIMIT SQL parameter. We only use the last LimitQuery
        self._query_builder.add_statement(limit)
        return self

    @override
    def offset(self, number: int) -> IStatements_two_generic[T, TRepo]:
        offset = clauses.Offset(number, dialect=self._dialect)
        self._query_builder.add_statement(offset)
        return self

    @override
    def count[TProp](
        self,
        selection: None | SelectCols[T, TProp] = lambda x: "*",
        alias="count",
        execute: bool = False,
    ) -> Optional[int]:
        if execute is True:
            return self.select_one(self.count(selection, alias, False), flavour=dict)[alias]

        selection = GlobalChecker.resolved_callback_object(selection, self.models)
        return clauses.Count(
            element=selection,
            alias_clause=alias,
            context=self._query_builder._context,
            dialect=self._dialect,
        )

    @override
    def where(self, conditions: WhereTypes) -> IStatements_two_generic[T, TRepo]:
        # FIXME [x]: I've wrapped self._model into tuple to pass it instance attr. Idk if it's correct

        conditions = GlobalChecker.resolved_callback_object(conditions, self._models)
        if not isinstance(conditions, Iterable):
            conditions = (conditions,)
        self._query_builder.add_statement(clauses.Where(*conditions))
        return self

    @override
    def having(self, conditions: WhereTypes) -> IStatements_two_generic[T, TRepo]:
        conditions = GlobalChecker.resolved_callback_object(conditions, self._models)
        if not isinstance(conditions, Iterable):
            conditions = (conditions,)
        self._query_builder.add_statement(clauses.Having(*conditions))
        return self

    @override
    def order[TValue](self, columns: Callable[[T], TValue], order_type: OrderTypes) -> IStatements_two_generic[T, TRepo]:
        query = GlobalChecker.resolved_callback_object(columns, self._models)
        order = clauses.Order(query, order_type, dialect=self._dialect)
        self._query_builder.add_statement(order)
        return self

    @override
    def concat(self, selector: SelectCols[T, str], alias: str = "concat") -> IAggregate:
        return func.Concat(values=selector, alias_clause=alias, context=self._query_builder._context, dialect=self._dialect)

    @override
    def max[TProp](
        self,
        column: SelectCols[T, TProp],
        alias: str = "max",
        execute: bool = False,
    ) -> int:
        column = GlobalChecker.resolved_callback_object(column, self.models)
        if execute is True:
            return self.select_one(self.max(column, alias, execute=False), flavour=dict)[alias]
        return func.Max(elements=column, alias_clause=alias, context=self._query_builder._context, dialect=self._dialect)

    @override
    def min[TProp](
        self,
        column: SelectCols[T, TProp],
        alias: str = "min",
        execute: bool = False,
    ) -> int:
        column = GlobalChecker.resolved_callback_object(column, self.models)
        if execute is True:
            return self.select_one(self.min(column, alias, execute=False), flavour=dict)[alias]
        return func.Min(elements=column, alias_clause=alias, context=self._query_builder._context, dialect=self._dialect)

    @override
    def sum[TProp](
        self,
        column: SelectCols[T, TProp],
        alias: str = "sum",
        execute: bool = False,
    ) -> int:
        column = GlobalChecker.resolved_callback_object(column, self.models)
        if execute is True:
            return self.select_one(self.sum(column, alias, execute=False), flavour=dict)[alias]
        return func.Sum(elements=column, alias_clause=alias, context=self._query_builder._context, dialect=self._dialect)

    @override
    def join[LTable: Table, LProp, RTable: Table, RProp](self, joins: tuple[TupleJoinType[LTable, LProp, RTable, RProp]]) -> JoinContext[tuple[*TupleJoinType[LTable, LProp, RTable, RProp]]]:
        return JoinContext(self, joins, self._query_builder._context, self._dialect)

    @override
    @clear_list
    def select[TValue, TFlavour, *Ts](
        self,
        selector: Optional[tuple[TValue, *Ts]] = None,
        *,
        flavour: Optional[Type[TFlavour]] = None,
        by: JoinType = JoinType.INNER_JOIN,
        **kwargs,
    ):
        if "alias" in kwargs:
            alias = kwargs.pop("alias")
            kwargs["alias_clause"] = alias
        select_clause = GlobalChecker.resolved_callback_object(selector, self._models)

        if selector is None:
            # COMMENT: if we do not specify any lambda function we assumed the user want to retreive only elements of the Model itself avoiding other models
            result = self.select(selector=lambda x: x, flavour=flavour, by=by)
            # COMMENT: Always we want to retrieve tuple[tuple[Any]]. That's the reason to return result[0] when we ensure the user want only objects of the first table.
            # Otherwise, we wil return the result itself
            if flavour:
                return result
            return () if not result else result[0]

        select = clauses.Select(
            self._models,
            columns=select_clause,
            dialect=self.dialect,
            **kwargs,
        )
        self._query_builder.add_statement(select)

        self._query_builder.by = by
        self._query: str = self._query_builder.query(self._dialect)

        if flavour:
            result = self._return_flavour(self.query, flavour, select, **kwargs)
            if issubclass(flavour, tuple) and isinstance(select_clause, Column | ClauseInfo):
                return tuple([x[0] for x in result])
            return result
        return self._return_model(select, self.query)

    @override
    def select_one[TValue, TFlavour, *Ts](
        self,
        selector: Optional[tuple[TValue, *Ts]] = None,
        *,
        flavour: Optional[Type[TFlavour]] = None,
        by: JoinType = JoinType.INNER_JOIN,
        **kwargs,
    ):
        self.limit(1)

        response = self.select(selector=selector, flavour=flavour, by=by, **kwargs)

        if not isinstance(response, Iterable):
            return response
        if flavour:
            return response[0] if response else None

        # response var could be return more than one element when we work with models an we
        # select columns from different tables using a join query
        # FIXME [x]: before it was if len(response) == 1 and len(response[0]) == 1: return response[0][0]
        if len(response) == 1:
            if isinstance(response[0], Iterable) and len(response[0]) == 1:
                return response[0][0]
            else:
                return response[0]
        return tuple([res[0] for res in response])

    @override
    def first[TValue, TFlavour, *Ts](
        self,
        selector: Optional[tuple[TValue, *Ts]] = None,
        *,
        flavour: Optional[Type[TFlavour]] = None,
        by: JoinType = JoinType.INNER_JOIN,
        **kwargs,
    ):
        return self.select_one(
            selector=selector,
            flavour=flavour,
            by=by,
            **kwargs,
        )

    @override
    def groupby[TProp](self, column: ColumnType[TProp] | Callable[[T], Any]) -> IStatements_two_generic[T]:
        column = GlobalChecker.resolved_callback_object(column, self.models)

        groupby = clauses.GroupBy(column=column, context=self._query_builder._context, dialect=self.dialect)
        # Only can be one LIMIT SQL parameter. We only use the last LimitQuery
        self._query_builder.add_statement(groupby)
        return self

    @override
    def alias[TProp](self, column: SelectCols[T, TProp], alias: AliasType[ClauseInfo[T]]) -> clauses.Alias[T]:
        column = GlobalChecker.resolved_callback_object(column, self.models)

        return clauses.Alias(
            table=column.table,
            column=column,
            alias_clause=alias,
            context=self._query_builder._context,
            dialect=self.dialect,
        )
