from __future__ import annotations
from typing import Annotated, Any, Callable, Iterable, Type, Optional, TYPE_CHECKING, cast, get_type_hints, overload, get_origin, get_args, ClassVar
from ormlambda.sql.elements import ClauseElement
from ormlambda.sql.types import TableType, ComparerType, ColumnType, TypeEngine
from ormlambda import ConditionType
from ormlambda import util

if TYPE_CHECKING:
    import re
    from ormlambda import Table
    from ormlambda.sql.comparer import Comparer, Regex, Like


from .metadata import (
    Metadata,
    PrimaryKey,
    AutoGenerated,
    AutoIncrement,
    Unique,
    CheckTypes,
    Default,
    NotNull,
)


class Column[TProp](ClauseElement):
    __visit_name__ = "column"
    PRIVATE_CHAR: ClassVar[str] = "_"

    __slots__ = (
        "_dtype",
        "column_name",
        "table",
        "is_primary_key",
        "is_auto_generated",
        "is_auto_increment",
        "is_unique",
        "is_not_null",
        "default_value",
        "sql_type",
        "__private_name",
        "_check",
    )

    table: Type[Table]

    @overload
    def __init__(self, *, column_name: str): ...

    @overload
    def __init__[T: Table](
        self,
        dtype: Type[TProp],
        *,
        is_primary_key: bool = False,
        is_auto_generated: bool = False,
        is_auto_increment: bool = False,
        is_unique: bool = False,
        is_not_null: bool = False,
        check_types: bool = True,
        default: Optional[Any] = None,
    ) -> None: ...

    def __init__[T: Table](
        self,
        dtype: Optional[Type[TProp]] = None,
        *,
        is_primary_key: bool = False,
        is_auto_generated: bool = False,
        is_auto_increment: bool = False,
        is_unique: bool = False,
        is_not_null: bool = False,
        check_types: bool = True,
        column_name: Optional[str] = None,
        default: Optional[Any] = None,
    ) -> None:
        if dtype is None and column_name is None:
            raise AttributeError("You must specify either the 'dtype' or 'column_name' attribute.")

        if column_name is not None:
            dtype = type(column_name)

        self._dtype: TypeEngine[TProp] = util.get_type(dtype)
        self.table: Optional[TableType[T]] = None
        self.column_name: Optional[str] = column_name
        self.__private_name: Optional[str] = None
        self._check = check_types
        self.default_value: TProp = default
        self.sql_type: TypeEngine[TProp] = None if not isinstance(dtype, TypeEngine) else dtype

        self.is_primary_key: bool = is_primary_key
        self.is_auto_generated: bool = is_auto_generated
        self.is_auto_increment: bool = is_auto_increment
        self.is_unique: bool = is_unique
        self.is_not_null: bool = is_not_null

    def __repr__(self) -> str:
        return f"{type(self).__name__}[{type(self.dtype).__name__}] => {self.column_name} {"PK" if self.is_primary_key else ''}"

    def __str__(self) -> str:
        return self.table.__table_name__ + "." + self.column_name

    def __set_name__(self, owner: TableType[Table], name: str) -> None:
        self.table: TableType[Table] = owner
        self.column_name = name
        self.__private_name = self.PRIVATE_CHAR + name

        self._fill_from_annotations(owner, name)

    def __get__(self, obj, objtype=None) -> ColumnType[TProp]:
        if not obj:
            return self
        return getattr(obj, self.__private_name)

    def __set__(self, obj, value):
        if self._check and value is not None:
            type_ = self.dtype.python_type
            if not isinstance(value, type_):
                raise ValueError(f"The '{self.column_name}' Column from '{self.table.__table_name__}' table, expected '{str(self.dtype)}' type. You passed '{type(value).__name__}' type")
        setattr(obj, self.__private_name, value)

    def __hash__(self) -> int:
        return hash(
            (
                self.dtype,
                self.column_name,
                self.table,
                self.is_primary_key,
                self.is_auto_generated,
                self.is_auto_increment,
                self.is_unique,
                self.is_not_null,
            )
        )

    @util.preload_module("ormlambda.sql.type_api")
    def _fill_from_annotations[T: Table](self, obj: Type[T], name: str) -> None:
        """Read the metada when using Annotated typing class, and set the attributes accordingly"""

        TypeEngine = util.preloaded.sql_type_api.TypeEngine

        annotations = get_type_hints(obj, include_extras=True)
        if name in annotations:
            annotation = annotations[name]
            if get_origin(annotation) is not Annotated:
                return FileNotFoundError

            _, *metadata = get_args(annotation)

            type ObjType = Metadata | TypeEngine

            ANNOTATED_SELECTOR: dict[ObjType, Callable[[Column, ObjType], None]] = {
                TypeEngine: lambda x, data: setattr(x, "_dtype", data),
                PrimaryKey: lambda x, _: setattr(x, "is_primary_key", True),
                AutoGenerated: lambda x, _: setattr(x, "is_auto_generated", True),
                AutoIncrement: lambda x, _: setattr(x, "is_auto_increment", True),
                Unique: lambda x, _: setattr(x, "is_unique", True),
                CheckTypes: lambda x, _: setattr(x, "_check", True),
                Default: lambda x, meta: (
                    setattr(x, "default_value", cast(Default, meta).value),
                    setattr(x, "is_auto_generated", True),
                ),
                NotNull: lambda x, _: setattr(x, "is_not_null", True),
            }

            for meta in metadata:
                if not isinstance(meta, Metadata | TypeEngine):
                    raise ValueError(f"When using 'Annotated' to pass metadata into Column, you must instantiate those metadata. You passed '{meta.__name__}'")
                func = ANNOTATED_SELECTOR.get(type(meta), None)

                if not func:
                    continue
                func(self, meta)

    @property
    def dtype(self) -> TypeEngine[TProp]:
        return self._dtype

    @util.preload_module("ormlambda.sql.comparer")
    def __comparer_creator(self, other: ColumnType, compare: ComparerType) -> Comparer:
        Comparer = util.preloaded.sql_comparer.Comparer

        return Comparer(self, other, compare)

    def __eq__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.EQUAL.value)

    def __ne__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.NOT_EQUAL.value)

    def __lt__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.LESS_THAN.value)

    def __le__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.LESS_THAN_OR_EQUAL.value)

    def __gt__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.GREATER_THAN.value)

    def __ge__(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.GREATER_THAN_OR_EQUAL.value)

    def contains(self, other: ColumnType) -> Comparer:
        if not isinstance(other, tuple) and isinstance(other, Iterable):
            other = tuple(other)

        return self.__comparer_creator(other, ConditionType.IN.value)

    def not_contains(self, other: ColumnType) -> Comparer:
        return self.__comparer_creator(other, ConditionType.NOT_IN.value)

    @util.preload_module("ormlambda.sql.comparer")
    def regex(self, pattern: str, flags: Optional[re.RegexFlag | Iterable[re.RegexFlag]] = None) -> Regex:
        Regex = util.preloaded.sql_comparer.Regex

        if not isinstance(flags, Iterable):
            flags = (flags,)
        return Regex(
            left_condition=self,
            right_condition=pattern,
            flags=flags,
        )

    @util.preload_module("ormlambda.sql.comparer")
    def like(self, pattern: str) -> Like:
        Like = util.preloaded.sql_comparer.Like

        return Like(self, pattern)

    @property
    def __name__(self) -> str:
        return self.column_name
