import pytest

from tests.support_common import validate_tdf3_file


@pytest.mark.integration
def test_test_data_directory_structure(tdf_v4_2_2_files, tdf_v4_3_1_files):
    """Test that the TDF files are properly generated by fixtures."""

    # Check v4.2.2 TDF files exist and are valid
    expected_v4_2_2_files = ["text", "binary", "with_attributes"]
    for file_key in expected_v4_2_2_files:
        assert file_key in tdf_v4_2_2_files, (
            f"v4.2.2 TDF file key should exist: {file_key}"
        )
        tdf_file_path = tdf_v4_2_2_files[file_key]
        validate_tdf3_file(
            tdf_file_path, f"otdfctl generated using target mode v4.2.2 {file_key}"
        )

    # Check v4.3.1 TDF files exist and are valid
    expected_v4_3_1_files = ["text", "binary", "with_attributes"]
    for file_key in expected_v4_3_1_files:
        assert file_key in tdf_v4_3_1_files, (
            f"v4.3.1 TDF file key should exist: {file_key}"
        )
        tdf_file_path = tdf_v4_3_1_files[file_key]
        validate_tdf3_file(
            tdf_file_path, f"otdfctl generated using target mode v4.3.1 {file_key}"
        )

    # Verify the TDF files are in the correct directory structure
    for file_path in tdf_v4_2_2_files.values():
        assert "v4.2.2" in str(file_path), (
            f"v4.2.2 TDF file should be in v4.2.2 directory: {file_path}"
        )

    for file_path in tdf_v4_3_1_files.values():
        assert "v4.3.1" in str(file_path), (
            f"v4.3.1 TDF file should be in v4.3.1 directory: {file_path}"
        )


@pytest.mark.integration
def test_sample_file_contents(sample_input_files):
    """Test that sample input files have expected content."""

    # Check text file has content
    text_file = sample_input_files["text"]
    assert text_file.exists(), f"Text file should exist: {text_file}"
    with open(text_file) as f:
        content = f.read()
    assert "Hello, World!" in content
    assert len(content) > 0

    # Check empty file is empty
    # empty_file = sample_input_files["empty"]
    # assert empty_file.exists(), f"Empty file should exist: {empty_file}"
    # assert empty_file.stat().st_size == 0

    # Check binary file exists and has content
    binary_file = sample_input_files["binary"]
    assert binary_file.exists(), f"Binary file should exist: {binary_file}"
    assert binary_file.stat().st_size > 0

    # Check attributes file has content
    attr_file = sample_input_files["with_attributes"]
    assert attr_file.exists(), f"Attributes file should exist: {attr_file}"
    with open(attr_file) as f:
        content = f.read()
    assert "Classification: SECRET" in content


@pytest.mark.integration
def test_target_mode_fixtures_exist(all_target_mode_tdf_files, known_target_modes):
    """Test that target mode fixtures generate TDF files correctly."""
    # Check that we have both versions
    assert "v4.2.2" in all_target_mode_tdf_files
    assert "v4.3.1" in all_target_mode_tdf_files

    # Check each version has the expected file types
    for target_mode in known_target_modes:
        tdf_files = all_target_mode_tdf_files[target_mode]

        # Check all expected file types exist
        expected_types = [
            "text",
            "binary",
            "with_attributes",
        ]  # Consider 'empty' as well
        for file_type in expected_types:
            assert file_type in tdf_files, f"Missing {file_type} TDF for {target_mode}"

            # Check the TDF file exists and is not empty
            tdf_path = tdf_files[file_type]
            validate_tdf3_file(
                tdf_path,
                f"otdfctl generated using target-mode {target_mode} {file_type}",
            )


@pytest.mark.integration
def test_v4_2_2_tdf_files(tdf_v4_2_2_files):
    """Test that v4.2.2 TDF fixtures work independently."""
    assert "text" in tdf_v4_2_2_files
    assert tdf_v4_2_2_files["text"].exists()


@pytest.mark.integration
def test_v4_3_1_tdf_files(tdf_v4_3_1_files):
    """Test that v4.3.1 TDF fixtures work independently."""
    assert "text" in tdf_v4_3_1_files
    assert tdf_v4_3_1_files["text"].exists()
