# OxaPay Python SDK

A Python SDK for interacting with the [OxaPay API](https://oxapay.com/)

## Installation

PyPi directory - [Tap Here](https://pypi.org/project/oxapay-python/)

Install the SDK from PyPI using [pip](https://pip.pypa.io/en/stable/installation/):

```bash
$ pip install oxapay-python
```

## Documentation 

Refer to the official [OxaPay API Documentation](https://docs.oxapay.com/api-reference) for detailed information on available methods and features.

## Authorization

To use the **OxaPay Python SDK**, you need to obtain the following keys from the OxaPay website:

- **Merchant Key**
- **Payout Key**
- **General Key**
    
These keys are essential for authenticating and securely interacting with OxaPay's APIs. Make sure to log in to your OxaPay account, navigate to the appropriate section, and generate or retrieve these keys before integrating the SDK into your application.

## Usage Examples

### Payments

```python
from oxapay_python import Payment

merchant_key = 'YOUR_MERCHANT_KEY'

payment = Payment(merchant_key)

```

### Payouts

```python
from oxapay_python import Payout

payout_key = 'YOUR_PAYOUT_KEY'

payout = Payout(payout_key)

```

### General

```python
from oxapay_python import General

general_key = 'YOUR_GENERAL_KEY'

general = General(general_key)

```

### Notes

- Replace YOUR_MERCHANT_KEY, YOUR_PAYOUT_KEY, and YOUR_GENERAL_KEY with the actual keys obtained from your OxaPay account.
- Always keep your keys secure and do not expose them in public repositories or logs.

## Payment Methods

### Generate Invoice 

- Create an invoice URL for payments.

**Request Example:**

```python
data = {
    "amount": 100,
    "currency": "USD",
    "lifetime": 30,
    "fee_paid_by_payer": 1,
    "under_paid_coverage": 2.5,
    "to_currency": "USDT",
    "auto_withdrawal": False,
    "mixed_payment": True,
    "callback_url": "https://example.com/callback",
    "return_url": "https://example.com/success",
    "email": "customer@oxapay.com",
    "order_id": "ORD-12345",
    "thanks_message": "Thanks message",
    "description": "Order #12345",
    "sandbox": False
}
result = payment.generate_invoice(data)
print(result)
```

**Response Example:**

```python
{
    "track_id": "184747701",
    "payment_url": "http://pay.oxapay.com/12373985/184747701",
    "expired_at": 1734546589,
    "date": 1734510589
}
```

### Generate White Label

- Create a white-label address.

**Request Example:**

```python
data = {
    "amount": 100,
    "currency": "USD",
    "pay_currency": "TRX",
    "network": "TRC20",
    "lifetime": 60,
    "fee_paid_by_payer": 1,
    "under_paid_coverage": 20,
    "to_currency": "USDT",
    "auto_withdrawal": False,
    "callback_url": "https://example.com/callback",
    "email": "customer@oxapay.com",
    "order_id": "ORD-12345",
    "description": "Order #12345",
}
result = payment.generate_white_label(data)
print(result)
```

**Response Example:**

```python
{
    "track_id": "197218291",
    "amount": 1,
    "currency": "usd",
    "pay_amount": 3.6819352872888,
    "pay_currency": "trx",
    "network": "Tron Network",
    "address": "TQraad2MBFgYUYhTFYeXS5jVpaLZuBwr3w",
    "callback_url": "https://example.com/callback",
    "description": "Order #12345",
    "email": "customer@oxapay.com",
    "fee_paid_by_payer": 1,
    "lifetime": 60,
    "order_id": "ORD-12345",
    "under_paid_coverage": 20,
    "rate": 3.67376929,
    "qr_code": "https://api.qrserver.com/v1/create-qr-code/?data=tron:TQraad2MBFgYUYhTFYeXS5jVpaLZuBwr3w?amount=3&size=150x150",
    "expired_at": 1734514150,
    "date": 1734510550
}
```

### Generate Static Address

- Create a static address.

**Request Example:**

```python
data = {
    "network": "TRON",
    "to_currency": "USDT",
    "auto_withdrawal": False,
    "callback_url": "https://example.com/callback",
    "email": "customer@oxapay.com",
    "order_id": "ORD-12345",
    "description": "Order #12345"
}
result = payment.generate_static_address(data)
print(result)
```

**Response Example:**

```python
{
    "track_id": "182796736",
    "network": "Tron Network",
    "address": "TX7U8wnnafXeXED3ChjeuiZFznL3BU1gX3",
    "qr_code": "https://api.qrserver.com/v1/create-qr-code/?data=tron:TX7U8wnnafXeXED3ChjeuiZFznL3BU1gX3&size=150x150",
    "date": 1734511347
}
```

### Payment Information

- Get information for a specific payment transaction.

**Request Example:**

```python
track_id = 184747701
result = payment.payment_information(track_id)
print(result)
```

**Response Example:**

```python
{
    "track_id": "168931368",
    "type": "invoice",
    "amount": 15,
    "currency": "USD",
    "status": "paid",
    "mixed_payment": false,
    "fee_paid_by_payer": 0,
    "under_paid_coverage": 21.5,
    "lifetime": 60,
    "callback_url": "https://example.com/callback",
    "return_url": "https://example.com/success",
    "email": "customer@oxapay.com",
    "order_id": "ORD-12345",
    "description": "Order #12345",
    "thanks_message": "Thanks Message",
    "expired_at": 1728121075,
    "date": 1728117475,
    "txs": [
        {
            "tx_hash": "0x2DehmnxMDFMhS3KHXCPpGSx41NeLBjH34q5vjN1oGbUDFFAMC9rG3",
            "amount": 0.00613418,
            "currency": "ETH",
            "network": "Ethereum Network",
            "address": "HEHCVjnxD8cQrFwfo4k9WXJXErpmK9Ym1nRaEiJJ8DP4",
            "status": "confirmed",
            "confirmations": 11,
            "auto_convert": {
                "processed": true,
                "amount": 14.73130872,
                "currency": "USDT"
            },
            "auto_withdrawal": {
                "processed": false
            },
            "date": 1728117736
        }
    ]
}
```


## Payout Methods

### Generate Payout

- Create a payout request.

**Request Example:**

```python
data = {
    "address": "RECEIVER_ADDRESS",
    "amount": 5,
    "currency": "TRX",
    "network": "TRC20",
    "callback_url": "https://example.com/callback",
    "memo": "Memo12345",
    "description": "Order #12345"
}
result = payout.generate_payout(data)
print(result)
```

**Response Example:**

```python
{
    "track_id": "258298451",
    "status": "processing"
}
```

### Payout Information

- Get information for a specific payout transaction.

**Request Example:**

```python
track_id = 258298451
result = payout.payout_information(track_id)
print(result)
```

**Response Example:**

```python
{
    "track_id": "258298351",
    "address": "1AmH3Qz2LooYa1YSyLhySuatwoRMsfznPJ",
    "currency": "BTC",
    "network": "Bitcoin Network",
    "amount": 0.0200000000,
    "fee": 0.0000100000,
    "status": "processing",
    "tx_hash": "",
    "description": "test",
    "internal": false,
    "memo": "test",
    "date": 1736501470
}
```

## General Methods

### Swap Request

- Allows you to swap your account assets.

**Request Example:**

```python
data = {
    "amount": 0.5,
    "from_currency": "BTC",
    "to_currency": "USDT"
}
result = general.swap_request(data)    
print(result)
```

**Response Example:**

```python
{
    "track_id": "393831199",
    "from_currency": "USDT",
    "to_currency": "BTC",
    "from_amount": 0.5000000000,
    "to_amount": 47438.330170780,
    "rate": 94876.66034156,
    "date": 1736508792
}
```

### Swap Calculate

- Calculate the amount you'll receive when swapping one cryptocurrency for another.

**Request Example:**

```python
data = {
    'amount': 12,
    'fromCurrency': 'USDT',
    'toCurrency': 'TRX'
}
result = general.swap_calculate(data)
print(result)
```

**Response Example:**

```python
{
    "to_amount": 94763.06,
    "rate": 94763.06000000,
    "amount": 1
}
```

### Get Account Balance

- Retrieve the account balance.

**Request Example:**

```python
result = general.get_account_balance()
print(result)
```

**Response Example:**

```python
[
    "BNB": 0.0000000000,
    "DGB": 0.0000000000,
    "XMR": 0.0000000000,
    "BTC": 2.3502949200,
    "ETH": 0.0000000000,
    "USDC": 0.0000000000,
    "POL": 0.0000000000,
    "SOL": 0.0000000000,
    "NOT": 0.0000000000,
    "SHIB": 0.0000000000,
    "TRX": 0.0000000000,
    "USDT": 10746.7139400000,
    "DOGS": 0.0000000000,
    "TON": 0.0000000000,
    "BCH": 0.0000000000,
    "DOGE": 0.0000000000,
    "LTC": 0.0000000000
]
```

## Common Methods

### Supported Currencies 

- Get the list of supported currencies.

**Request Example:**

```python
oxapay_service = Common()
result = oxapay_service.supported_currencies()
print(result)
```

**Response Example:**

```python
{
    "BTC": {
        "symbol": "BTC",
        "name": "Bitcoin",
        "status": true,
        "networks": {
            "Bitcoin": {
                "network": "Bitcoin",
                "name": "Bitcoin Network",
                "required_confirmations": 2,
                "withdraw_fee": 0.01,
                "withdraw_min": 1.0e-5,
                "deposit_min": 1.0e-6,
                "static_fixed_fee": 2
            }
        }
    },
    "ETH": {
        "symbol": "ETH",
        "name": "Ethereum",
        "status": true,
        "networks": {
            "Ethereum": {
                "network": "Ethereum",
                "name": "Ethereum Network",
                "required_confirmations": 10,
                "withdraw_fee": 0.0011,
                "withdraw_min": 0.00011,
                "deposit_min": 1.0e-6,
                "static_fixed_fee": 1
            },
            "Base": {
                "network": "Base",
                "name": "Base",
                "required_confirmations": 10,
                "withdraw_fee": 0.0017,
                "withdraw_min": 0.0001,
                "deposit_min": 1.0e-6,
                "static_fixed_fee": 0
            }
        }
    },
    { ... }, 
}
```

### Supported Networks

- Get the list of supported networks.

**Request Example:**

```python
oxapay_service = Common()
result = oxapay_service.supported_networks()
print(result)
```

**Response Example:**

```python
{
    "list": [
        "Bitcoin Network",
        "Ethereum Network",
        "Tron Network",
        "Binance Smart Chain",
        "Dogecoin Network",
        "Solana Network",
        "Polygon Network",
        "Litecoin Network",
        "DigiByte Network",
        "Monero Network",
        "TON Network",
        "Bitcoin Cash Network",
        "Base"
    ]
}
```
### Price

- Get current prices for supported assets.

**Request Example:**

```python
oxapay_service = Common()
result = oxapay_service.price()
print(result)
```

**Response Example:**

```python
{
    "DOGE": 0.3341700000,
    "BNB": 695.7900000000,
    "XMR": 198.4200000000,
    "BTC": 94876.5500000000,
    "ETH": 3301.4500000000,
    "USDC": 1.0001000000,
    "POL": 0.4591000000,
    "SOL": 191.3500000000,
    "NOT": 0.0059799000,
    "SHIB": 0.0000216300,
    "TRX": 0.2422000000,
    "USDT": 1.0000000000,
    "DOGS": 0.0004590000,
    "TON": 5.2515000000,
    "BCH": 433.5000000000,
    "DGB": 0.0124400000,
    "LTC": 103.9900000000
}
```

## Requirements

* **Python 3** or higher
* requests library:  **pip install requests**
