from .request_builder import RequestBuilder


class General:

    def __init__(self, general_key: str):
        self.general_key = general_key
        self.version = 'v1'
        self.request_builder = RequestBuilder(general_key)

    def swap_request(self, data):
        """
        Allows you to your account assets.
        
        :param dict data:
                        str from_currency (Required): Currency symbol to from
                        str to_currency (Required): Currency symbol to swap to
                        decimal amount (Required): Amount of currency to swap
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/general/swap'
        return self.request_builder.send_request(uri, data = data)


    def swap_history(self, params: dict = {}):
        """
        Get the list of swap transactions.
        
        :param dict data:
                    string status: Filter payouts by status
                    string type: Filter payouts by type
                    string currency: Filter payouts by a specific currency
                    string network: Filter payouts by network
                    string address: Filter payouts by the expected address
                    decimal from_amount: Filter payouts with amounts ≥ specified value
                    decimal to_amount: Filter payouts with amounts ≤ specified value
                    integer from_date: Filter payouts with date ≥ specified value
                    integer to_date: Filter payouts with date ≤ specified value
                    string sort_by: Sort by create_date (default)
                    string sort_type: Order by asc/desc
                    int size: Records per page
                    int page: Page number to retrieve
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + f'/general/swap'
        return self.request_builder.send_request(uri, params=params, method="GET")
    
    def swap_pairs(self):
        """
        Get a list of swappable cryptocurrencies along with their minimum swap amounts.
        
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/general/swap/pairs'
        return self.request_builder.send_request(uri, method="GET")
    
    def swap_calculate(self, data: dict):
        """
        Calculate the amount you'll receive when swapping one cryptocurrency for another.

        :param dict data:
                        str from_currency (Required): Currency symbol to swap from
                        str to_currency (Required): Currency symbol to swap to
                        decimal amount (Required): Amount of currency to swap
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/general/swap/calculate'
        return self.request_builder.send_request(uri, data=data)
    
    def swap_rate(self, data: dict):
        """
        Allows you to fetch real-time swap rates for cryptocurrency pairs.
        
        :param dict data:
                        str from_currency (Required): Currency symbol to swap from
                        str to_currency (Required): Currency symbol to swap to
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/general/swap/rate'
        return self.request_builder.send_request(uri, data=data)
    
    
    def get_account_balance(self):
        """
        Retrieve the account balance.

        Sends a GET request to the '/general/account/balance' endpoint.

        :return: JSON response containing account balance details.
        :raises RequestExceptionsBuilder: If the request fails or the server returns an error.
        """
        uri = self.version + '/general/account/balance'
        return self.request_builder.send_request(uri=uri, method='get')