from .request_builder import RequestBuilder


class Payment:

    def __init__(self, merchant_key: str):
        self.payment_key = merchant_key
        self.version = 'v1'
        self.request_builder = RequestBuilder(merchant_key)

    def generate_invoice(self, data: dict):
        """
        Create an invoice url for payments.

        :param dict data:
                         decimal amount (Required): Payment amount
                         str currency: Payment currency
                         str return_url: Redirect URL
                         str callback_url: Payment callback URL
                         int lifetime: Payment duration (min)
                         str order_id: Unique order ID
                         decimal under_paid_coverage: Cover underpaid amounts
                         bool mixed_payment: Allow mixed payments
                         bool auto_withdrawal: Auto transfer funds
                         str to_currency: Target currency for conversion
                         str email: Payer's email address
                         str thanks_message: Custom thanks message for payer
                         str description: Invoice description
                         bool fee_paid_by_payer: Payer covers transaction fee
                         bool sandbox: Specify sandbox mode
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/payment/invoice'
        return self.request_builder.send_request(uri, data)
    
    def generate_white_label(self, data: dict):
        """
        Create a white-label address.

        :param dict data:
                         decimal amount (Required): Payment amount 
                         str pay_currency (Required): Actual payment currency 
                         str currency: Requested Payment currency
                         str network: Blockchain network
                         str callback_url: Payment callback URL
                         int lifetime: Payment duration (min)
                         str order_id: Unique order ID
                         decimal under_paid_coverage: Cover underpaid amounts
                         bool auto_withdrawal: Auto transfer funds
                         str to_currency: Target currency for conversion
                         str email: Payer's email address
                         bool fee_paid_by_payer: Payer covers transaction fee
                         str description: Invoice description
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/payment/white-label'
        return self.request_builder.send_request(uri, data)
    
    def generate_static_address(self, data: dict):
        """
        Create a static address.
        
        :param dict data:
                         str network (Required): Blockchain network
                         str to_currency: Target currency for conversion
                         bool auto_withdrawal: Auto transfer funds
                         str callback_url: Payment callback URL
                         str email: Payer's email address
                         str order_id: Unique order ID
                         str description: Invoice description
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/payment/static-address'
        return self.request_builder.send_request(uri, data)
    
    def revoking_static_address(self, data: dict):
        """
        Revoke a static address.

        :param dict data:
                        str address (Required): Address to revoke 
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/payment/static-address/revoke'
        return self.request_builder.send_request(uri, data)

    def static_address_list(self, params: dict = {}):
        """
        Retrieve a list of static addresses.

        :param dict data:
                    int track_id: Filter addresses by a ID
                    string network: Filter addresses by network
                    string currency: Filter addresses by currency
                    string address: Filter addresses by expected address
                    boolean have_tx: Filter addresses with transactions
                    string order_id: Filter addresses by order ID
                    string email: Filter addresses by email
                    int page: Page number to retrieve
                    int size: Records per page
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + f'/payment/static-address'
        return self.request_builder.send_request(uri,params=params,method='GET')
    
    def payment_information(self, trackId: str):
        """
        Get information for a specific payment transaction.

        :param dict data:
                    int track_id (Required): Transaction id
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + f'/payment/{trackId}'
        return self.request_builder.send_request(uri,method='GET')


    def payment_history(self, params: dict = {}):
        """
        Retrieve a list of payment history.

        :param dict data:
                    int track_id: Filter payments by a ID
                    string type: Filter payments by type
                    string status: Filter payments by status
                    string pay_currency: Filter payments by a specific paid currency
                    string currency: Filter payments by a specific currency
                    string network: Filter payments by network
                    string address: Filter payments by the expected address
                    decimal from_amount: Filter payments with amounts ≥ specified value
                    decimal to_amount: Filter payments with amounts ≤ specified value
                    int page: Page number to retrieve
                    int size: Records per page
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """
        
        uri = self.version + '/payment'
        return self.request_builder.send_request(uri, params=params, method='GET')

    def accepted_currencies(self):
        """
        Get the list of accepted currencies for payments.

        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """
        
        uri = self.version + f'/payment/accepted-currencies'
        return self.request_builder.send_request(uri,method='GET')
    