from .request_builder import RequestBuilder


class Payout:

    def __init__(self, payout_key: str):
        self.payout_key = payout_key
        self.version = 'v1'
        self.request_builder = RequestBuilder(payout_key)

    def generate_payout(self, data):
        """
        Create a payout request.

        :param dict data:
                        str address (Required): Recipient's wallet address
                        str currency (Required): currency symbol
                        decimal amount (Required): Exact amount of currency to send
                        str network: Blockchain network for payout
                        str callback_url: Payout callback URL
                        str description: Payout description or additional info
                        str memo: Memo or tag for transactions
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/payout'
        return self.request_builder.send_request(uri, data)


    def payout_information(self, trackId: str):
        """
        Get information for a specific payout transaction.
        
        :param dict data:
                    int track_id (Required): Transaction id
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + f'/payout/{trackId}'
        return self.request_builder.send_request(uri, method="GET")
    
    def payout_history(self, params: dict = {}):
        """
        Retrieve a list of payout transactions.

        :param dict data:
                    string status: Filter payouts by status
                    string type: Filter payouts by type
                    string currency: Filter payouts by a specific currency
                    string network: Filter payouts by network
                    string address: Filter payouts by the expected address
                    decimal from_amount: Filter payouts with amounts ≥ specified value
                    decimal to_amount: Filter payouts with amounts ≤ specified value
                    integer from_date: Filter payouts with date ≥ specified value
                    integer to_date: Filter payouts with date ≤ specified value
                    string sort_by: Sort by create_date (default)
                    string sort_type: Order by ascending/descending
                    int size: Records per page
                    int page: Page number to retrieve
        :return: JSON response from the API.
        :raises RequestExceptionsBuilder: If the request fails.
        """

        uri = self.version + '/payout'
        return self.request_builder.send_request(uri, params=params , method="GET")
    
    