import requests
import json
from .request_exceptions import RequestExceptionsBuilder


class RequestBuilder:

    def __init__(self, api_key: str = None):
        """
        Initialize the RequestBuilder.

        :param str api_key: The secret API key for authorization.
        :param str api_url: Base API URL (default: 'https://api.oxapay.com/').

        """
        self.api_url = 'https://api.oxapay.com/'
        self.api_key = api_key

    def send_request(self, uri: str, data: dict = None, params: dict = None, method: str = 'post'):
        """
        Send an HTTP request to the API.

        :param str uri: The endpoint to append to the base API URL.
        :param dict data: Request body data (for POST requests).
        :param dict params: Query parameters (for GET requests).
        :param str method: The HTTP method to use ('post', 'get').
        :return: JSON response data if successful.
        :raises RequestExceptionsBuilder: If an error occurs during the request.
        """

        url = self.api_url + uri
        
        headers = {
            'Content-Type': 'application/json',
            'origin': 'oxa-python-package-v-1.0.0'
        }
        
        if 'payment' in uri:
            headers['merchant_api_key'] = self.api_key
        elif 'payout' in uri:
            headers['payout_api_key'] = self.api_key
        elif 'general' in uri:
            headers['general_api_key'] = self.api_key

        try:
            if method.lower() == 'get':
                response = requests.get(url=url, headers=headers, params=params)
            else:
                json_body_data = json.dumps(data)
                response = requests.request(method=method, url=url, headers=headers, data=json_body_data)
        except Exception as e:
            raise RequestExceptionsBuilder('Problems with requests library', response_code=0, uri=uri, errors=[e])

        response_code = response.status_code
        
        try:
            response_json = response.json()
        except Exception as e:
            raise RequestExceptionsBuilder(str(e), response_code=response_code, uri=uri)

        if response_json:
            if response_code != 200 or (response_json['status'] != 200):
                error_message = response_json.get('message', 'Unknown error occurred')
                errors = response_json.get('error', [])
                raise RequestExceptionsBuilder(error_message, response_code, uri, errors=errors)

            if response_json['status'] == 200:
                return response_json['data']
            
        return True