"""
Asynchronous message interaction.
"""

from __future__ import annotations

import pact_ffi
from pact.interaction._base import Interaction


class AsyncMessageInteraction(Interaction):
    """
    An asynchronous message interaction.

    This class defines an asynchronous message interaction between a consumer
    and a provider. It defines the kind of messages a consumer can accept, and
    the is agnostic of the underlying protocol, be it a message queue, Apache
    Kafka, or some other asynchronous protocol.
    """

    def __init__(self, pact_handle: pact_ffi.PactHandle, description: str) -> None:
        """
        Initialise a new Asynchronous Message Interaction.

        This function should not be called directly. Instead, an
        AsyncMessageInteraction should be created using the
        [`upon_receiving`][pact.Pact.upon_receiving] method of a
        [`Pact`][pact.Pact] instance using the `"Async"` interaction type.

        Args:
            pact_handle:
                The Pact instance this interaction belongs to.

            description:
                Description of the interaction. This must be unique within the
                Pact.
        """
        super().__init__(description)
        self.__handle = pact_ffi.new_message_interaction(pact_handle, description)

    @property
    def _handle(self) -> pact_ffi.InteractionHandle:
        """
        Handle for the Interaction.

        This is used internally by the library to pass the Interaction to the
        underlying Pact library.
        """
        return self.__handle

    @property
    def _interaction_part(self) -> pact_ffi.InteractionPart:
        """
        Interaction part.

        Where interactions have multiple parts, this property keeps track
        of which part is currently being set.

        As this is an asynchronous message interaction, this will always
        return a [`REQUEST`][pact_ffi.InteractionPart.REQUEST], as there the
        consumer of the message does not send any responses.
        """
        return pact_ffi.InteractionPart.REQUEST
