import traceback

# Internal package imports
from ._pdf import PDF, FPDF_AVAILABLE, add_doc_entry_pdf
from ._data import PANDAS_CORE_DOCS, PANDAS_PLOT_DOCS, MPL_PYPLOT_DOCS, MPL_AXES_DOCS

# Default config
OUTPUT_FILENAME = "essential_plotting_docs.pdf"


def generate_docs_pdf(output_filename=OUTPUT_FILENAME):
    """Generates the PDF documentation file from embedded data.

    Args:
        output_filename (str, optional): The path where the PDF file will be saved.
                                         Defaults to "essential_plotting_docs.pdf".

    Raises:
        ImportError: If the 'fpdf2' library is not installed/available.
        Exception: Catches and reports other errors during PDF generation.
    """
    if not FPDF_AVAILABLE:
        raise ImportError("Required dependency 'fpdf2' not found. "
                          "Please install it, e.g., using 'pip install fpdf2'.")

    try:
        pdf = PDF(orientation='P', unit='mm', format='A4')
        pdf.set_auto_page_break(auto=True, margin=15)
        pdf.set_margins(left=15, top=15, right=15)
        pdf.add_page()
        pdf.set_font('helvetica', '', 10)

        # --- Write Header --- # Keep simple for library
        # pdf.set_font('helvetica', 'B', 16)
        # pdf.cell(0, 10, "Essential Pandas and Matplotlib Plotting Reference", ln=1, align='C')
        # pdf.set_font('helvetica', '', 10)
        # pdf.cell(0, 6, f"Generated by pandaplotlibdocs library", ln=1, align='C')
        # pdf.cell(0, 6, "Simplified descriptions and examples for common functions.", ln=1, align='C')
        # pdf.ln(10)

        # Write Sections
        sections = [
            ("Pandas Core Functions & Methods", PANDAS_CORE_DOCS, "Common functions and methods for creating, inspecting, selecting, and manipulating Pandas DataFrames and Series."),
            ("Pandas Plotting (`df.plot.*`)", PANDAS_PLOT_DOCS, "These methods are called on a Pandas DataFrame (`df`) or Series (`series`) using the `.plot` accessor. Example: `df.plot.line(...)`"),
            ("Matplotlib (`pyplot`)", MPL_PYPLOT_DOCS, "These are typically imported as `import matplotlib.pyplot as plt` and operate on the 'current' figure or axes."),
            ("Matplotlib (`Axes` object)", MPL_AXES_DOCS, "These methods are called on an `Axes` object (`ax`), usually obtained via `fig, ax = plt.subplots()`. This is the preferred ('object-oriented') way to use Matplotlib.")
        ]

        print("Generating PDF document...") # Add basic progress print
        for title, doc_list, intro_text in sections:
            print(f"  Processing section: {title}...")
            pdf.chapter_title(title)
            pdf.chapter_body(intro_text)
            # Sort entries alphabetically by name within each section
            sorted_docs = sorted(doc_list, key=lambda x: x.get('name', ''))
            for entry in sorted_docs:
                # Simple print for progress, can be removed if too verbose
                # print(f"    - Adding {entry.get('prefix', '')}{entry.get('name', '?')}")
                add_doc_entry_pdf(pdf, entry)
            # Add space or page break before next section
            if title != sections[-1][0]:
                 if pdf.get_y() > pdf.h / 2:
                     pdf.add_page()
                 else:
                     pdf.ln(10)

        # Save PDF
        print(f"\nSaving PDF to {output_filename}...")
        pdf.output(output_filename)
        print("PDF generation complete.")

    except Exception as e:
        print(f"\nAn unexpected error occurred during PDF generation: {e}")
        traceback.print_exc()
        raise 