"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimeTrigger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cron = require("cron-parser");
class TimeTrigger extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.id = id;
        this.props = props;
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     */
    bind(target) {
        const scopedId = this.id + '-' + target.node.id;
        const { second, ...restOfCronSchedule } = this.props.schedule.cron;
        const secondOfCronSchedule = (second === null || second === undefined) ? '0' : second;
        if (secondOfCronSchedule === '0') {
            // In this case, we don't need a trigger queue, a plain old regular EventBridge rule will do the job,
            // Because no sub-minute trigger intervals are required.
            // To be discussed: 
            //   How to handle non-zero values which still boil down to only 1 execution per minute, so every value that is just a single number
            //   It might be better to also just ignore the value in this case, since EventBridge anyway doesn't guarantuee execution on the first second of any minute,
            //   so our queue-based solution anyway doesn't guarantuee absolute execution times, but only relative to the EventBridge execution time.
            // @ts-ignore this local is unused but since this is a CDK construct, its still required since the constructor call has side effects
            const defaultScheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, scopedId + '-trigger-rule', {
                enabled: true,
                targets: [new aws_cdk_lib_1.aws_events_targets.LambdaFunction(target)],
                schedule: aws_cdk_lib_1.aws_events.Schedule.cron(restOfCronSchedule)
            });
            return;
        }
        const cronSchedule = cron.parseExpression(secondOfCronSchedule + ' * * * * *');
        // TODO: Account for the time it takes to invoke the Lambda. Possible solutions that come to mind (not sure which are possible)
        //       - Measure the time the invoke took, substract from next wait
        //       - Invoke without waiting for finishing
        //       - Instead of invoking directly, send message to a queue and trigger the Lambda with that
        const wait = new aws_cdk_lib_1.aws_stepfunctions.Wait(this, scopedId + '-wait', {
            time: aws_cdk_lib_1.aws_stepfunctions.WaitTime.secondsPath('$')
        });
        const invoke = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, scopedId + '-invoke', {
            lambdaFunction: target,
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({})
        });
        const waitThenInvoke = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, scopedId + '-waitinvoke')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.numberGreaterThan('$', 0), wait.next(invoke))
            .otherwise(invoke);
        const seconds = [];
        cronSchedule.fields.second.forEach(s => {
            if (seconds.length === 0 || s !== seconds[seconds.length - 1]) {
                seconds.push(s);
            }
        });
        const createLoopItems = new aws_cdk_lib_1.aws_stepfunctions.Pass(this, scopedId + '-loopitems', {
            result: aws_cdk_lib_1.aws_stepfunctions.Result.fromArray(seconds.map((s, i) => i === 0 ? s : (s - seconds[i - 1])))
        });
        const loop = new aws_cdk_lib_1.aws_stepfunctions.Map(this, scopedId + '-loop', {
            itemsPath: '$',
            maxConcurrency: 1,
        }).iterator(waitThenInvoke);
        const stateMachine = createLoopItems.next(loop);
        const triggerFunction = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, scopedId + '-trigger-function', {
            stateMachineType: aws_cdk_lib_1.aws_stepfunctions.StateMachineType.EXPRESS,
            timeout: aws_cdk_lib_1.Duration.seconds(90),
            definition: stateMachine
        });
        // @ts-ignore this local is unused but since this is a CDK construct, its still required since the constructor call has side effects
        const triggerScheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, scopedId + 'trigger-rule', {
            enabled: true,
            targets: [new aws_cdk_lib_1.aws_events_targets.SfnStateMachine(triggerFunction)],
            schedule: aws_cdk_lib_1.aws_events.Schedule.cron(restOfCronSchedule)
        });
    }
}
exports.TimeTrigger = TimeTrigger;
_a = JSII_RTTI_SYMBOL_1;
TimeTrigger[_a] = { fqn: "paws-lambda-time-trigger.TimeTrigger", version: "1.0.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDJDQUF1QztBQUN2Qyw2Q0FBK0g7QUFDL0gsb0NBQW9DO0FBbUJwQyxNQUFhLFdBQVksU0FBUSxzQkFBUztJQUN0QyxZQUFZLEtBQWdCLEVBQVUsRUFBVSxFQUFVLEtBQXVCO1FBQzdFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFEaUIsT0FBRSxHQUFGLEVBQUUsQ0FBUTtRQUFVLFVBQUssR0FBTCxLQUFLLENBQWtCO0lBRWpGLENBQUM7Ozs7SUFFRCxJQUFJLENBQUMsTUFBNEI7UUFDN0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEVBQUUsR0FBRyxHQUFHLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7UUFFaEQsTUFBTSxFQUFFLE1BQU0sRUFBRSxHQUFHLGtCQUFrQixFQUFFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDO1FBQ25FLE1BQU0sb0JBQW9CLEdBQUcsQ0FBQyxNQUFNLEtBQUssSUFBSSxJQUFJLE1BQU0sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFFdEYsSUFBSSxvQkFBb0IsS0FBSyxHQUFHLEVBQUU7WUFDOUIscUdBQXFHO1lBQ3JHLHdEQUF3RDtZQUV4RCxvQkFBb0I7WUFDcEIsb0lBQW9JO1lBQ3BJLDRKQUE0SjtZQUM1Six5SUFBeUk7WUFFekksb0lBQW9JO1lBQ3BJLE1BQU0sbUJBQW1CLEdBQUcsSUFBSSx3QkFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsUUFBUSxHQUFHLGVBQWUsRUFBRTtnQkFDOUUsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsT0FBTyxFQUFFLENBQUMsSUFBSSxnQ0FBa0IsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ3hELFFBQVEsRUFBRSx3QkFBVSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUM7YUFDekQsQ0FBQyxDQUFDO1lBRUgsT0FBTztTQUNWO1FBRUQsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxvQkFBb0IsR0FBRyxZQUFZLENBQUMsQ0FBQztRQUUvRSwrSEFBK0g7UUFDL0gscUVBQXFFO1FBQ3JFLCtDQUErQztRQUMvQyxpR0FBaUc7UUFFakcsTUFBTSxJQUFJLEdBQUcsSUFBSSwrQkFBaUIsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFFBQVEsR0FBRyxPQUFPLEVBQUU7WUFDOUQsSUFBSSxFQUFFLCtCQUFpQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDO1NBQ3BELENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTSxHQUFHLElBQUkscUNBQXVCLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxRQUFRLEdBQUcsU0FBUyxFQUFFO1lBQ2hGLGNBQWMsRUFBRSxNQUFNO1lBQ3RCLE9BQU8sRUFBRSwrQkFBaUIsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQztTQUN0RCxDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxJQUFJLCtCQUFpQixDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsUUFBUSxHQUFHLGFBQWEsQ0FBQzthQUM5RSxJQUFJLENBQUMsK0JBQWlCLENBQUMsU0FBUyxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO2FBQzlFLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUV2QixNQUFNLE9BQU8sR0FBYSxFQUFFLENBQUM7UUFDN0IsWUFBWSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ25DLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxFQUFFO2dCQUMzRCxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ25CO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDSCxNQUFNLGVBQWUsR0FBRyxJQUFJLCtCQUFpQixDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsUUFBUSxHQUFHLFlBQVksRUFBRTtZQUM5RSxNQUFNLEVBQUUsK0JBQWlCLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxPQUFPLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUN4RyxDQUFDLENBQUM7UUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLCtCQUFpQixDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsUUFBUSxHQUFHLE9BQU8sRUFBRTtZQUM3RCxTQUFTLEVBQUUsR0FBRztZQUNkLGNBQWMsRUFBRSxDQUFDO1NBQ3BCLENBQUMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFNUIsTUFBTSxZQUFZLEdBQUcsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVoRCxNQUFNLGVBQWUsR0FBRyxJQUFJLCtCQUFpQixDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsUUFBUSxHQUFHLG1CQUFtQixFQUFFO1lBQzdGLGdCQUFnQixFQUFFLCtCQUFpQixDQUFDLGdCQUFnQixDQUFDLE9BQU87WUFDNUQsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixVQUFVLEVBQUUsWUFBWTtTQUMzQixDQUFDLENBQUM7UUFFSCxvSUFBb0k7UUFDcEksTUFBTSxtQkFBbUIsR0FBRyxJQUFJLHdCQUFVLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxRQUFRLEdBQUcsY0FBYyxFQUFFO1lBQzdFLE9BQU8sRUFBRSxJQUFJO1lBQ2IsT0FBTyxFQUFFLENBQUMsSUFBSSxnQ0FBa0IsQ0FBQyxlQUFlLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDbEUsUUFBUSxFQUFFLHdCQUFVLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztTQUN6RCxDQUFDLENBQUM7SUFDUCxDQUFDOztBQS9FTCxrQ0FnRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcclxuaW1wb3J0IHsgRHVyYXRpb24sIGF3c19sYW1iZGEsIGF3c19zdGVwZnVuY3Rpb25zLCBhd3Nfc3RlcGZ1bmN0aW9uc190YXNrcywgYXdzX2V2ZW50cywgYXdzX2V2ZW50c190YXJnZXRzIH0gZnJvbSAnYXdzLWNkay1saWInO1xyXG5pbXBvcnQgKiBhcyBjcm9uIGZyb20gJ2Nyb24tcGFyc2VyJztcclxuXHJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXHJcbmV4cG9ydCBpbnRlcmZhY2UgQ3Jvbk9wdGlvbnNXaXRoU2Vjb25kcyBleHRlbmRzIGF3c19ldmVudHMuQ3Jvbk9wdGlvbnMge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcclxuICAgIHJlYWRvbmx5IHNlY29uZD86IHN0cmluZztcclxufVxyXG5cclxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxyXG5leHBvcnQgaW50ZXJmYWNlIFRpbWVUcmlnZ2VyU2NoZWR1bGUge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXHJcbiAgICByZWFkb25seSBjcm9uOiBDcm9uT3B0aW9uc1dpdGhTZWNvbmRzO1xyXG59XHJcblxyXG5leHBvcnQgaW50ZXJmYWNlIFRpbWVUcmlnZ2VyUHJvcHMge1xyXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxyXG4gICAgcmVhZG9ubHkgc2NoZWR1bGU6IFRpbWVUcmlnZ2VyU2NoZWR1bGU7XHJcbn1cclxuXHJcbmV4cG9ydCBjbGFzcyBUaW1lVHJpZ2dlciBleHRlbmRzIENvbnN0cnVjdCBpbXBsZW1lbnRzIGF3c19sYW1iZGEuSUV2ZW50U291cmNlIHtcclxuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIHByaXZhdGUgaWQ6IHN0cmluZywgcHJpdmF0ZSBwcm9wczogVGltZVRyaWdnZXJQcm9wcykge1xyXG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XHJcbiAgICB9XHJcblxyXG4gICAgYmluZCh0YXJnZXQ6IGF3c19sYW1iZGEuSUZ1bmN0aW9uKTogdm9pZCB7XHJcbiAgICAgICAgY29uc3Qgc2NvcGVkSWQgPSB0aGlzLmlkICsgJy0nICsgdGFyZ2V0Lm5vZGUuaWQ7XHJcblxyXG4gICAgICAgIGNvbnN0IHsgc2Vjb25kLCAuLi5yZXN0T2ZDcm9uU2NoZWR1bGUgfSA9IHRoaXMucHJvcHMuc2NoZWR1bGUuY3JvbjtcclxuICAgICAgICBjb25zdCBzZWNvbmRPZkNyb25TY2hlZHVsZSA9IChzZWNvbmQgPT09IG51bGwgfHwgc2Vjb25kID09PSB1bmRlZmluZWQpID8gJzAnIDogc2Vjb25kO1xyXG5cclxuICAgICAgICBpZiAoc2Vjb25kT2ZDcm9uU2NoZWR1bGUgPT09ICcwJykge1xyXG4gICAgICAgICAgICAvLyBJbiB0aGlzIGNhc2UsIHdlIGRvbid0IG5lZWQgYSB0cmlnZ2VyIHF1ZXVlLCBhIHBsYWluIG9sZCByZWd1bGFyIEV2ZW50QnJpZGdlIHJ1bGUgd2lsbCBkbyB0aGUgam9iLFxyXG4gICAgICAgICAgICAvLyBCZWNhdXNlIG5vIHN1Yi1taW51dGUgdHJpZ2dlciBpbnRlcnZhbHMgYXJlIHJlcXVpcmVkLlxyXG5cclxuICAgICAgICAgICAgLy8gVG8gYmUgZGlzY3Vzc2VkOiBcclxuICAgICAgICAgICAgLy8gICBIb3cgdG8gaGFuZGxlIG5vbi16ZXJvIHZhbHVlcyB3aGljaCBzdGlsbCBib2lsIGRvd24gdG8gb25seSAxIGV4ZWN1dGlvbiBwZXIgbWludXRlLCBzbyBldmVyeSB2YWx1ZSB0aGF0IGlzIGp1c3QgYSBzaW5nbGUgbnVtYmVyXHJcbiAgICAgICAgICAgIC8vICAgSXQgbWlnaHQgYmUgYmV0dGVyIHRvIGFsc28ganVzdCBpZ25vcmUgdGhlIHZhbHVlIGluIHRoaXMgY2FzZSwgc2luY2UgRXZlbnRCcmlkZ2UgYW55d2F5IGRvZXNuJ3QgZ3VhcmFudHVlZSBleGVjdXRpb24gb24gdGhlIGZpcnN0IHNlY29uZCBvZiBhbnkgbWludXRlLFxyXG4gICAgICAgICAgICAvLyAgIHNvIG91ciBxdWV1ZS1iYXNlZCBzb2x1dGlvbiBhbnl3YXkgZG9lc24ndCBndWFyYW50dWVlIGFic29sdXRlIGV4ZWN1dGlvbiB0aW1lcywgYnV0IG9ubHkgcmVsYXRpdmUgdG8gdGhlIEV2ZW50QnJpZGdlIGV4ZWN1dGlvbiB0aW1lLlxyXG5cclxuICAgICAgICAgICAgLy8gQHRzLWlnbm9yZSB0aGlzIGxvY2FsIGlzIHVudXNlZCBidXQgc2luY2UgdGhpcyBpcyBhIENESyBjb25zdHJ1Y3QsIGl0cyBzdGlsbCByZXF1aXJlZCBzaW5jZSB0aGUgY29uc3RydWN0b3IgY2FsbCBoYXMgc2lkZSBlZmZlY3RzXHJcbiAgICAgICAgICAgIGNvbnN0IGRlZmF1bHRTY2hlZHVsZVJ1bGUgPSBuZXcgYXdzX2V2ZW50cy5SdWxlKHRoaXMsIHNjb3BlZElkICsgJy10cmlnZ2VyLXJ1bGUnLCB7XHJcbiAgICAgICAgICAgICAgICBlbmFibGVkOiB0cnVlLFxyXG4gICAgICAgICAgICAgICAgdGFyZ2V0czogW25ldyBhd3NfZXZlbnRzX3RhcmdldHMuTGFtYmRhRnVuY3Rpb24odGFyZ2V0KV0sXHJcbiAgICAgICAgICAgICAgICBzY2hlZHVsZTogYXdzX2V2ZW50cy5TY2hlZHVsZS5jcm9uKHJlc3RPZkNyb25TY2hlZHVsZSlcclxuICAgICAgICAgICAgfSk7XHJcblxyXG4gICAgICAgICAgICByZXR1cm47XHJcbiAgICAgICAgfVxyXG5cclxuICAgICAgICBjb25zdCBjcm9uU2NoZWR1bGUgPSBjcm9uLnBhcnNlRXhwcmVzc2lvbihzZWNvbmRPZkNyb25TY2hlZHVsZSArICcgKiAqICogKiAqJyk7XHJcblxyXG4gICAgICAgIC8vIFRPRE86IEFjY291bnQgZm9yIHRoZSB0aW1lIGl0IHRha2VzIHRvIGludm9rZSB0aGUgTGFtYmRhLiBQb3NzaWJsZSBzb2x1dGlvbnMgdGhhdCBjb21lIHRvIG1pbmQgKG5vdCBzdXJlIHdoaWNoIGFyZSBwb3NzaWJsZSlcclxuICAgICAgICAvLyAgICAgICAtIE1lYXN1cmUgdGhlIHRpbWUgdGhlIGludm9rZSB0b29rLCBzdWJzdHJhY3QgZnJvbSBuZXh0IHdhaXRcclxuICAgICAgICAvLyAgICAgICAtIEludm9rZSB3aXRob3V0IHdhaXRpbmcgZm9yIGZpbmlzaGluZ1xyXG4gICAgICAgIC8vICAgICAgIC0gSW5zdGVhZCBvZiBpbnZva2luZyBkaXJlY3RseSwgc2VuZCBtZXNzYWdlIHRvIGEgcXVldWUgYW5kIHRyaWdnZXIgdGhlIExhbWJkYSB3aXRoIHRoYXRcclxuXHJcbiAgICAgICAgY29uc3Qgd2FpdCA9IG5ldyBhd3Nfc3RlcGZ1bmN0aW9ucy5XYWl0KHRoaXMsIHNjb3BlZElkICsgJy13YWl0Jywge1xyXG4gICAgICAgICAgICB0aW1lOiBhd3Nfc3RlcGZ1bmN0aW9ucy5XYWl0VGltZS5zZWNvbmRzUGF0aCgnJCcpXHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIGNvbnN0IGludm9rZSA9IG5ldyBhd3Nfc3RlcGZ1bmN0aW9uc190YXNrcy5MYW1iZGFJbnZva2UodGhpcywgc2NvcGVkSWQgKyAnLWludm9rZScsIHtcclxuICAgICAgICAgICAgbGFtYmRhRnVuY3Rpb246IHRhcmdldCxcclxuICAgICAgICAgICAgcGF5bG9hZDogYXdzX3N0ZXBmdW5jdGlvbnMuVGFza0lucHV0LmZyb21PYmplY3Qoe30pXHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIGNvbnN0IHdhaXRUaGVuSW52b2tlID0gbmV3IGF3c19zdGVwZnVuY3Rpb25zLkNob2ljZSh0aGlzLCBzY29wZWRJZCArICctd2FpdGludm9rZScpXHJcbiAgICAgICAgICAgIC53aGVuKGF3c19zdGVwZnVuY3Rpb25zLkNvbmRpdGlvbi5udW1iZXJHcmVhdGVyVGhhbignJCcsIDApLCB3YWl0Lm5leHQoaW52b2tlKSlcclxuICAgICAgICAgICAgLm90aGVyd2lzZShpbnZva2UpO1xyXG5cclxuICAgICAgICBjb25zdCBzZWNvbmRzOiBudW1iZXJbXSA9IFtdO1xyXG4gICAgICAgIGNyb25TY2hlZHVsZS5maWVsZHMuc2Vjb25kLmZvckVhY2gocyA9PiB7XHJcbiAgICAgICAgICAgIGlmIChzZWNvbmRzLmxlbmd0aCA9PT0gMCB8fCBzICE9PSBzZWNvbmRzW3NlY29uZHMubGVuZ3RoIC0gMV0pIHtcclxuICAgICAgICAgICAgICAgIHNlY29uZHMucHVzaChzKTtcclxuICAgICAgICAgICAgfVxyXG4gICAgICAgIH0pO1xyXG4gICAgICAgIGNvbnN0IGNyZWF0ZUxvb3BJdGVtcyA9IG5ldyBhd3Nfc3RlcGZ1bmN0aW9ucy5QYXNzKHRoaXMsIHNjb3BlZElkICsgJy1sb29waXRlbXMnLCB7XHJcbiAgICAgICAgICAgIHJlc3VsdDogYXdzX3N0ZXBmdW5jdGlvbnMuUmVzdWx0LmZyb21BcnJheShzZWNvbmRzLm1hcCgocywgaSkgPT4gaSA9PT0gMCA/IHMgOiAocyAtIHNlY29uZHNbaSAtIDFdKSkpXHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIGNvbnN0IGxvb3AgPSBuZXcgYXdzX3N0ZXBmdW5jdGlvbnMuTWFwKHRoaXMsIHNjb3BlZElkICsgJy1sb29wJywge1xyXG4gICAgICAgICAgICBpdGVtc1BhdGg6ICckJyxcclxuICAgICAgICAgICAgbWF4Q29uY3VycmVuY3k6IDEsXHJcbiAgICAgICAgfSkuaXRlcmF0b3Iod2FpdFRoZW5JbnZva2UpO1xyXG5cclxuICAgICAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBjcmVhdGVMb29wSXRlbXMubmV4dChsb29wKTtcclxuXHJcbiAgICAgICAgY29uc3QgdHJpZ2dlckZ1bmN0aW9uID0gbmV3IGF3c19zdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZSh0aGlzLCBzY29wZWRJZCArICctdHJpZ2dlci1mdW5jdGlvbicsIHtcclxuICAgICAgICAgICAgc3RhdGVNYWNoaW5lVHlwZTogYXdzX3N0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lVHlwZS5FWFBSRVNTLFxyXG4gICAgICAgICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDkwKSxcclxuICAgICAgICAgICAgZGVmaW5pdGlvbjogc3RhdGVNYWNoaW5lXHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIC8vIEB0cy1pZ25vcmUgdGhpcyBsb2NhbCBpcyB1bnVzZWQgYnV0IHNpbmNlIHRoaXMgaXMgYSBDREsgY29uc3RydWN0LCBpdHMgc3RpbGwgcmVxdWlyZWQgc2luY2UgdGhlIGNvbnN0cnVjdG9yIGNhbGwgaGFzIHNpZGUgZWZmZWN0c1xyXG4gICAgICAgIGNvbnN0IHRyaWdnZXJTY2hlZHVsZVJ1bGUgPSBuZXcgYXdzX2V2ZW50cy5SdWxlKHRoaXMsIHNjb3BlZElkICsgJ3RyaWdnZXItcnVsZScsIHtcclxuICAgICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcclxuICAgICAgICAgICAgdGFyZ2V0czogW25ldyBhd3NfZXZlbnRzX3RhcmdldHMuU2ZuU3RhdGVNYWNoaW5lKHRyaWdnZXJGdW5jdGlvbildLFxyXG4gICAgICAgICAgICBzY2hlZHVsZTogYXdzX2V2ZW50cy5TY2hlZHVsZS5jcm9uKHJlc3RPZkNyb25TY2hlZHVsZSlcclxuICAgICAgICB9KTtcclxuICAgIH1cclxufSJdfQ==