from __future__ import annotations
from dataclasses import dataclass
from typing import Optional, Tuple
from urllib.parse import urlencode
import os, secrets, time

from ._http import HTTP, EU_API, US_API

# ---------- Data ----------
@dataclass
class OAuthToken:
    access_token: str
    token_type: str = "bearer"
    locationid: Optional[int] = None  # 1=US, 2=EU (pCloud convention)

def _auth_host_for(base_api: str) -> str:
    # Explicit mapping, matching official SDKs’ split of auth vs API hosts
    return "https://e.pcloud.com" if base_api == EU_API else "https://u.pcloud.com"

def _api_for_locationid(locationid: Optional[int]) -> str:
    # pCloud commonly returns 2 for EU (eapi), 1 for US (api)
    return EU_API if locationid == 2 else US_API

# ---------- Code flow (with optional redirect_uri) ----------
class PCloudOAuth2Flow:
    """
    OAuth2 Authorization Code flow.

    - EU auth host: https://e.pcloud.com
    - US auth host: https://u.pcloud.com
    - `redirect_uri` is OPTIONAL. If None, the user copies the code shown on the page.
    - After exchange, the returned `locationid` tells you which API host to use.
    """

    def __init__(
        self,
        app_key: str,
        app_secret: str,
        redirect_uri: Optional[str] = None,
        *,
        location: str = "EU",
        http: Optional[HTTP] = None,
    ) -> None:
        self.app_key = app_key
        self.app_secret = app_secret
        self.redirect_uri = redirect_uri
        self.http = http or HTTP(base_url=EU_API if location.upper() == "EU" else US_API)

    def authorize_url(
        self,
        *,
        state: Optional[str] = None,
        force_reapprove: bool = False,
        device_name: Optional[str] = None,
        prompt: Optional[str] = None,
    ) -> str:
        params = {
            "client_id": self.app_key,
            "response_type": "code",
        }
        if self.redirect_uri:
            # must exactly match app settings if provided
            params["redirect_uri"] = self.redirect_uri
        if state:
            params["state"] = state
        if force_reapprove:
            params["force_reapprove"] = 1
        if device_name:
            params["device_name"] = device_name
        if prompt:
            params["prompt"] = prompt

        return f"{_auth_host_for(self.http.base_url)}/oauth2/authorize?{urlencode(params)}"

    # For convenience, keep the old name
    start = authorize_url

    def finish(self, code: str) -> OAuthToken:
        payload = {
            "client_id": self.app_key,
            "client_secret": self.app_secret,
            "code": code,
            "grant_type": "authorization_code",
        }
        if self.redirect_uri:
            payload["redirect_uri"] = self.redirect_uri

        # Try the region we started with first
        data = self.http.request("oauth2_token", payload, http_method="POST")
        token = data.get("access_token")
        locid = data.get("locationid")

        if not token and locid in (1, 2):
            # If server hinted a location, retry there
            alt_http = HTTP(base_url=_api_for_locationid(locid), token=None, timeout=self.http.timeout)
            data = alt_http.request("oauth2_token", payload, http_method="POST")
            token = data.get("access_token")
            locid = data.get("locationid")

        if not token:
            raise RuntimeError("No access_token in OAuth response")

        return OAuthToken(access_token=token, locationid=locid)

# ---------- Poll-token flow (mirrors JS `response_type=poll_token`) ----------
class PCloudOAuth2Poll:
    """
    Poll-token flow used by the JS SDK:
      1) Generate a `request_id`
      2) Open authorize URL with `response_type=poll_token&request_id=...`
      3) Call `oauth2_token` with { client_id, request_id } until you get { access_token, locationid }

    This avoids handling a redirect page and is nice for desktop apps.
    """

    def __init__(self, *, location: str = "EU", http: Optional[HTTP] = None):
        self.http = http or HTTP(base_url=EU_API if location.upper() == "EU" else US_API)

    @staticmethod
    def new_request_id(nbytes: int = 30) -> str:
        # ~40-ish URL-safe chars like the JS SDK uses
        return secrets.token_urlsafe(nbytes)

    def authorize_url(self, client_id: str, request_id: str) -> str:
        q = {"client_id": client_id, "request_id": request_id, "response_type": "poll_token"}
        return f"{_auth_host_for(self.http.base_url)}/oauth2/authorize?{urlencode(q)}"

    # Convenience “blocking” poller (you can also implement your own UI loop)
    def poll_until_token(
        self,
        client_id: str,
        request_id: str,
        *,
        timeout_s: int = 120,
        interval_s: float = 1.5,
    ) -> OAuthToken:
        deadline = time.time() + timeout_s
        last_err: Optional[Exception] = None

        # Try both regions the way the JS SDK does
        endpoints = (EU_API, US_API) if self.http.base_url == EU_API else (US_API, EU_API)

        while time.time() < deadline:
            for base in endpoints:
                try:
                    h = HTTP(base_url=base, token=None, timeout=self.http.timeout)
                    data = h.request("oauth2_token", {"client_id": client_id, "request_id": request_id}, http_method="POST")
                    token = data.get("access_token")
                    if token:
                        return OAuthToken(access_token=token, locationid=data.get("locationid"))
                except Exception as e:
                    last_err = e
            time.sleep(interval_s)

        raise TimeoutError(f"poll_token timeout after {timeout_s}s; last error: {last_err!r}")
