"""
Comprehensive integration tests for pCloud SDK
End-to-end workflow tests, real API interactions, performance benchmarks,
and CLI testing
"""

import os
import subprocess
import sys
import tempfile
import time

import pytest
import responses
from responses import matchers

from pcloud_sdk import PCloudSDK
from pcloud_sdk.exceptions import PCloudException

from .test_config import (
    get_oauth2_credentials,
    get_test_credentials,
    requires_oauth2_credentials,
    requires_real_credentials,
    safe_cleanup_temp_dir,
    skip_if_no_integration_tests,
)


class TestEndToEndWorkflows:
    """End-to-end workflow tests with mocked API responses"""

    def setup_method(self):
        """Setup for each test"""
        self.temp_dir = tempfile.mkdtemp()
        self.token_file = os.path.join(self.temp_dir, "test_credentials.json")

    def teardown_method(self):
        """Cleanup after each test"""
        safe_cleanup_temp_dir(self.temp_dir)

    @responses.activate
    def test_complete_file_management_workflow(self):
        """Test complete file management workflow"""
        # Mock login
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "test_token_123",
                "userid": 12345,
                "email": "test@example.com",
                "quota": 10737418240,
                "usedquota": 1073741824,
            },
            status=200,
        )

        # Mock user info for credential saving
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "email": "test@example.com",
                "userid": 12345,
                "quota": 10737418240,
                "usedquota": 1073741824,
            },
            status=200,
        )

        # Mock file upload
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_create",
            json={"result": 0, "uploadid": 12345},
            status=200,
        )
        responses.add(
            responses.PUT,
            "https://eapi.pcloud.com/upload_write",
            json={"result": 0},
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_save",
            json={
                "result": 0,
                "metadata": [
                    {
                        "fileid": 54321,
                        "name": "test_file.txt",
                        "size": 1024,
                        "isfolder": False,
                    }
                ],
            },
            status=200,
        )

        # Mock file download
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/getfilelink",
            json={
                "result": 0,
                "hosts": ["c123.pcloud.com"],
                "path": "/cBRFZF7ZTKMDlKfpKv5VIQbNVrBJNIZ0/test_file.txt",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://c123.pcloud.com/cBRFZF7ZTKMDlKfpKv5VIQbNVrBJNIZ0/test_file.txt",
            body=b"Test file content for download",
            headers={"content-length": str(len(b"Test file content for download"))},
            status=200,
        )

        # Mock file delete
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/deletefile",
            json={"result": 0, "metadata": {"isdeleted": True}},
            status=200,
        )

        # Initialize SDK and login
        sdk = PCloudSDK(token_file=self.token_file)
        login_info = sdk.login("test@example.com", "password", location_id=2)

        assert login_info["access_token"] == "test_token_123"
        assert sdk.is_authenticated()

        # Create test file for upload
        test_file = os.path.join(self.temp_dir, "test_file.txt")
        test_content = b"Test file content for upload"
        with open(test_file, "wb") as f:
            f.write(test_content)

        # Upload file
        upload_result = sdk.file.upload(test_file, folder_id=0)
        file_id = upload_result["metadata"][0]["fileid"]

        assert file_id == 54321
        assert upload_result["metadata"][0]["name"] == "test_file.txt"

        # Download file
        download_dir = os.path.join(self.temp_dir, "downloads")
        os.makedirs(download_dir)

        download_success = sdk.file.download(file_id, download_dir)
        assert download_success is True

        # Verify downloaded file
        downloaded_file = os.path.join(download_dir, "test_file.txt")
        assert os.path.exists(downloaded_file)

        with open(downloaded_file, "rb") as f:
            downloaded_content = f.read()
        assert downloaded_content == b"Test file content for download"

        # Delete file
        delete_result = sdk.file.delete(file_id)
        assert delete_result is True

        # Logout
        sdk.logout()
        assert not sdk.is_authenticated()

    @responses.activate
    def test_complete_folder_management_workflow(self):
        """Test complete folder management workflow"""
        # Mock login
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "test_token_123",
                "userid": 12345,
                "email": "test@example.com",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={"result": 0, "email": "test@example.com", "userid": 12345},
            status=200,
        )

        # Mock folder creation
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/createfolder",
            json={
                "result": 0,
                "metadata": {
                    "folderid": 11111,
                    "name": "TestFolder",
                    "isfolder": True,
                    "parentfolderid": 0,
                },
            },
            status=200,
        )

        # Mock subfolder creation
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/createfolder",
            json={
                "result": 0,
                "metadata": {
                    "folderid": 22222,
                    "name": "SubFolder",
                    "isfolder": True,
                    "parentfolderid": 11111,
                },
            },
            status=200,
        )

        # Mock folder listing
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/listfolder",
            json={
                "result": 0,
                "metadata": {
                    "folderid": 11111,
                    "name": "TestFolder",
                    "contents": [
                        {"folderid": 22222, "name": "SubFolder", "isfolder": True}
                    ],
                },
            },
            status=200,
        )

        # Mock folder rename
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/renamefolder",
            json={
                "result": 0,
                "metadata": {
                    "folderid": 22222,
                    "name": "RenamedSubFolder",
                    "isfolder": True,
                },
            },
            status=200,
        )

        # Mock folder delete
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/deletefolderrecursive",
            json={
                "result": 0,
                "metadata": {"isdeleted": True, "deletedfiles": 0, "deletedfolders": 2},
            },
            status=200,
        )

        # Initialize SDK
        sdk = PCloudSDK(token_file=self.token_file)
        sdk.login("test@example.com", "password", location_id=2)

        # Create main folder
        main_folder_id = sdk.folder.create("TestFolder", parent=0)
        assert main_folder_id == 11111

        # Create subfolder
        sub_folder_id = sdk.folder.create("SubFolder", parent=main_folder_id)
        assert sub_folder_id == 22222

        # List folder contents
        contents = sdk.folder.get_content(folder_id=main_folder_id)
        assert len(contents) == 1
        assert contents[0]["name"] == "SubFolder"

        # Rename subfolder
        renamed_id = sdk.folder.rename(sub_folder_id, "RenamedSubFolder")
        assert renamed_id == sub_folder_id

        # Delete entire folder structure
        delete_result = sdk.folder.delete_recursive(main_folder_id)
        assert delete_result["metadata"]["isdeleted"] is True
        assert delete_result["metadata"]["deletedfolders"] == 2

    @responses.activate
    def test_oauth2_to_file_operations_workflow(self):
        """Test OAuth2 authentication followed by file operations"""
        # Mock OAuth2 token exchange
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/oauth2_token",
            json={"result": 0, "access_token": "oauth2_token_456", "locationid": 2},
            status=200,
        )

        # Mock user info
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "email": "oauth@example.com",
                "userid": 67890,
                "quota": 10737418240,
                "usedquota": 2147483648,
            },
            status=200,
        )

        # Mock file operations
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_create",
            json={"result": 0, "uploadid": 99999},
            status=200,
        )
        responses.add(
            responses.PUT,
            "https://eapi.pcloud.com/upload_write",
            json={"result": 0},
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_save",
            json={
                "result": 0,
                "metadata": [{"fileid": 88888, "name": "oauth_file.txt", "size": 512}],
            },
            status=200,
        )

        # Initialize SDK with OAuth2
        sdk = PCloudSDK(
            app_key="test_client_id",
            app_secret="test_client_secret",
            auth_type="oauth2",
            token_file=self.token_file,
        )

        # Get authorization URL
        auth_url = sdk.get_auth_url("http://localhost:8080/callback")
        assert "https://my.pcloud.com/oauth2/authorize" in auth_url
        assert "test_client_id" in auth_url

        # Exchange code for token
        token_info = sdk.authenticate("oauth_code_123", location_id=2)
        assert token_info["access_token"] == "oauth2_token_456"

        # Verify user info
        user_email = sdk.user.get_user_email()
        assert user_email == "oauth@example.com"

        # Perform file operation
        test_file = os.path.join(self.temp_dir, "oauth_file.txt")
        with open(test_file, "wb") as f:
            f.write(b"OAuth2 file content")

        upload_result = sdk.file.upload(test_file, folder_id=0)
        assert upload_result["metadata"][0]["fileid"] == 88888

    @responses.activate
    def test_multi_account_workflow(self):
        """Test workflow with multiple accounts"""
        account1_file = os.path.join(self.temp_dir, "account1.json")
        account2_file = os.path.join(self.temp_dir, "account2.json")

        # Mock responses for account 1
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "token1_123",
                "userid": 11111,
                "email": "user1@example.com",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={"result": 0, "email": "user1@example.com", "userid": 11111},
            status=200,
        )

        # Mock responses for account 2
        responses.add(
            responses.GET,
            "https://api.pcloud.com/userinfo",  # Changed for US server
            json={
                "result": 0,
                "auth": "token2_456",
                "userid": 22222,
                "email": "user2@example.com",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://api.pcloud.com/userinfo",  # Changed for US server
            json={"result": 0, "email": "user2@example.com", "userid": 22222},
            status=200,
        )

        # Initialize SDK for account 1 (EU server)
        sdk1 = PCloudSDK(token_file=account1_file)
        login1 = sdk1.login("user1@example.com", "password1", location_id=2)

        assert login1["access_token"] == "token1_123"
        assert sdk1.user.get_user_email() == "user1@example.com"

        # Initialize SDK for account 2 (US server)
        sdk2 = PCloudSDK(token_file=account2_file)
        login2 = sdk2.login("user2@example.com", "password2", location_id=1)

        assert login2["access_token"] == "token2_456"
        assert sdk2.user.get_user_email() == "user2@example.com"

        # Verify isolation
        assert sdk1.get_saved_email() == "user1@example.com"
        assert sdk2.get_saved_email() == "user2@example.com"
        assert sdk1.app.get_access_token() != sdk2.app.get_access_token()
        assert sdk1.app.get_location_id() != sdk2.app.get_location_id()

        # Verify credentials are saved separately
        assert os.path.exists(account1_file)
        assert os.path.exists(account2_file)

    @responses.activate
    def test_error_recovery_workflow(self):
        """Test workflow with various error conditions and recovery"""
        sdk = PCloudSDK(token_file=self.token_file)

        # First, mock a failed login
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={"result": 2000, "error": "Invalid username or password"},
            status=200,
        )

        # Login should fail
        with pytest.raises(PCloudException, match="Invalid email or password"):
            sdk.login("wrong@example.com", "wrong_password", location_id=2)

        # Now mock successful login
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "recovery_token_123",
                "userid": 99999,
                "email": "correct@example.com",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={"result": 0, "email": "correct@example.com", "userid": 99999},
            status=200,
        )

        # Successful login after failed attempt
        login_info = sdk.login("correct@example.com", "correct_password", location_id=2)
        assert login_info["access_token"] == "recovery_token_123"

        # Mock file upload failure followed by success
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_create",
            json={"result": 5000, "error": "Upload session creation failed"},
            status=200,
        )

        test_file = os.path.join(self.temp_dir, "test_recovery.txt")
        with open(test_file, "wb") as f:
            f.write(b"Recovery test content")

        # Upload should fail initially
        with pytest.raises(
            PCloudException, match="Erreur lors de la création de la session d'upload"
        ):
            sdk.file.upload(test_file, folder_id=0)

        # Mock successful upload after recovery
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_create",
            json={"result": 0, "uploadid": 77777},
            status=200,
        )
        responses.add(
            responses.PUT,
            "https://eapi.pcloud.com/upload_write",
            json={"result": 0},
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_save",
            json={
                "result": 0,
                "metadata": [
                    {"fileid": 66666, "name": "test_recovery.txt", "size": 20}
                ],
            },
            status=200,
        )

        # Upload should succeed after retry
        upload_result = sdk.file.upload(test_file, folder_id=0)
        assert upload_result["metadata"][0]["fileid"] == 66666


class TestProgressTrackingIntegration:
    """Integration tests for progress tracking with file operations"""

    def setup_method(self):
        """Setup for each test"""
        self.temp_dir = tempfile.mkdtemp()
        self.token_file = os.path.join(self.temp_dir, "test_credentials.json")

    def teardown_method(self):
        """Cleanup after each test"""
        safe_cleanup_temp_dir(self.temp_dir)

    @responses.activate
    def test_upload_with_progress_tracking(self):
        """Test file upload with progress tracking integration"""
        # Mock login and file operations
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "test_token",
                "userid": 12345,
                "email": "test@example.com",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={"result": 0, "email": "test@example.com", "userid": 12345},
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_create",
            json={"result": 0, "uploadid": 12345},
            status=200,
        )
        responses.add(
            responses.PUT,
            "https://eapi.pcloud.com/upload_write",
            json={"result": 0},
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/upload_save",
            json={
                "result": 0,
                "metadata": [{"fileid": 54321, "name": "progress_test.txt"}],
            },
            status=200,
        )

        # Track progress calls
        progress_calls = []

        def track_progress(bytes_transferred, total_bytes, percentage, speed, **kwargs):
            progress_calls.append(
                {
                    "bytes": bytes_transferred,
                    "total": total_bytes,
                    "percent": percentage,
                    "speed": speed,
                    "operation": kwargs.get("operation"),
                    "filename": kwargs.get("filename"),
                    "status": kwargs.get("status"),
                }
            )

        # Create test file
        test_file = os.path.join(self.temp_dir, "progress_test.txt")
        test_content = b"Progress tracking test content"
        with open(test_file, "wb") as f:
            f.write(test_content)

        # Upload with progress tracking
        sdk = PCloudSDK(token_file=self.token_file)
        sdk.login("test@example.com", "password", location_id=2)

        upload_result = sdk.file.upload(
            test_file, folder_id=0, progress_callback=track_progress
        )

        # Verify upload succeeded
        assert upload_result["metadata"][0]["fileid"] == 54321

        # Verify progress was tracked
        assert len(progress_calls) >= 3  # At least starting, progress, completed
        assert progress_calls[0]["status"] == "starting"
        assert progress_calls[-1]["status"] == "completed"
        assert all(call["operation"] == "upload" for call in progress_calls)
        assert all(call["filename"] == "progress_test.txt" for call in progress_calls)

    @responses.activate
    def test_download_with_progress_tracking(self):
        """Test file download with progress tracking integration"""
        # Mock login and file operations
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "test_token",
                "userid": 12345,
                "email": "test@example.com",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={"result": 0, "email": "test@example.com", "userid": 12345},
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/getfilelink",
            json={
                "result": 0,
                "hosts": ["c123.pcloud.com"],
                "path": "/path/to/download_test.txt",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://c123.pcloud.com/path/to/download_test.txt",
            body=b"Downloaded content with progress tracking",
            status=200,
            stream=True,
        )

        # Track progress calls
        progress_calls = []

        def track_progress(bytes_transferred, total_bytes, percentage, speed, **kwargs):
            progress_calls.append(
                {
                    "bytes": bytes_transferred,
                    "total": total_bytes,
                    "percent": percentage,
                    "operation": kwargs.get("operation"),
                    "status": kwargs.get("status"),
                }
            )

        # Download with progress tracking
        sdk = PCloudSDK(token_file=self.token_file)
        sdk.login("test@example.com", "password", location_id=2)

        download_dir = os.path.join(self.temp_dir, "downloads")
        os.makedirs(download_dir)

        success = sdk.file.download(
            12345, download_dir, progress_callback=track_progress
        )

        # Verify download succeeded
        assert success is True

        # Verify progress was tracked
        assert len(progress_calls) >= 2  # At least starting and completed
        assert progress_calls[0]["status"] == "starting"
        assert progress_calls[-1]["status"] == "completed"
        assert all(call["operation"] == "download" for call in progress_calls)

    @responses.activate
    def test_multiple_operations_with_shared_progress(self):
        """Test multiple operations sharing the same progress tracker"""
        from pcloud_sdk.progress_utils import DetailedProgress

        # Mock responses for multiple operations
        self._mock_login_responses()
        self._mock_file_operations()

        # Shared progress tracker
        shared_progress = DetailedProgress()

        # Create test files
        files = []
        for i in range(3):
            test_file = os.path.join(self.temp_dir, f"multi_test_{i}.txt")
            with open(test_file, "wb") as f:
                f.write(f"Content for file {i}".encode())
            files.append(test_file)

        # Perform multiple uploads with shared progress tracker
        sdk = PCloudSDK(token_file=self.token_file)
        sdk.login("test@example.com", "password", location_id=2)

        upload_results = []
        for test_file in files:
            result = sdk.file.upload(
                test_file, folder_id=0, progress_callback=shared_progress
            )
            upload_results.append(result)

        # Verify all uploads succeeded
        assert len(upload_results) == 3

        # Verify progress tracker received calls from all operations
        assert len(shared_progress.checkpoints) >= 9  # At least 3 calls per upload

    def _mock_login_responses(self):
        """Helper to mock standard login responses"""
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "test_token",
                "userid": 12345,
                "email": "test@example.com",
            },
            status=200,
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={"result": 0, "email": "test@example.com", "userid": 12345},
            status=200,
        )

    def _mock_file_operations(self):
        """Helper to mock file operation responses"""
        # Mock multiple upload operations
        for i in range(5):  # More than needed for flexibility
            responses.add(
                responses.GET,
                "https://eapi.pcloud.com/upload_create",
                json={"result": 0, "uploadid": 10000 + i},
                status=200,
            )
            responses.add(
                responses.PUT,
                "https://eapi.pcloud.com/upload_write",
                json={"result": 0},
                status=200,
            )
            responses.add(
                responses.GET,
                "https://eapi.pcloud.com/upload_save",
                json={
                    "result": 0,
                    "metadata": [{"fileid": 20000 + i, "name": f"multi_test_{i}.txt"}],
                },
                status=200,
            )


class TestPerformanceBenchmarks:
    """Performance benchmark tests for SDK operations"""

    def setup_method(self):
        """Setup for each test"""
        self.temp_dir = tempfile.mkdtemp()
        self.token_file = os.path.join(self.temp_dir, "test_credentials.json")

    def teardown_method(self):
        """Cleanup after each test"""
        safe_cleanup_temp_dir(self.temp_dir)

    @pytest.mark.performance
    @responses.activate
    def test_sdk_initialization_performance(self):
        """Benchmark SDK initialization time"""
        # Mock minimal responses
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "test_token",
                "userid": 12345,
                "email": "test@example.com",
            },
            status=200,
        )

        start_time = time.time()

        # Initialize SDK multiple times
        for i in range(100):
            PCloudSDK(token_file=f"{self.token_file}_{i}")

        end_time = time.time()
        elapsed = end_time - start_time

        # Should be fast (under 1 second for 100 initializations)
        assert elapsed < 1.0
        print(
            f"SDK initialization: {elapsed:.3f}s for 100 instances "
            f"({elapsed*10:.1f}ms per instance)"
        )

    @pytest.mark.performance
    def test_progress_callback_performance(self):
        """Benchmark progress callback performance"""
        from pcloud_sdk.progress_utils import SilentProgress, SimpleProgressBar

        temp_log = os.path.join(self.temp_dir, "perf_test.log")

        # Test different progress trackers
        trackers = {"simple": SimpleProgressBar(), "silent": SilentProgress(temp_log)}

        for name, tracker in trackers.items():
            start_time = time.time()

            # Simulate 1000 progress updates
            for i in range(1000):
                tracker(
                    i,
                    1000,
                    i / 10.0,
                    1024.0,
                    filename="perf_test.txt",
                    operation="upload",
                )

            end_time = time.time()
            elapsed = end_time - start_time

            # Should complete quickly
            assert elapsed < 2.0
            print(
                f"{name} progress tracker: {elapsed:.3f}s for 1000 calls ({elapsed*1000:.1f}¼s per call)"
            )

    @pytest.mark.performance
    @responses.activate
    def test_token_loading_performance(self):
        """Benchmark token loading performance"""
        # Create test credentials
        test_credentials = {
            "email": "test@example.com",
            "access_token": "test_token_123",
            "location_id": 2,
            "auth_type": "direct",
            "user_info": {"userid": 12345},
            "saved_at": time.time(),
        }

        import json

        with open(self.token_file, "w") as f:
            json.dump(test_credentials, f)

        start_time = time.time()

        # Load credentials multiple times
        for i in range(100):
            sdk = PCloudSDK(token_file=self.token_file)
            sdk._load_saved_credentials()

        end_time = time.time()
        elapsed = end_time - start_time

        # Should be fast
        assert elapsed < 1.0
        print(
            f"Token loading: {elapsed:.3f}s for 100 loads ({elapsed*10:.1f}ms per load)"
        )

    @pytest.mark.performance
    def test_memory_usage_pattern(self):
        """Test memory usage patterns during operations"""
        import gc

        # Force garbage collection
        gc.collect()

        # Get initial memory usage (approximation)
        initial_objects = len(gc.get_objects())

        # Create multiple SDK instances
        sdks = []
        for i in range(50):
            sdk = PCloudSDK(token_file=f"{self.token_file}_{i}", token_manager=False)
            sdks.append(sdk)

        # Check memory growth
        mid_objects = len(gc.get_objects())

        # Clear references
        sdks.clear()
        gc.collect()

        # Check memory after cleanup
        final_objects = len(gc.get_objects())

        print(
            f"Memory usage: Initial={initial_objects}, Peak={mid_objects}, Final={final_objects}"
        )

        # Memory should not grow excessively
        growth = mid_objects - initial_objects
        cleanup = mid_objects - final_objects

        # Should clean up most of the allocated objects
        assert cleanup / growth > 0.5  # At least 50% cleanup


class TestCLIIntegration:
    """Tests for CLI functionality integration"""

    def setup_method(self):
        """Setup for each test"""
        self.temp_dir = tempfile.mkdtemp()

    def teardown_method(self):
        """Cleanup after each test"""
        safe_cleanup_temp_dir(self.temp_dir)

    def test_cli_module_import(self):
        """Test that CLI module can be imported"""
        try:
            from pcloud_sdk import cli

            assert hasattr(cli, "main") or hasattr(cli, "app") or callable(cli)
        except ImportError:
            # CLI might not be implemented yet
            pytest.skip("CLI module not implemented")

    def test_cli_help_command(self):
        """Test CLI help functionality"""
        try:
            # Try to run CLI help command
            result = subprocess.run(
                [sys.executable, "-m", "pcloud_sdk.cli", "--help"],
                capture_output=True,
                text=True,
                timeout=10,
            )

            # Should not crash and should show help
            assert result.returncode in [0, 2]  # 0 for success, 2 for argparse help
            assert "usage" in result.stdout.lower() or "usage" in result.stderr.lower()

        except (subprocess.TimeoutExpired, FileNotFoundError):
            pytest.skip("CLI not available or not responding")

    def test_cli_basic_commands(self):
        """Test basic CLI commands"""
        try:
            # Test version command
            result = subprocess.run(
                [sys.executable, "-m", "pcloud_sdk.cli", "--version"],
                capture_output=True,
                text=True,
                timeout=10,
            )

            if result.returncode == 0:
                assert "2.0" in result.stdout or "2.0" in result.stderr
            else:
                pytest.skip("Version command not implemented")

        except (subprocess.TimeoutExpired, FileNotFoundError):
            pytest.skip("CLI not available")

    def test_cli_configuration(self):
        """Test CLI configuration handling"""
        # This test would verify CLI can handle configuration files
        # Implementation depends on actual CLI design
        pytest.skip("CLI configuration tests require actual CLI implementation")


class TestRobustnessAndReliability:
    """Tests for robustness and reliability under various conditions"""

    def setup_method(self):
        """Setup for each test"""
        self.temp_dir = tempfile.mkdtemp()
        self.token_file = os.path.join(self.temp_dir, "test_credentials.json")

    def teardown_method(self):
        """Cleanup after each test"""
        safe_cleanup_temp_dir(self.temp_dir)

    @responses.activate
    def test_network_interruption_handling(self):
        """Test handling of network interruptions"""
        # Mock intermittent network failures
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            body=Exception("Network timeout"),
            headers={"content-length": "0"},
        )
        responses.add(
            responses.GET,
            "https://eapi.pcloud.com/userinfo",
            json={
                "result": 0,
                "auth": "recovered_token",
                "userid": 12345,
                "email": "test@example.com",
            },
            status=200,
        )

        sdk = PCloudSDK(token_file=self.token_file)

        # First attempt should fail
        with pytest.raises(Exception):
            sdk.login("test@example.com", "password", location_id=2)

        # Second attempt should succeed (simulating network recovery)
        # Note: In practice, this would require retry logic in the SDK
        try:
            login_info = sdk.login("test@example.com", "password", location_id=2)
            assert login_info["access_token"] == "recovered_token"
        except Exception:
            # If retry logic isn't implemented, that's expected
            pass

    # @responses.activate # Removed decorator, will use RequestsMock context manager in thread
    @pytest.mark.skip(
        reason="Complex interaction with responses in threads; network calls not consistently mocked. Needs further investigation."
    )
    def test_concurrent_operations(self):
        """Test concurrent operations from multiple threads"""
        import threading

        num_workers = 5
        results = []
        errors = []

        def worker(worker_id):
            with responses.RequestsMock() as rsps:
                # Mock for the login userinfo call (getauth=1)
                rsps.add(
                    responses.GET,
                    "https://eapi.pcloud.com/userinfo",  # Assuming EU server for location_id=2
                    match=[
                        matchers.query_param_matcher(
                            {
                                "getauth": "1",
                                "username": f"user{worker_id}@example.com",
                                "logout": "1",
                            }
                        )
                    ],
                    json={
                        "result": 0,
                        "auth": f"token_{worker_id}",
                        "userid": 10000 + worker_id,
                        "email": f"user{worker_id}@example.com",
                    },
                    status=200,
                )
                # Note: The second userinfo call for _save_credentials is not mocked
                # because token_manager=False is used in the SDK instance.

                try:
                    token_file = os.path.join(
                        self.temp_dir, f"concurrent_{worker_id}.json"
                    )
                    # Disable token manager to simplify
                    sdk = PCloudSDK(
                        token_file=token_file, token_manager=False, location_id=2
                    )
                    login_info = sdk.login(
                        f"user{worker_id}@example.com", "password", location_id=2
                    )
                    results.append(login_info["access_token"])
                except Exception as e:
                    errors.append(e)

        threads = []
        for i in range(num_workers):
            thread = threading.Thread(target=worker, args=(i,))
            threads.append(thread)
            thread.start()

        for thread in threads:
            thread.join()

        if errors:
            print(f"DEBUG: Errors encountered in worker threads: {errors}")

        assert len(errors) == 0, f"Errors occurred in worker threads: {errors}"
        assert (
            len(results) == num_workers
        ), f"Expected {num_workers} results, got {len(results)}"
        assert (
            len(set(results)) == num_workers
        ), f"Expected {num_workers} unique tokens, got {len(set(results))}"

    def test_resource_cleanup(self):
        """Test proper resource cleanup"""
        # Create multiple SDK instances
        sdks = []
        for i in range(10):
            token_file = os.path.join(self.temp_dir, f"cleanup_test_{i}.json")
            sdk = PCloudSDK(token_file=token_file, token_manager=False)
            sdks.append(sdk)

        # Clear references and force garbage collection
        del sdks
        import gc

        gc.collect()

        # Verify no credential files were created (token_manager=False)
        for i in range(10):
            token_file = os.path.join(self.temp_dir, f"cleanup_test_{i}.json")
            assert not os.path.exists(token_file)

    @responses.activate
    def test_malformed_response_handling(self):
        """Test handling of malformed API responses"""
        malformed_responses = [
            # Missing result field
            {"auth": "token123", "userid": 12345},
            # Invalid JSON structure
            {"result": "not_a_number", "error": "Invalid"},
            # Completely wrong format
            "This is not JSON at all",
            # Empty response
            "",
            # Null response
            None,
        ]

        sdk = PCloudSDK(token_file=self.token_file)

        for malformed_response in malformed_responses:
            responses.reset()

            if isinstance(malformed_response, str) and malformed_response != "":
                responses.add(
                    responses.GET,
                    "https://eapi.pcloud.com/userinfo",
                    body=malformed_response,
                    headers={"content-length": str(len(malformed_response))},
                    status=200,
                )
            elif malformed_response == "":
                responses.add(
                    responses.GET,
                    "https://eapi.pcloud.com/userinfo",
                    body="",
                    headers={"content-length": "0"},
                    status=200,
                )
            elif malformed_response is None:
                responses.add(
                    responses.GET,
                    "https://eapi.pcloud.com/userinfo",
                    body="",
                    headers={"content-length": "0"},
                    status=200,
                )
            else:
                responses.add(
                    responses.GET,
                    "https://eapi.pcloud.com/userinfo",
                    json=malformed_response,
                    status=200,
                )

            # Should handle gracefully without crashing
            try:
                sdk.login("test@example.com", "password", location_id=2)
            except Exception as e:
                # Expect specific pCloud exceptions, not generic crashes
                assert isinstance(e, (PCloudException, ValueError, TypeError))


@pytest.mark.integration
@pytest.mark.real_api
class TestRealAPIIntegration:
    """Integration tests with real pCloud API - require credentials"""

    @requires_real_credentials
    @skip_if_no_integration_tests
    def test_real_complete_workflow(self):
        """Test complete workflow with real pCloud API"""
        creds = get_test_credentials()

        temp_dir = tempfile.mkdtemp()
        token_file = os.path.join(temp_dir, "real_integration_test.json")

        try:
            # Initialize SDK
            sdk = PCloudSDK(token_file=token_file)

            # Login
            login_info = sdk.login(
                creds["email"], creds["password"], location_id=creds["location_id"]
            )
            assert "access_token" in login_info
            assert sdk.is_authenticated()

            # Get user info
            user_email = sdk.user.get_user_email()
            assert user_email == creds["email"]

            # Create test folder
            test_folder_id = sdk.folder.create("SDK_Integration_Test", parent=0)
            assert isinstance(test_folder_id, int)

            # Create test file
            test_file = os.path.join(temp_dir, "integration_test.txt")
            test_content = b"Real integration test content"
            with open(test_file, "wb") as f:
                f.write(test_content)

            # Upload file to test folder
            upload_result = sdk.file.upload(test_file, folder_id=test_folder_id)
            # Real API returns metadata as object, not array
            file_id = upload_result["metadata"]["fileid"]

            # Download file
            download_dir = os.path.join(temp_dir, "downloads")
            os.makedirs(download_dir)
            download_success = sdk.file.download(file_id, download_dir)
            assert download_success is True

            # Verify downloaded content
            downloaded_file = os.path.join(download_dir, "integration_test.txt")
            with open(downloaded_file, "rb") as f:
                downloaded_content = f.read()
            assert downloaded_content == test_content

            # Clean up
            sdk.file.delete(file_id)
            sdk.folder.delete(test_folder_id)
            sdk.logout()

        finally:
            # Clean up local files
            safe_cleanup_temp_dir(temp_dir)

    @requires_oauth2_credentials
    @skip_if_no_integration_tests
    def test_real_oauth2_workflow(self):
        """Test OAuth2 workflow with real pCloud API"""
        oauth_creds = get_oauth2_credentials()

        temp_dir = tempfile.mkdtemp()
        token_file = os.path.join(temp_dir, "oauth2_integration_test.json")

        try:
            # Initialize SDK with OAuth2
            sdk = PCloudSDK(
                app_key=oauth_creds["client_id"],
                app_secret=oauth_creds["client_secret"],
                auth_type="oauth2",
                token_file=token_file,
            )

            # Get authorization URL
            auth_url = sdk.get_auth_url("http://localhost:8080/callback")
            print(f"Authorization URL: {auth_url}")

            # Note: In real test, user would need to visit URL and get code
            # This test would require manual intervention or browser automation
            pytest.skip("OAuth2 test requires manual authorization step")

        finally:
            safe_cleanup_temp_dir(temp_dir)

    @requires_real_credentials
    @skip_if_no_integration_tests
    def test_real_large_file_operations(self):
        """Test large file operations with real API"""
        creds = get_test_credentials()

        temp_dir = tempfile.mkdtemp()

        try:
            # Create large test file (100MB)
            large_file = os.path.join(temp_dir, "large_test_file.dat")
            with open(large_file, "wb") as f:
                chunk = b"A" * (1024 * 1024)  # 1MB chunk
                for _ in range(100):  # 100MB total
                    f.write(chunk)

            # Initialize SDK
            sdk = PCloudSDK()
            sdk.login(
                creds["email"], creds["password"], location_id=creds["location_id"]
            )

            # Upload with progress tracking
            progress_calls = []

            def track_progress(*args, **kwargs):
                # percentage is the 3rd positional argument (index 2)
                if args and len(args) > 2:
                    progress_calls.append(args[2])
                else:
                    progress_calls.append(
                        0
                    )  # Should not happen if callback is called correctly

            upload_result = sdk.file.upload(
                large_file, folder_id=0, progress_callback=track_progress
            )
            # Real API returns metadata as object, not array
            file_id = upload_result["metadata"]["fileid"]

            # Verify progress was tracked
            assert len(progress_calls) > 10  # Should have many progress updates
            assert max(progress_calls) >= 90  # Should reach high percentage

            # Download large file
            download_dir = os.path.join(temp_dir, "downloads")
            os.makedirs(download_dir)

            download_success = sdk.file.download(
                file_id, download_dir, progress_callback=track_progress
            )
            assert download_success is True

            # Verify file size
            downloaded_file = os.path.join(download_dir, "large_test_file.dat")
            assert os.path.getsize(downloaded_file) == 100 * 1024 * 1024

            # Clean up remote file
            sdk.file.delete(file_id)

        finally:
            safe_cleanup_temp_dir(temp_dir)
