"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2023

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This module is intended to distributed as part of automatically generated code by the
peakrdl-python tool. It provides a set of classes used by the autogenerated code to represent
asynchronous registers and fields
"""
from enum import Enum
from typing import List, Union, Iterator, Tuple, Optional, Dict, TypeVar, cast
from typing import AsyncGenerator
from abc import ABC, abstractmethod
from contextlib import asynccontextmanager
from array import array as Array
import sys
from warnings import warn

from .utility_functions import get_array_typecode, swap_msb_lsb_ordering
from .base import AsyncAddressMap, AsyncRegFile
from .async_memory import  MemoryAsyncReadOnly, MemoryAsyncWriteOnly, MemoryAsyncReadWrite, \
    AsyncMemory, ReadableAsyncMemory, WritableAsyncMemory
from .async_memory import  MemoryAsyncReadOnlyLegacy, MemoryAsyncWriteOnlyLegacy, \
    MemoryAsyncReadWriteLegacy
from .async_memory import ReadableAsyncMemoryLegacy, WritableAsyncMemoryLegacy
from .callbacks import AsyncCallbackSet, AsyncCallbackSetLegacy
from .base_register import BaseReg, BaseRegArray, RegisterWriteVerifyError
from .base_field import FieldEnum, FieldSizeProps, FieldMiscProps, \
    _FieldReadOnlyFramework, _FieldWriteOnlyFramework

# pylint: disable=duplicate-code
if sys.version_info >= (3, 11):
    from typing import Self
else:
    from typing_extensions import Self
# pylint: enable=duplicate-code

# pylint: disable=duplicate-code
if sys.version_info >= (3, 10):
    # type guarding was introduced in python 3.10
    from typing import TypeGuard
else:
    from typing_extensions import TypeGuard
# pylint: enable=duplicate-code

# pylint: disable=redefined-slots-in-subclass,too-many-lines


class AsyncReg(BaseReg, ABC):
    """
        base class of async register wrappers

        Note:
            It is not expected that this class will be instantiated under normal
            circumstances however, it is useful for type checking
        """

    __slots__: List[str] = []

    # pylint: disable=too-many-arguments,duplicate-code
    def __init__(self, *,
                 address: int,
                 width: int,
                 accesswidth: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Union[AsyncAddressMap, AsyncRegFile, AsyncMemory, 'AsyncRegArray']):

        if not isinstance(parent, (AsyncAddressMap, AsyncRegFile,
                                   MemoryAsyncReadOnly, MemoryAsyncWriteOnly,
                                   MemoryAsyncReadWrite, AsyncRegArray, MemoryAsyncReadOnlyLegacy,
                                   MemoryAsyncWriteOnlyLegacy,
                                   MemoryAsyncReadWriteLegacy)):
            raise TypeError(f'bad parent type got: {type(parent)}')

        if not isinstance(parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            raise TypeError(f'callback set type is wrong, got {type(parent._callbacks)}')

        super().__init__(address=address, width=width, accesswidth=accesswidth,
                         logger_handle=logger_handle, inst_name=inst_name, parent=parent)

    @property
    def _callbacks(self) -> Union[AsyncCallbackSet, AsyncCallbackSetLegacy]:
        # pylint: disable=protected-access
        if self.parent is None:
            raise RuntimeError('Parent must be set')

        if isinstance(self.parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            return self.parent._callbacks

        raise TypeError(f'unhandled parent callback type: {type(self.parent._callbacks)}')

    @property
    @abstractmethod
    def fields(self) -> \
            Iterator[Union['FieldAsyncReadOnly', 'FieldAsyncWriteOnly', 'FieldAsyncReadWrite']]:
        """
        generator that produces has all the fields within the register
        """


class RegAsyncReadOnly(AsyncReg, ABC):
    """
    class for an async read only register

    Args:
        callbacks: set of callback to be used for accessing the hardware or simulator
        address: address of the register
        width: width of the register in bits
        accesswidth: minimum access width of the register in bits
        logger_handle: name to be used logging messages associate with this
            object

    """

    __slots__: List[str] = ['__in_context_manager', '__register_state']

    # pylint: disable=too-many-arguments, duplicate-code
    def __init__(self, *,
                 address: int,
                 width: int,
                 accesswidth: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Union[AsyncAddressMap, AsyncRegFile, ReadableAsyncMemory,
                               ReadableAsyncMemoryLegacy]):

        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         parent=parent, width=width, accesswidth=accesswidth)

        self.__in_context_manager: bool = False
        self.__register_state: int = 0
    # pylint: enable=too-many-arguments, duplicate-code

    @asynccontextmanager
    async def single_read(self) -> AsyncGenerator[Self, None]:
        """
        Context manager to allow multiple field accesses to be performed with a single
        read of the register
        """
        self.__register_state = await self.read()
        # pylint: disable=duplicate-code
        self.__in_context_manager = True
        # this try/finally is needed to make sure that in the event of an exception
        # the state flags are not left incorrectly set
        try:
            yield self
        finally:
            self.__in_context_manager = False

    async def read(self) -> int:
        """Asynchronously read value from the register

        Returns:
            The value from register

        """
        # pylint: disable=duplicate-code
        if self.__in_context_manager:
            return self.__register_state

        read_block_callback = self._callbacks.read_block_callback
        read_callback = self._callbacks.read_callback
        # pylint: enable=duplicate-code

        if read_callback is not None:

            return await read_callback(addr=self.address,
                                       width=self.width,
                                       accesswidth=self.accesswidth)

        if read_block_callback is not None:
            array_read_result = \
                await read_block_callback(addr=self.address,
                                          width=self.width,
                                          accesswidth=self.accesswidth,
                                          length=1)
            return array_read_result[0]

        raise RuntimeError('This function does not have a useable callback')

    @property
    def readable_fields(self) -> Iterator[Union['FieldAsyncReadOnly', 'FieldAsyncReadWrite']]:
        """
        generator that produces has all the readable fields within the register
        """
        def is_readable(field: Union['FieldAsyncReadOnly',
                                     'FieldAsyncWriteOnly',
                                     'FieldAsyncReadWrite']) -> \
                TypeGuard[Union['FieldAsyncReadOnly', 'FieldAsyncReadWrite']]:
            return isinstance(field, (FieldAsyncReadOnly, FieldAsyncReadWrite))

        return filter(is_readable, self.fields)

    async def read_fields(self) -> Dict['str', Union[bool, Enum, int]]:
        """
        asynchronously read the register and return a dictionary of the field values
        """
        return_dict: Dict['str', Union[bool, Enum, int]] = {}
        async with self.single_read() as reg:
            for field in reg.readable_fields:
                return_dict[field.inst_name] = await field.read()

        return return_dict

    @property
    def _is_readable(self) -> bool:
        return True

    @property
    def _is_writeable(self) -> bool:
        return False


class RegAsyncWriteOnly(AsyncReg, ABC):
    """
    class for an asynchronous write only register
    """

    __slots__: List[str] = []

    # pylint: disable=too-many-arguments, duplicate-code, useless-parent-delegation
    def __init__(self, *,
                 address: int,
                 width: int,
                 accesswidth: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Union[AsyncAddressMap, AsyncRegFile,
                               WritableAsyncMemory, WritableAsyncMemoryLegacy]):

        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         parent=parent, width=width, accesswidth=accesswidth)
    # pylint: enable=too-many-arguments, duplicate-code

    async def write(self, data: int) -> None:
        """Asynchronously writes a value to the register

        Args:
            data: data to be written

        Raises:
            ValueError: if the value provided is outside the range of the
                permissible values for the register
            TypeError: if the type of data is wrong
        """

        # this method check the types and range checks the data
        self._validate_data(data=data)

        # pylint: disable=duplicate-code
        self._logger.info('Writing data:%X to %X', data, self.address)
        # pylint: enable=duplicate-code

        if self._callbacks.write_callback is not None:

            await self._callbacks.write_callback(addr=self.address,
                                                 width=self.width,
                                                 accesswidth=self.accesswidth,
                                                 data=data)

        elif self._callbacks.write_block_callback is not None:

            if isinstance(self._callbacks, AsyncCallbackSetLegacy):
                data_as_array = Array(get_array_typecode(self.width), [data])
                await self._callbacks.write_block_callback(addr=self.address,
                                                           width=self.width,
                                                           accesswidth=self.accesswidth,
                                                           data=data_as_array)

            if isinstance(self._callbacks, AsyncCallbackSet):
                await self._callbacks.write_block_callback(addr=self.address,
                                                           width=self.width,
                                                           accesswidth=self.accesswidth,
                                                           data=[data])

        else:
            # pylint: disable-next=duplicate-code
            raise RuntimeError('This function does not have a useable callback')

    @property
    def writable_fields(self) -> Iterator[Union['FieldAsyncWriteOnly', 'FieldAsyncReadWrite']]:
        """
        generator that produces has all the writable fields within the register
        """
        def is_writable(field: Union['FieldAsyncReadOnly',
                                     'FieldAsyncWriteOnly',
                                     'FieldAsyncReadWrite']) -> \
                TypeGuard[Union['FieldAsyncWriteOnly', 'FieldAsyncReadWrite']]:
            return isinstance(field, (FieldAsyncWriteOnly, FieldAsyncReadWrite))

        return filter(is_writable, self.fields)

    @abstractmethod
    async def write_fields(self, **kwargs) -> None:  # type: ignore[no-untyped-def]
        """
        Do an async write to the register, updating any field included in
        the arguments
        """

    @property
    def _is_readable(self) -> bool:
        return False

    @property
    def _is_writeable(self) -> bool:
        return True


class RegAsyncReadWrite(RegAsyncReadOnly, RegAsyncWriteOnly, ABC):
    """
    class for an async read and write only register

    """
    __slots__: List[str] = ['__in_read_write_context_manager', '__in_read_context_manager',
                            '__register_state']

    # pylint: disable=too-many-arguments, duplicate-code
    def __init__(self, *,
                 address: int,
                 width: int,
                 accesswidth: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Union[AsyncAddressMap, AsyncRegFile, MemoryAsyncReadWrite,
                               MemoryAsyncReadWriteLegacy]):

        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         parent=parent, width=width, accesswidth=accesswidth)

        self.__in_read_write_context_manager: bool = False
        self.__in_read_context_manager: bool = False
        self.__register_state: Optional[int] = None

    # pylint: enable=too-many-arguments, duplicate-code

    @asynccontextmanager
    async def single_read_modify_write(self, verify: bool = False, skip_write: bool = False) -> \
            AsyncGenerator[Self, None]:
        """
        Context manager to allow multiple field reads/write to be done with a single set of
        field operations

        Args:
            verify (bool): very the write with a read afterwards
            skip_write (bool): skip the write back at the end

        """
        # pylint: disable=duplicate-code
        if self.__in_read_context_manager:
            raise RuntimeError('using the `single_read_modify_write` context manager within the '
                               'single_read` is not permitted')

        if skip_write is True:
            warn('The `skip_write` argument will be removed in the future, use `single_read`'
                 ' instead',
                 DeprecationWarning, stacklevel=2)
        # pylint: enable=duplicate-code

        self.__register_state = await self.read()

        # pylint: disable=duplicate-code
        self.__in_read_write_context_manager = True
        # this try/finally is needed to make sure that in the event of an exception
        # the state flags are not left incorrectly set
        try:
            yield self
        finally:
            self.__in_read_write_context_manager = False
        # pylint: enable=duplicate-code
        if not skip_write:
            await self.write(self.__register_state, verify)

        # clear the register states at the end of the context manager
        self.__register_state = None

    @asynccontextmanager
    async def single_read(self) -> \
            AsyncGenerator[Self, None]:
        """
        Context manager to allow multiple field reads with a single register read
        """
        # pylint: disable=duplicate-code
        if self.__in_read_write_context_manager:
            raise RuntimeError('using the `single_read` context manager within the '
                               'single_read_modify_write` is not permitted')
        self.__in_read_context_manager = True
        # pylint: enable=duplicate-code
        try:
            async with super().single_read() as reg:
                yield reg
        finally:
            self.__in_read_context_manager = False

    async def write(self, data: int, verify: bool = False) -> None:
        """
        Writes a value to the register

        Args:
            data: data to be written
            verify: set to True to read back the register to verify the read has occurred correctly

        Raises:
            ValueError: if the value provided is outside the range of the
                        permissible values for the register
            TypeError: if the type of data is wrong
            RegisterWriteVerifyError: the read back data after the write does not match the
                                      expected value
        """
        if self.__in_read_context_manager:
            raise RuntimeError('writes within the single read context manager are not permitted')

        if self.__in_read_write_context_manager:
            # pylint: disable=duplicate-code
            if self.__register_state is None:
                raise RuntimeError('The internal register state should never be None in the '
                                   'context manager')
            self.__register_state = data
        else:
            await super().write(data)
            if verify:
                read_back = await self.read()
                if read_back != data:
                    raise RegisterWriteVerifyError(f'Readback {read_back:X} '
                                                   f'after writing {data:X}')

    async def read(self) -> int:
        """
        Asynchronously read value from the register
        """
        if self.__in_read_write_context_manager:
            # pylint: disable=duplicate-code
            if self.__register_state is None:
                raise RuntimeError('The internal register state should never be None in the '
                                   'context manager')
            return self.__register_state

        # the single read context manager is handled in the base class so does need any
        # handling here

        return await super().read()

    async def read_fields(self) -> Dict['str', Union[bool, Enum, int]]:
        """
        asynchronously read the register and return a dictionary of the field values
        """
        return_dict: Dict['str', Union[bool, Enum, int]] = {}
        async with self.single_read() as reg:
            for field in reg.readable_fields:
                return_dict[field.inst_name] = await field.read()

        return return_dict

    async def write_fields(self, **kwargs) -> None:  # type: ignore[no-untyped-def]
        """
        asynchronously read-modify-write to the register, updating any field included in
        the arguments
        """
        if len(kwargs) == 0:
            raise ValueError('no command args')

        async with self.single_read_modify_write() as reg:
            for field_name, field_value in kwargs.items():
                if field_name not in reg.systemrdl_python_child_name_map.values():
                    raise ValueError(f'{field_name} is not a member of the register')

                field = getattr(reg, field_name)
                await field.write(field_value)

    @property
    def _is_readable(self) -> bool:
        return True

    @property
    def _is_writeable(self) -> bool:
        return True


ReadableAsyncRegister = Union[RegAsyncReadOnly, RegAsyncReadWrite]
WritableAsyncRegister = Union[RegAsyncWriteOnly, RegAsyncReadWrite]


# pylint: disable-next=invalid-name
AsyncRegArrayElementType= TypeVar('AsyncRegArrayElementType', bound=AsyncReg)


class AsyncRegArray(BaseRegArray, ABC):
    """
    base class of register array wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    # pylint: disable=too-many-arguments,duplicate-code

    __slots__: List[str] = ['__in_context_manager', '__register_cache',
                            '__register_address_array']

    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: Union[AsyncAddressMap, AsyncRegFile, AsyncMemory],
                 width: int,
                 accesswidth: int,
                 address: int,
                 stride: int,
                 dimensions: Tuple[int, ...],
                 elements: Optional[Dict[Tuple[int, ...], AsyncRegArrayElementType]] = None):

        self.__in_context_manager: bool = False
        self.__register_cache: Optional[Union[Array, List[int]]] = None
        self.__register_address_array: Optional[List[int]] = None

        if not isinstance(parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            raise TypeError(f'callback set type is wrong, got {type(parent._callbacks)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address, width=width, accesswidth=accesswidth,
                         stride=stride, dimensions=dimensions, elements=elements)

    @property
    def __empty_array_cache(self) -> Array:
        return Array(get_array_typecode(self.width), self.__empty_list_cache)

    @property
    def __empty_list_cache(self) -> List[int]:
        return [0 for _ in range(self.__number_cache_entries)]

    async def __block_read_legacy(self) -> Array:
        """
        Read all the contents of the array in the most optimal way, ideally with a block operation
        """
        if not isinstance(self._callbacks, AsyncCallbackSetLegacy):
            raise RuntimeError('This function should only be used with legacy callbacks')

        read_block_callback = self._callbacks.read_block_callback
        read_callback = self._callbacks.read_callback

        if read_block_callback is not None:

            data_read = await read_block_callback(addr=self.address,
                                                  width=self.width,
                                                  accesswidth=self.accesswidth,
                                                  length=self.__number_cache_entries)

            if not isinstance(data_read, Array):
                raise TypeError('The read block callback is expected to return an array')

            return data_read

        if read_callback is not None:
            # there is not read_block_callback defined so we must used individual read
            data_array = self.__empty_array_cache

            if self.__register_address_array is None:
                raise RuntimeError('This address array has not be initialised')

            for entry, address in enumerate(self.__register_address_array):
                data_entry = await read_callback(addr=address,
                                                 width=self.width,
                                                 accesswidth=self.accesswidth)

                data_array[entry] = data_entry

            return data_array

        raise RuntimeError('There is no usable callback')

    async def __block_write_legacy(self, data: Array, verify: bool) -> None:
        """
        Write all the contents of the array in the most optimal way, ideally with a block operation
        """
        if not isinstance(self._callbacks, AsyncCallbackSetLegacy):
            raise RuntimeError('This function should only be used with legacy callbacks')

        write_block_callback = self._callbacks.write_block_callback
        write_callback = self._callbacks.write_callback

        if write_block_callback is not None:
            await write_block_callback(addr=self.address,
                                       width=self.width,
                                       accesswidth=self.width,
                                       data=data)

        elif write_callback is not None:
            # there is not write_block_callback defined so we must used individual write

            if self.__register_address_array is None:
                raise RuntimeError('This address array has not be initialised')

            for entry_index, entry_data in enumerate(data):
                entry_address = self.__register_address_array[entry_index]
                await write_callback(addr=entry_address,
                                     width=self.width,
                                     accesswidth=self.accesswidth,
                                     data=entry_data)

        else:
            raise RuntimeError('No suitable callback')

        if verify:
            read_back_verify_data = self.__block_read_legacy()
            if read_back_verify_data != data:
                raise RegisterWriteVerifyError('Read back block miss-match')

    async def __block_read(self) -> List[int]:
        """
        Read all the contents of the array in the most optimal way, ideally with a block operation
        """
        if not isinstance(self._callbacks, AsyncCallbackSet):
            raise RuntimeError('This function should only be used with non-legacy callbacks')

        read_block_callback = self._callbacks.read_block_callback
        read_callback = self._callbacks.read_callback

        if read_block_callback is not None:
            data_read = \
                await read_block_callback(addr=self.address,
                                          width=self.width,
                                          accesswidth=self.accesswidth,
                                          length=self.__number_cache_entries)


            if not isinstance(data_read, List):
                raise TypeError('The read block callback is expected to return an array')

            return data_read

        if read_callback is not None:
            # there is not read_block_callback defined so we must used individual read
            data_list = self.__empty_list_cache

            if self.__register_address_array is None:
                raise RuntimeError('This address array has not be initialised')

            for entry, address in enumerate(self.__register_address_array):
                data = await read_callback(addr=address,
                                           width=self.width,
                                           accesswidth=self.accesswidth)

                data_list[entry] = data

            return data_list

        raise RuntimeError('There is no usable callback')

    async def __block_write(self, data: List[int], verify: bool) -> None:
        """
        Write all the contents of the array in the most optimal way, ideally with a block operation
        """
        if not isinstance(self._callbacks, AsyncCallbackSet):
            raise RuntimeError('This function should only be used with non-legacy callbacks')

        write_block_callback = self._callbacks.write_block_callback
        write_callback = self._callbacks.write_callback

        if write_block_callback is not None:
            await write_block_callback(addr=self.address,
                                       width=self.width,
                                       accesswidth=self.width,
                                       data=data)
        elif write_callback is not None:
            # there is not write_block_callback defined so we must used individual write

            if self.__register_address_array is None:
                raise RuntimeError('This address array has not be initialised')

            for entry_index, entry_data in enumerate(data):
                entry_address = self.__register_address_array[entry_index]
                await write_callback(addr=entry_address,
                                     width=self.width,
                                     accesswidth=self.accesswidth,
                                     data=entry_data)

        else:
            raise RuntimeError('No suitable callback')

        if verify:
            read_back_verify_data = self.__block_read()
            if read_back_verify_data != data:
                raise RegisterWriteVerifyError('Read back block miss-match')

    def __cache_entry(self, addr: int, width: int, accesswidth: int) -> int:
        """
        Validate the data provided and determine the cache entry

        Args:
            addr: Address to write to
            width: Width of the register in bits
            accesswidth: Minimum access width of the register in bits

        Returns:
            cache entry

        """
        if not isinstance(width, int):
            raise TypeError(f'Width should be an int byt got {type(width)}')
        if width != self.width:
            raise ValueError('Requested Read width does not match the expected value')
        if not isinstance(accesswidth, int):
            raise TypeError(f'accesswidth should be an int byt got {type(accesswidth)}')
        if accesswidth != self.accesswidth:
            raise ValueError('Requested Read accesswidth does not match the expected value')
        if not isinstance(addr, int):
            raise TypeError(f'addr should be an int byt got {type(addr)}')
        if not self.address <= addr < (self.address + self.size):
            raise ValueError(f'Requested address 0x{addr:X} is out of range 0x{self.address:X} to '
                             f'0x{self.address + self.size - (self.width >> 3):X}')
        cache_entry = (addr - self.address) // (self.width >> 3)
        if self.__register_address_array is None:
            raise RuntimeError('The address table should always be populated here')
        if self.__register_address_array[cache_entry] != addr:
            raise RuntimeError(f'The calculated cache entry for address 0x{addr:X}')
        return cache_entry

    async def __cache_read(self, addr: int, width: int, accesswidth: int) -> int:
        """
        Used to replace the normal callbacks with those that access the cache

        Args:
            addr: Address to write to
            width: Width of the register in bits
            accesswidth: Minimum access width of the register in bits

        Returns:
            value inputted by the used
        """
        if self.__register_cache is None:
            raise RuntimeError('The cache array should be initialised')
        return self.__register_cache[self.__cache_entry(addr=addr,
                                                        width=width,
                                                        accesswidth=accesswidth)]

    async def __cache_write(self, addr: int, width: int, accesswidth: int, data: int) -> None:
        """
        Used to replace the normal callbacks with those that access the cache

        Args:
            addr: Address to write to
            width: Width of the register in bits
            accesswidth: Minimum access width of the register in bits
            data: value to be written to the register

        Returns:
            None
        """
        if not isinstance(data, int):
            raise TypeError(f'Data should be an int byt got {type(data)}')
        if self.__register_cache is None:
            raise RuntimeError('The cache array should be initialised')
        self.__register_cache[self.__cache_entry(addr=addr,
                                                 width=width,
                                                 accesswidth=accesswidth)] = data

    @property
    def __cache_callbacks(self) -> AsyncCallbackSet:
        return AsyncCallbackSet(read_callback=self.__cache_read,
                                write_callback=self.__cache_write)

    @property
    def __number_cache_entries(self) -> int:
        return self.size // (self.width >> 3)

    async def __initialise_cache(self, skip_initial_read: bool) -> Union[Array, List[int]]:
        if isinstance(self._callbacks, AsyncCallbackSet):
            if skip_initial_read:
                return self.__empty_list_cache
            return await self.__block_read()

        if isinstance(self._callbacks, AsyncCallbackSetLegacy):
            if skip_initial_read:
                return self.__empty_array_cache
            return await self.__block_read_legacy()

        raise TypeError('Unhandled callback type')

    @asynccontextmanager
    async def _cached_access(self, verify: bool = False, skip_write: bool = False,
                             skip_initial_read: bool = False) -> \
            AsyncGenerator[Self, None]:
        """
        Context manager to allow multiple field reads/write to be done with a single set of
        field operations

        Args:
            verify (bool): very the write with a read afterwards
            skip_write (bool): skip the write back at the end

        """
        self.__register_address_array = \
            [self.address + (i * (self.width >> 3)) for i in range(self.__number_cache_entries)]
        self.__register_cache = await self.__initialise_cache(skip_initial_read=skip_initial_read)
        self.__in_context_manager = True
        # this try/finally is needed to make sure that in the event of an exception
        # the state flags are not left incorrectly set
        try:
            yield self
        finally:
            self.__in_context_manager = False
        if not skip_write:
            if isinstance(self._callbacks, AsyncCallbackSet):
                if not isinstance(self.__register_cache, list):
                    raise TypeError('Register cache should be a list in non-legacy mode')
                await self.__block_write(self.__register_cache, verify)
            if isinstance(self._callbacks, AsyncCallbackSetLegacy):
                if not isinstance(self.__register_cache, Array):
                    raise TypeError('Register cache should be a Array in legacy mode')
                await self.__block_write_legacy(self.__register_cache, verify)

        # clear the register states at the end of the context manager
        self.__register_address_array = None
        self.__register_cache = None

    @property
    def _callbacks(self) -> Union[AsyncCallbackSet, AsyncCallbackSetLegacy]:
        # pylint: disable=protected-access
        if self.__in_context_manager:
            return self.__cache_callbacks

        if self.parent is None:
            raise RuntimeError('Parent must be set')

        if isinstance(self.parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            return self.parent._callbacks

        raise TypeError(f'unhandled parent callback type: {type(self.parent._callbacks)}')


class RegAsyncReadOnlyArray(AsyncRegArray, ABC):
    """
    base class for a array of async read only registers
    """
    __slots__: List[str] = []

    # pylint: disable=too-many-arguments,duplicate-code
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: Union[AsyncRegFile, AsyncAddressMap, ReadableAsyncMemory,
                               ReadableAsyncMemoryLegacy],
                 address: int,
                 width: int,
                 accesswidth: int,
                 stride: int,
                 dimensions: Tuple[int, ...],
                 elements: Optional[Dict[Tuple[int, ...], RegAsyncReadOnly]] = None):

        if not isinstance(parent, (AsyncRegFile, AsyncAddressMap,
                                   MemoryAsyncReadOnly, MemoryAsyncReadOnlyLegacy,
                                   MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy)):
            raise TypeError('parent should be either AsyncRegFile, AsyncAddressMap, '
                            'MemoryAsyncReadOnly, MemoryAsyncReadWrite '
                            f'got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address, width=width, accesswidth=accesswidth,
                         stride=stride, dimensions=dimensions, elements=elements)
    # pylint: enable=too-many-arguments,duplicate-code

    @asynccontextmanager
    async def single_read(self) -> \
            AsyncGenerator[Self, None]:
        """
        Context manager to allow multiple field reads/write to be done with a single set of
        field operations

        Args:


        Returns:

        """
        async with self._cached_access(verify=False, skip_write=True,
                                       skip_initial_read=False) as reg_array:
            yield reg_array

    @property
    def _is_readable(self) -> bool:
        return True

    @property
    def _is_writeable(self) -> bool:
        return False


class RegAsyncWriteOnlyArray(AsyncRegArray, ABC):
    """
    base class for a array of async write only registers
    """
    __slots__: List[str] = []

    # pylint: disable=too-many-arguments,duplicate-code
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: Union[AsyncRegFile, AsyncAddressMap,
                               WritableAsyncMemory, WritableAsyncMemoryLegacy],
                 address: int,
                 width: int,
                 accesswidth: int,
                 stride: int,
                 dimensions: Tuple[int, ...],
                 elements: Optional[Dict[Tuple[int, ...], RegAsyncWriteOnly]] = None):

        if not isinstance(parent, (AsyncRegFile, AsyncAddressMap,
                                   MemoryAsyncWriteOnly, MemoryAsyncReadWrite,
                                   MemoryAsyncWriteOnlyLegacy, MemoryAsyncReadWriteLegacy)):
            raise TypeError('parent should be either AsyncRegFile, AsyncAddressMap, '
                            'MemoryAsyncWriteOnly, MemoryAsyncReadWrite '
                            f'got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address, width=width, accesswidth=accesswidth,
                         stride=stride, dimensions=dimensions, elements=elements)
    # pylint: enable=too-many-arguments,duplicate-code

    @asynccontextmanager
    async def single_write(self) -> \
            AsyncGenerator[Self, None]:
        """
        Context manager to allow multiple field reads/write to be done with a single set of
        field operations

        Args:


        Returns:

        """
        async with self._cached_access(verify=False, skip_write=False,
                                       skip_initial_read=True) as reg_array:
            yield reg_array

    @property
    def _is_readable(self) -> bool:
        return False

    @property
    def _is_writeable(self) -> bool:
        return True


class RegAsyncReadWriteArray(AsyncRegArray, ABC):
    """
    base class for a array of read and write registers
    """
    __slots__: List[str] = []

    # pylint: disable=too-many-arguments,duplicate-code
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: Union[AsyncRegFile, AsyncAddressMap,
                               MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy],
                 address: int,
                 width: int,
                 accesswidth: int,
                 stride: int,
                 dimensions: Tuple[int, ...],
                 elements: Optional[Dict[Tuple[int, ...], RegAsyncReadWrite]] = None):

        if not isinstance(parent, (AsyncRegFile, AsyncAddressMap,
                                   MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy)):
            raise TypeError('parent should be either AsyncRegFile, AsyncAddressMap, '
                            'MemoryAsyncReadWrite '
                            f'got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address, width=width, accesswidth=accesswidth,
                         stride=stride, dimensions=dimensions, elements=elements)
    # pylint: enable=too-many-arguments,duplicate-code

    @asynccontextmanager
    async def single_read_modify_write(self, verify: bool = False, skip_write: bool = False) -> \
            AsyncGenerator[Self, None]:
        """
        Context manager to allow multiple field reads/write to be done with a single set of
        field operations

        Args:
            verify (bool): very the write with a read afterwards
            skip_write (bool): skip the write back at the end

        Returns:

        """
        if skip_write is True:
            warn('The `skip_write` argument will be removed in the future, use `single_read`'
                 ' instead',
                 DeprecationWarning, stacklevel=2)

        async with self._cached_access(verify=verify, skip_write=skip_write,
                                       skip_initial_read=False) as reg_array:
            yield reg_array

    @property
    def _is_readable(self) -> bool:
        return True

    @property
    def _is_writeable(self) -> bool:
        return True


ReadableAsyncRegisterArray = Union[RegAsyncReadOnlyArray, RegAsyncReadWriteArray]
WriteableAsyncRegisterArray = Union[RegAsyncWriteOnlyArray, RegAsyncReadWriteArray]


class FieldAsyncReadOnly(_FieldReadOnlyFramework, ABC):
    """
    class for an asynchronous read only register field

    Args:
        parent_register: register within which the field resides
        size_props: object defining the msb, lsb, high bit, low bit and width
        logger_handle: name to be used logging messages associate with this
            object

    """
    __slots__: List[str] = []

    def __init__(self, *,
                 parent_register: ReadableAsyncRegister,
                 size_props: FieldSizeProps,
                 misc_props: FieldMiscProps,
                 logger_handle: str,
                 inst_name: str):

        if not isinstance(parent_register, (RegAsyncReadWrite, RegAsyncReadOnly)):
            raise TypeError(f'size_props must be of {type(RegAsyncReadWrite)} or '
                            f'{type(RegAsyncReadOnly)} but got {type(parent_register)}')

        # pylint: disable=duplicate-code
        super().__init__(logger_handle=logger_handle,
                         size_props=size_props,
                         misc_props=misc_props,
                         parent_register=parent_register,
                         inst_name=inst_name)
        # pylint: enable=duplicate-code

    async def read(self) -> int:  # pylint: disable=invalid-overridden-method
        """
        Asynchronously reads the register that this field is located in and retries the field
        value applying the required masking and shifting

        Returns:
            field value

        """
        return self.decode_read_value(await self.parent_register.read())

    @property
    def parent_register(self) -> ReadableAsyncRegister:
        """
        parent register the field is placed in
        """

        # this cast is OK because an explict typing check was done in the __init__
        return cast(ReadableAsyncRegister, self.parent)


class FieldAsyncWriteOnly(_FieldWriteOnlyFramework, ABC):
    """
    class for an asynchronous write only register field

    Args:
        parent_register: register within which the field resides
        size_props: object defining the msb, lsb, high bit, low bit and width
        logger_handle: name to be used logging messages associate with this
            object

    """
    __slots__: List[str] = []

    def __init__(self, *,
                 parent_register: WritableAsyncRegister,
                 size_props: FieldSizeProps,
                 misc_props: FieldMiscProps,
                 logger_handle: str,
                 inst_name: str):

        if not isinstance(parent_register, (RegAsyncReadWrite, RegAsyncWriteOnly)):
            raise TypeError(f'size_props must be of {type(RegAsyncReadWrite)} or '
                            f'{type(RegAsyncWriteOnly)} but got {type(parent_register)}')

        # pylint: disable=duplicate-code
        super().__init__(logger_handle=logger_handle,
                         size_props=size_props,
                         misc_props=misc_props,
                         parent_register=parent_register,
                         inst_name=inst_name)
        # pylint: enable=duplicate-code

    async def write(self, value: int) -> None:  # pylint: disable=invalid-overridden-method
        """
        The behaviour of this method depends on whether the field is located in
        a readable register or not:

        If the register is readable, the method will perform an async read-modify-write
        on the register updating the field with the value provided

        If the register is not writable all other field values will be asynchronously written
        with zero.

        Args:
            value: field value to update to

        """

        # This code closely resembles the non-async version but is duplicated for clarity
        # pylint: disable=duplicate-code
        self._write_value_checks(value=value)

        if self.msb0:
            value = swap_msb_lsb_ordering(value=value, width=self.width)
        # pylint: enable=duplicate-code

        if (self.high == (self.register_data_width - 1)) and (self.low == 0):
            # special case where the field occupies the whole register,
            # there a straight write can be performed
            new_reg_value = value
        else:
            # do a read, modify write
            if isinstance(self.parent_register, RegAsyncReadWrite):
                reg_value = await self.parent_register.read()
                masked_reg_value = reg_value & self.inverse_bitmask
                new_reg_value = masked_reg_value | (value << self.low)
            elif isinstance(self.parent_register, RegAsyncWriteOnly):
                new_reg_value = value << self.low
            else:
                raise TypeError('Unhandled parent type')

        await self.parent_register.write(new_reg_value)

    @property
    def parent_register(self) -> WritableAsyncRegister:
        """
        parent register the field is placed in
        """

        # this cast is OK because an explict typing check was done in the __init__
        return cast(WritableAsyncRegister, self.parent)


class FieldAsyncReadWrite(FieldAsyncReadOnly, FieldAsyncWriteOnly, ABC):
    """
    class for an asyncronous read/write register field

    Args:
        parent_register: register within which the field resides
        size_props: object defining the msb, lsb, high bit, low bit and width
        logger_handle: name to be used logging messages associate with this
            object

    """
    __slots__: List[str] = []

    def __init__(self, *,
                 parent_register: RegAsyncReadWrite,
                 size_props: FieldSizeProps,
                 misc_props: FieldMiscProps,
                 logger_handle: str,
                 inst_name: str):

        if not isinstance(parent_register, RegAsyncReadWrite):
            raise TypeError(f'size_props must be of {type(RegAsyncReadWrite)} '
                            f'but got {type(parent_register)}')

        # pylint: disable=duplicate-code
        super().__init__(logger_handle=logger_handle,
                         size_props=size_props,
                         misc_props=misc_props,
                         parent_register=parent_register,
                         inst_name=inst_name)
        # pylint: enable=duplicate-code

    @property
    def parent_register(self) -> RegAsyncReadWrite:
        """
        parent register the field is placed in
        """

        # this cast is OK because an explict typing check was done in the __init__
        return cast(RegAsyncReadWrite, self.parent)


class FieldEnumAsyncReadWrite(FieldAsyncReadWrite, FieldEnum, ABC):
    """
    class for an async read/write register field with an enumerated value
    """
    __slots__: List[str] = []

    @property
    def parent_register(self) -> RegAsyncReadWrite:
        """
        parent register the field is placed in
        """

        # this cast is OK because an explict typing check was done in the __init__
        return cast(RegAsyncReadWrite, self.parent)


class FieldEnumAsyncReadOnly(FieldAsyncReadOnly, FieldEnum, ABC):
    """
    class for an async read only register field with an enumerated value
    """
    __slots__: List[str] = []


class FieldEnumAsyncWriteOnly(FieldAsyncWriteOnly, FieldEnum, ABC):
    """
    class for an async write only register field with an enumerated value
    """
    __slots__: List[str] = []
