"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2023

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This module is intended to distributed as part of automatically generated code by the
peakrdl-python tool. It provides a set of base classes used by the autogenerated code
"""
from typing import Dict, Optional

from .base import Base
from ._callbacks import MemoryReadCallback, MemoryWriteCallback


class _MemoryContent:

    __slots__ = ['__value', '__default_value', '__length']

    def __init__(self, *,
                 default_value: int):
        self.__value:Dict[int, int] = {}
        self.__default_value = default_value

    def __getitem__(self, item: int) -> int:
        if item in self.__value:
            return self.__value[item]

        return self.__default_value

    def __setitem__(self, key: int, value: int) -> None:
        self.__value[key] = value


class Memory(Base):
    """
    Simulation of a memory, this is implemented using a sparse approach to avoid the simulator
    storing every possible entry in a register map, which could be very big.
    """
    __slots__ = ['__value', '__width', '__length',
                 '__read_callback', '__write_callback']

    def __init__(self, *,
                 width: int,
                 length: int,
                 default_value: int,
                 full_inst_name: str):

        super().__init__(full_inst_name=full_inst_name)

        self.__value = _MemoryContent(default_value=default_value)
        self.__width = width
        self.__length = length
        self.__read_callback: Optional[MemoryReadCallback] = None
        self.__write_callback: Optional[MemoryWriteCallback] = None

    @property
    def read_callback(self) -> Optional[MemoryReadCallback]:
        """
        Callback made during each read operation
        """
        return self.__read_callback

    @read_callback.setter
    def read_callback(self, callback: Optional[MemoryReadCallback]) -> None:
        self.__read_callback = callback

    @property
    def write_callback(self) -> Optional[MemoryWriteCallback]:
        """
        Callback made during each write operation
        """
        return self.__write_callback

    @write_callback.setter
    def write_callback(self, callback: Optional[MemoryWriteCallback]) -> None:
        self.__write_callback = callback

    def read(self, offset: int) -> int:
        """
        Read a memory word

        Args:
            offset (int): Word offset in the memory

        Returns:
            memory word content

        """
        self.__offset_range_check(offset)
        value = self.value[offset]
        if self.read_callback is not None:
            # pylint does not recognise that the property is returning a callback therefore it
            # is legal to call it.
            # pylint: disable-next=not-callable
            self.read_callback(offset=offset, value=value)
        return value

    def write(self, offset: int, data: int) -> None:
        """
        Write a memory word

        Args:
            offset: (int): Word offset in the memory
            data: data word

        Returns:
            None

        """
        self.__offset_range_check(offset)
        if self.write_callback is not None:
            # pylint does not recognise that the property is returning a callback therefore it
            # is legal to call it.
            # pylint: disable-next=not-callable
            self.write_callback(offset=offset, value=data)
        self.value[offset] = data

    @property
    def _width_in_bytes(self) -> int:
        def roundup_pow2(x: int) -> int:
            return 1 << (x - 1).bit_length()

        return roundup_pow2(self.__width) // 8

    def byte_offset_to_word_offset(self, byte_offset: int) -> int:
        """
        Determine a memory word offset

        Args:
            byte_offset (int): byte offset from memory start

        Returns:
            word offset

        """
        return byte_offset // self._width_in_bytes

    @property
    def value(self) -> _MemoryContent:
        """
        Access to the memory content, bypassing the callbacks
        """
        return self.__value

    def __offset_range_check(self, offset: int) -> None:
        if not 0 <= offset < self.__length:
            raise IndexError(f'offset must in in range 0 to {self.__length-1}, got {offset:d}')
