"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2023

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This module is intended to distributed as part of automatically generated code by the
peakrdl-python tool. It provides a set of classes used by the autogenerated code to represent
sections e.g. AddressMap and RegFile
"""
from __future__ import annotations
import warnings
from typing import Optional, Union, TYPE_CHECKING,overload, Literal
from collections.abc import Iterator, Iterable
from abc import ABC, abstractmethod
import sys
from enum import IntEnum

from .base import Node, NodeArray, IterationClassification

from .callbacks import NormalCallbackSet, AsyncCallbackSet
from .callbacks import NormalCallbackSetLegacy, AsyncCallbackSetLegacy

if sys.version_info >= (3, 10):
    # type guarding was introduced in python 3.10
    from typing import TypeGuard
else:
    from typing_extensions import TypeGuard

if TYPE_CHECKING:
    from .memory import Memory, MemoryArray
    from .async_memory import AsyncMemory, AsyncMemoryArray
    from .register_and_field import Reg, RegArray
    from .register_and_field import WritableRegister, ReadableRegister
    from .async_register_and_field import AsyncReg, AsyncRegArray
    from .async_register_and_field import ReadableAsyncRegister, WritableAsyncRegister
    from .register_and_field import ReadableRegisterArray, WriteableRegisterArray
    from .async_register_and_field import ReadableAsyncRegisterArray, WriteableAsyncRegisterArray

UDPStruct = dict[str, 'UDPType']
UDPType = Union[str, int, bool, IntEnum, UDPStruct]

class BaseSection(Node, Iterable[Union[Node, NodeArray]], ABC):
    """
    base class of non-async and sync sections (AddressMaps and RegFile)

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.SECTION

    def get_children(self, unroll: bool = False) -> Iterator[Union[Node, NodeArray]]:
        """
        generator that produces all the readable_registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        if unroll:
            for child in iter(self):
                if isinstance(child, NodeArray):
                    yield from child
                else:
                    yield child
        else:
            yield from iter(self)

class Section(BaseSection, ABC):
    """
    base class of non-async sections (AddressMaps and RegFile)

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []

    def get_writable_registers(self, unroll:bool=False) -> \
            Iterator[Union[WritableRegister, WriteableRegisterArray]]:
        """
        generator that produces all the readable_registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        def is_writable(item: Union[Reg, RegArray]) -> \
                TypeGuard[Union[WritableRegister, WriteableRegisterArray]]:
            # pylint: disable-next=protected-access
            return item._is_writeable

        return filter(is_writable, self.get_registers(unroll=unroll))

    def get_readable_registers(self, unroll: bool = False) -> \
            Iterator[Union[ReadableRegister, ReadableRegisterArray]]:
        """
        generator that produces all the readable_registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        def is_readable(item: Union[Reg, RegArray]) ->\
                TypeGuard[Union[ReadableRegister, ReadableRegisterArray]]:
            # pylint: disable-next=protected-access
            return item._is_readable

        return filter(is_readable, self.get_registers(unroll=unroll))

    @overload
    def get_registers(self, unroll: Literal[True]) -> Iterator[Union[Reg]]:
        ...

    @overload
    def get_registers(self, unroll: Literal[False]) -> Iterator[Union[Reg, RegArray]]:
        ...

    @overload
    def get_registers(self, unroll: bool) -> Iterator[Union[Reg, RegArray]]:
        ...


    def get_registers(self, unroll: bool = False) -> Iterator[Union[Reg, RegArray]]:
        """
        generator that produces all the registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        def is_reg(node: Union[Node, NodeArray]) -> TypeGuard[Union[Reg, RegArray]]:
            # The _is_reg is needed within this library to avoid circular imports
            # pylint:disable-next=protected-access
            return node._iteration_classification is IterationClassification.REGISTER

        yield from filter(is_reg, self.get_children(unroll=unroll))


class AddressMap(Section, ABC):
    """
    base class of address map wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = ['__callbacks']

    def __init__(self, *,
                 callbacks: Optional[Union[NormalCallbackSet, NormalCallbackSetLegacy]],
                 address: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Optional['AddressMap']):

        # only the top-level address map should have callbacks assigned, everything else should
        # use its parent callback
        if parent is None:
            if not isinstance(callbacks, (NormalCallbackSet, NormalCallbackSetLegacy)):
                raise TypeError(f'callback type wrong, got {type(callbacks)}')
            if isinstance(callbacks, NormalCallbackSetLegacy):
                warnings.warn('Support for the legacy callback using the array types will be '
                              'withdrawn in the future, please consider changing to the list '
                              'versions', category=DeprecationWarning)
            self.__callbacks = callbacks
        else:
            if not callbacks is None:
                raise RuntimeError('Callbacks must be None when a parent is set')
            if not isinstance(parent._callbacks, (NormalCallbackSet, NormalCallbackSetLegacy)):
                raise TypeError(f'callback type wrong, got {type(callbacks)}')

        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         parent=parent)


    def get_sections(self, unroll: bool = False) -> \
            Iterator[Union['AddressMap', RegFile, AddressMapArray, RegFileArray]]:
        """
        generator that produces all the AddressMap and RegFile children of this node

        Args:
            unroll: Whether to unroll child array or not

        Returns:

        """
        def is_section(node: Union[Node, NodeArray]) -> \
                TypeGuard[Union['AddressMap', RegFile, AddressMapArray, RegFileArray]]:
            return isinstance(node, (AddressMap, RegFile, AddressMapArray, RegFileArray))

        yield from filter(is_section, self.get_children(unroll=unroll))

    def get_memories(self, unroll: bool = False) -> \
            Iterator[Union['Memory', 'MemoryArray']]:
        """
        generator that produces all the Memory children of this node

        Args:
            unroll: Whether to unroll child array or not

        Returns:

        """
        def is_mem(node: Union[Node, NodeArray]) -> \
                TypeGuard[Union['Memory', 'MemoryArray']]:
            # The _is_mem is needed within this library to avoid circular imports
            # pylint:disable-next=protected-access
            return node._iteration_classification is IterationClassification.MEMORY

        yield from filter(is_mem, self.get_children(unroll=unroll))


    @property
    def _callbacks(self) -> Union[NormalCallbackSet, NormalCallbackSetLegacy]:
        # pylint: disable=protected-access
        if self.parent is None:
            return self.__callbacks

        if isinstance(self.parent._callbacks, (NormalCallbackSet, NormalCallbackSetLegacy)):
            return self.parent._callbacks

        raise TypeError(f'unhandled parent callback type: {type(self.parent._callbacks)}')


class AsyncSection(BaseSection, ABC):
    """
    base class of async sections (AddressMaps and RegFile)

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """
    __slots__: list[str] = []

    def get_writable_registers(self, unroll: bool = False) -> \
            Iterator[Union[WritableAsyncRegister, WriteableAsyncRegisterArray]]:
        """
        generator that produces all the readable_registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        def is_writable(item: Union[AsyncReg, AsyncRegArray]) -> \
                TypeGuard[Union[WritableAsyncRegister, WriteableAsyncRegisterArray]]:
            # pylint: disable-next=protected-access
            return item._is_writeable

        return filter(is_writable, self.get_registers(unroll=unroll))

    def get_readable_registers(self, unroll: bool = False) -> \
            Iterator[Union[ReadableAsyncRegister, ReadableAsyncRegisterArray]]:
        """
        generator that produces all the readable_registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """
        def is_readable(item: Union[AsyncReg, AsyncRegArray]) -> \
                TypeGuard[Union[ReadableAsyncRegister, ReadableAsyncRegisterArray]]:
            # pylint: disable-next=protected-access
            return item._is_readable

        return filter(is_readable, self.get_registers(unroll=unroll))

    @overload
    def get_registers(self, unroll: Literal[True]) -> Iterator[Union[AsyncReg]]:
        ...

    @overload
    def get_registers(self, unroll: Literal[False]) -> Iterator[Union[AsyncReg, AsyncRegArray]]:
        ...

    @overload
    def get_registers(self, unroll: bool) -> Iterator[Union[AsyncReg, AsyncRegArray]]:
        ...

    def get_registers(self, unroll: bool = False) -> Iterator[Union[AsyncReg, AsyncRegArray]]:
        """
        generator that produces all the registers of this node

        Args:
            unroll: Whether to unroll child array or not
        """

        def is_reg(node: Union[Node, NodeArray]) -> TypeGuard[Union[AsyncReg, AsyncRegArray]]:
            # The _is_reg is needed within this library to avoid circular imports
            # pylint:disable-next=protected-access
            return node._iteration_classification is IterationClassification.REGISTER

        yield from filter(is_reg, self.get_children(unroll=unroll))

    @property
    @abstractmethod
    def _callbacks(self) -> Union[AsyncCallbackSet, AsyncCallbackSetLegacy]:
        ...


class AsyncAddressMap(AsyncSection, ABC):
    """
    base class of address map wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__: list[str] = ['__callbacks']

    def __init__(self, *,
                 callbacks: Optional[Union[AsyncCallbackSet, AsyncCallbackSetLegacy]],
                 address: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Optional['AsyncAddressMap']):

        # only the top-level address map should have callbacks assigned, everything else should
        # use its parent callback
        if parent is None:
            if not isinstance(callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
                raise TypeError(f'callback type wrong, got {type(callbacks)}')
            if isinstance(callbacks, AsyncCallbackSetLegacy):
                warnings.warn('Support for the legacy callback using the array types will be '
                              'withdrawn in the future, please consider changing to the list '
                              'versions', category=DeprecationWarning)
            self.__callbacks = callbacks
        else:
            if not callbacks is None:
                raise RuntimeError('Callbacks must be None when a parent is set')
            if not isinstance(parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
                raise TypeError(f'callback type wrong, got {type(callbacks)}')

        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         parent=parent)

    def get_sections(self, unroll: bool = False) -> \
            Iterator[Union['AsyncAddressMap', AsyncRegFile,
                           AsyncAddressMapArray, AsyncRegFileArray]]:
        """
        generator that produces all the AddressMap and RegFile children of this node

        Args:
            unroll: Whether to unroll child array or not

        Returns:

        """
        def is_section(node: Union[Node, NodeArray]) -> \
                TypeGuard[Union['AsyncAddressMap', AsyncRegFile,
                                AsyncAddressMapArray, AsyncRegFileArray]]:
            return isinstance(node, (AsyncAddressMap, AsyncRegFile,
                                     AsyncAddressMapArray, AsyncRegFileArray))

        yield from filter(is_section, self.get_children(unroll=unroll))

    def get_memories(self, unroll: bool = False) -> \
            Iterator[Union['AsyncMemory', 'AsyncMemoryArray']]:
        """
        generator that produces all the Memory children of this node

        Args:
            unroll: Whether to unroll child array or not

        Returns:

        """
        def is_mem(node: Union[Node, NodeArray]) -> \
                TypeGuard[Union['AsyncMemory', 'AsyncMemoryArray']]:
            # The _is_mem is needed within this library to avoid circular imports
            # pylint:disable-next=protected-access
            return node._iteration_classification is IterationClassification.MEMORY

        yield from filter(is_mem, self.get_children(unroll=unroll))

    @property
    def _callbacks(self) -> Union[AsyncCallbackSet, AsyncCallbackSetLegacy]:
        # pylint: disable=protected-access
        if self.parent is None:
            return self.__callbacks

        if isinstance(self.parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            return self.parent._callbacks

        raise TypeError(f'unhandled parent callback type: {type(self.parent._callbacks)}')



class AddressMapArray(NodeArray[AddressMap], ABC):
    """
    base class for a array of address maps
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.SECTION


    # pylint: disable-next=too-many-arguments
    def __init__(self, *, logger_handle: str, inst_name: str,
                 parent: AddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[AddressMap, ...]]] = None):

        if not isinstance(parent, AddressMap):
            raise TypeError(f'parent should be AddressMap got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address, stride=stride,
                         dimensions=dimensions, elements=elements)


class AsyncAddressMapArray(NodeArray[AsyncAddressMap], ABC):
    """
    base class for a array of address maps
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.SECTION


    # pylint: disable-next=too-many-arguments
    def __init__(self, *, logger_handle: str, inst_name: str,
                 parent: AsyncAddressMap,
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[AsyncAddressMap, ...]]] = None):

        if not isinstance(parent, AsyncAddressMap):
            raise TypeError(f'parent should be AsyncAddressMap got {type(parent)}')
        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions, elements=elements)


class RegFile(Section, ABC):
    """
    base class of register file wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__: list[str] = []

    def __init__(self, *,
                 address: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Union[AddressMap, 'RegFile']):

        if not isinstance(parent._callbacks, (NormalCallbackSet, NormalCallbackSetLegacy)):
            raise TypeError(f'parent._callbacks type wrong, got {type(parent._callbacks)}')

        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         parent=parent)

    def get_sections(self, unroll: bool = False) -> \
            Iterator[Union['RegFile','RegFileArray']]:
        """
        generator that produces all the RegFile children of this node

        Args:
            unroll: Whether to unroll child array or not

        Returns:

        """
        def is_section(node: Union[Node, NodeArray]) -> \
                TypeGuard[Union['RegFile','RegFileArray']]:
            return isinstance(node, (RegFile,RegFileArray))

        yield from filter(is_section, self.get_children(unroll=unroll))

    @property
    def _callbacks(self) -> Union[NormalCallbackSet, NormalCallbackSetLegacy]:
        # pylint: disable=protected-access
        if self.parent is None:
            raise RuntimeError('Parent must be set')

        if isinstance(self.parent._callbacks, (NormalCallbackSet, NormalCallbackSetLegacy)):
            return self.parent._callbacks

        raise TypeError(f'unhandled parent callback type: {type(self.parent._callbacks)}')



class AsyncRegFile(AsyncSection, ABC):
    """
    base class of register file wrappers

    Note:
        It is not expected that this class will be instantiated under normal
        circumstances however, it is useful for type checking
    """

    __slots__: list[str] = []

    def __init__(self, *,
                 address: int,
                 logger_handle: str,
                 inst_name: str,
                 parent: Union[AsyncAddressMap, 'AsyncRegFile']):
        super().__init__(address=address,
                         logger_handle=logger_handle,
                         inst_name=inst_name,
                         parent=parent)

        if not isinstance(parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            raise TypeError(f'parent._callbacks type wrong, got {type(parent._callbacks)}')

    def get_sections(self, unroll: bool = False) -> \
            Iterator[Union['AsyncRegFile','AsyncRegFileArray']]:
        """
        generator that produces all the RegFile children of this node

        Args:
            unroll: Whether to unroll child array or not

        Returns:

        """
        def is_section(node: Union[Node, NodeArray]) -> \
                TypeGuard[Union['AsyncRegFile','AsyncRegFileArray']]:
            return isinstance(node, (AsyncRegFile,AsyncRegFileArray))

        yield from filter(is_section, self.get_children(unroll=unroll))

    @property
    def _callbacks(self) -> Union[AsyncCallbackSet, AsyncCallbackSetLegacy]:
        # pylint: disable=protected-access
        if self.parent is None:
            raise RuntimeError('Parent must be set')

        if isinstance(self.parent._callbacks, (AsyncCallbackSet, AsyncCallbackSetLegacy)):
            return self.parent._callbacks

        raise TypeError(f'unhandled parent callback type: {type(self.parent._callbacks)}')


class RegFileArray(NodeArray[RegFile], ABC):
    """
    base class for a array of register files
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.SECTION

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: Union[AddressMap, RegFile],
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[RegFile, ...]]] = None):


        if not isinstance(parent, (AddressMap,RegFile)):
            raise TypeError(f'parent should be either AddressMap or RegFile got {type(parent)}')
        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions, elements=elements)


class AsyncRegFileArray(NodeArray[AsyncRegFile], ABC):
    """
    base class for a array of register files
    """
    __slots__: list[str] = []
    _iteration_classification = IterationClassification.SECTION

    # pylint: disable-next=too-many-arguments
    def __init__(self, *,
                 logger_handle: str, inst_name: str,
                 parent: Union[AsyncAddressMap, AsyncRegFile],
                 address: int,
                 stride: int,
                 dimensions: tuple[int, ...],
                 elements: Optional[tuple[tuple[tuple[int, ...], ...],
                 tuple[AsyncRegFile, ...]]] = None):

        if not isinstance(parent, (AsyncAddressMap,AsyncRegFile)):
            raise TypeError(f'parent should be either AsyncAddressMap or AsyncRegFile '
                            f'got {type(parent)}')

        super().__init__(logger_handle=logger_handle, inst_name=inst_name,
                         parent=parent, address=address,
                         stride=stride, dimensions=dimensions, elements=elements)
