"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2025

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as
published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This package is intended to distributed as part of automatically generated code by the PeakRDL
Python tool. It provide the base class for the autogenerated tests
"""
from abc import ABC, abstractmethod
from typing import Union, Optional
from unittest.mock import patch, Mock, call
from itertools import product, chain, combinations
from collections.abc import Iterable
from array import array as Array

from ..lib import FieldReadWrite, FieldReadOnly, FieldWriteOnly
from ..lib import FieldEnumReadWrite, FieldEnumReadOnly, FieldEnumWriteOnly
from ..lib import RegReadOnly, RegReadWrite, RegWriteOnly
from ..lib import MemoryReadOnly, MemoryReadOnlyLegacy
from ..lib import MemoryWriteOnly, MemoryWriteOnlyLegacy
from ..lib import MemoryReadWrite, MemoryReadWriteLegacy
from ..lib import RegisterWriteVerifyError
from ..sim_lib.register import Register as SimRegister
from ..sim_lib.register import MemoryRegister as SimMemoryRegister
from ..sim_lib.register import Field as SimField
from ..sim_lib.memory import Memory as SimMemory

from .utilities import reverse_bits, expected_reg_write_data
from .utilities import reg_value_for_field_read_with_random_base
from .utilities import random_int_field_value, random_field_parent_reg_value
from .utilities import random_encoded_field_value, reg_value_for_field_read
from .utilities import random_reg_value, RandomReg
from .utilities import RegWriteTestSequence,RegWriteZeroStartTestSequence
from .utilities import random_memory_entry,random_memory_entry_value

from ._common_base_test_class import CommonTestBase, NodeIterators

# This module is planned to be split, see #272, for now the length is supressed
# pylint:disable=too-many-lines


class LibTestBase(CommonTestBase, ABC):
    """
    Base Test class for the autogenerated register test when in async mode
    """

    # The following may look odd by a second layer of indirection is required to effectively patch
    # the read and write within tests

    # pylint:disable=missing-function-docstring

    def outer_read_callback(self, addr: int, width: int, accesswidth: int) -> int:
        return self.read_callback(addr=addr, width=width, accesswidth=accesswidth)

    @abstractmethod
    def read_callback(self, addr: int, width: int, accesswidth: int) -> int:
        ...

    def outer_write_callback(self, addr: int, width: int, accesswidth: int,  data: int) -> None:
        return self.write_callback(addr=addr, width=width, accesswidth=accesswidth, data=data)

    @abstractmethod
    def write_callback(self, addr: int, width: int, accesswidth: int,  data: int) -> None:
        ...

    # pylint:enable=missing-function-docstring

    def __single_int_field_read_test(self, fut: Union[FieldReadOnly, FieldReadOnly]) -> None:

        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback', return_value=0) as read_callback_mock:

            # read back - zero, this is achieved by setting the register to inverse bitmask
            read_callback_mock.return_value = fut.inverse_bitmask
            self.assertEqual(fut.read(), 0)
            read_callback_mock.assert_called_once_with(
                addr=fut.parent_register.address,
                width=fut.parent_register.width,
                accesswidth=fut.parent_register.accesswidth)

            # read back - max_value, this is achieved by setting the register to bitmask
            read_callback_mock.reset_mock()
            read_callback_mock.return_value = fut.bitmask
            self.assertEqual(fut.read(), fut.max_value)
            read_callback_mock.assert_called_once_with(
                addr=fut.parent_register.address,
                width=fut.parent_register.width,
                accesswidth=fut.parent_register.accesswidth)

            # read back - random value
            read_callback_mock.reset_mock()
            random_value = random_field_parent_reg_value(fut)
            read_callback_mock.return_value = random_value
            field_value = (random_value & fut.bitmask) >> fut.low
            if fut.msb == fut.high:
                self.assertEqual(fut.read(), field_value)
            else:
                self.assertEqual(fut.read(),
                                 reverse_bits(value=field_value, number_bits=fut.width))
            read_callback_mock.assert_called_once_with(
                addr=fut.parent_register.address,
                width=fut.parent_register.width,
                accesswidth=fut.parent_register.accesswidth)

            # at the end of the read tests the write should not have been called
            write_callback_mock.assert_not_called()

    def __single_int_field_write_test(self, fut: Union[FieldReadOnly, FieldWriteOnly]) -> None:

        # pylint:disable-next=protected-access
        readable_reg = fut.parent_register._is_readable

        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback', return_value=0) as read_callback_mock:
            if not isinstance(fut, (FieldWriteOnly, FieldReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a writable field')

            for reg_base_value,field_value in product(
                    [0, fut.parent_register.max_value, random_field_parent_reg_value(fut)],
                    [0, fut.max_value, random_int_field_value(fut)]):
                read_callback_mock.reset_mock()
                write_callback_mock.reset_mock()
                read_callback_mock.return_value = reg_base_value

                fut.write(field_value)

                # special case if the field is the full size of the register, the read is skipped
                # similarly, if the register is not readable it is skipped
                if (fut.width < fut.parent_register.width) and readable_reg:
                    read_callback_mock.assert_called_once_with(
                        addr=fut.parent_register.address,
                        width=fut.parent_register.width,
                        accesswidth=fut.parent_register.accesswidth)
                else:
                    read_callback_mock.assert_not_called()

                write_callback_mock.assert_called_once_with(
                    addr=fut.parent_register.address,
                    width=fut.parent_register.width,
                    accesswidth=fut.parent_register.accesswidth,
                    data=expected_reg_write_data(fut=fut,
                                                 reg_base_value=reg_base_value,
                                                 readable_reg=readable_reg,
                                                 field_value=field_value))

            # check invalid write values bounce
            with self.assertRaises(ValueError):
                fut.write(fut.max_value + 1)

            with self.assertRaises(ValueError):
                fut.write(-1)

    def _single_int_field_read_and_write_test(
            self,
            fut: Union[FieldReadOnly, FieldReadOnly, FieldWriteOnly],
            is_sw_readable: bool,
            is_sw_writable: bool) -> None:
        """
        Check the ability to read and write to integer (non-encoded) field
        """

        # the lsb, msb, high, low, bitmask, inv_bitmask and max_value are all checked as part of
        # `test_field_properties` so do not need to checked here. Similarly, the properties of
        # parent register are checked as part of `test_register_properties`

        self.__single_int_field_simulator_read_and_write_test(fut=fut,
                                                              is_sw_readable=is_sw_readable,
                                                              is_sw_writable=is_sw_writable)

        if is_sw_readable:
            if not isinstance(fut, (FieldReadOnly, FieldReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a readable field')
            self.__single_int_field_read_test(fut=fut)

        if is_sw_writable:
            if not isinstance(fut, (FieldWriteOnly, FieldReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a writable field')
            self.__single_int_field_write_test(fut=fut)

    def __single_enum_field_read_test(self,
                                      fut: Union[FieldEnumReadOnly, FieldEnumReadOnly],
                                      enum_definition: dict[str, int],
                                      ) -> None:

        # pylint does not realise this is a class being returned rather than an object, so
        # is unhappy with the name
        # pylint:disable-next=invalid-name
        EnumCls = fut.enum_cls

        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback', return_value=0) as read_callback_mock:

            # read back - each legal enum value
            for enum_name, enum_value in enum_definition.items():
                read_callback_mock.reset_mock()
                reg_value = reg_value_for_field_read_with_random_base(fut=fut,
                                                                      field_value=enum_value)
                read_callback_mock.return_value = reg_value
                self.assertEqual(fut.read(), EnumCls[enum_name])
                read_callback_mock.assert_called_once_with(
                    addr=fut.parent_register.address,
                    width=fut.parent_register.width,
                    accesswidth=fut.parent_register.accesswidth)

            # check register values that don't map to a legal enum value create an exception
            # this check is only relevant if there are potential field values that do not map to
            # the enum
            if len(enum_definition) < (2**fut.width):
                # there are two versions of this:
                # 1) for small fields (up to 8 bit wide) every value is tested
                # 2) for large fields (typically occurring with a sparse enum 100 values are
                #    checked
                legal_enum_values_set = set(enum_definition.values())
                if fut.width <= 8:
                    bad_field_value_iter = set(range(fut.max_value+1))
                else:
                    bad_field_value_iter = {random_int_field_value(fut) for _ in range(100)}

                for bad_field_value in bad_field_value_iter - legal_enum_values_set:
                    read_callback_mock.reset_mock()
                    reg_value = reg_value_for_field_read_with_random_base(
                        fut=fut,
                        field_value=bad_field_value)
                    read_callback_mock.return_value = reg_value
                    with self.assertRaises(ValueError):
                        _ = fut.read()
                    read_callback_mock.assert_called_once_with(
                        addr=fut.parent_register.address,
                        width=fut.parent_register.width,
                        accesswidth=fut.parent_register.accesswidth)

            # at the end of the read tests the write should not have been called
            write_callback_mock.assert_not_called()

    def __single_enum_field_write_test(self,
                                      fut: Union[FieldEnumWriteOnly, FieldEnumReadWrite],
                                      enum_definition: dict[str, int]) -> None:

        # pylint does not realise this is a class being returned rather than an object, so
        # is unhappy with the name
        # pylint:disable-next=invalid-name
        EnumCls = fut.enum_cls

        # pylint:disable-next=protected-access
        readable_reg = fut.parent_register._is_readable

        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback', return_value=0) as read_callback_mock:

            for enum_name, enum_value in enum_definition.items():
                reg_value = random_field_parent_reg_value(fut)
                read_callback_mock.return_value = reg_value
                fut.write(EnumCls[enum_name])

                # the read is skipped if the register is not readable or has the same width
                # as the field
                if (fut.width < fut.parent_register.width) and readable_reg:
                    read_callback_mock.assert_called_once_with(
                        addr=fut.parent_register.address,
                        width=fut.parent_register.width,
                        accesswidth=fut.parent_register.accesswidth)
                else:
                    read_callback_mock.assert_not_called()

                write_callback_mock.assert_called_once_with(
                    addr=fut.parent_register.address,
                    width=fut.parent_register.width,
                    accesswidth=fut.parent_register.accesswidth,
                    data=expected_reg_write_data(fut=fut,
                                                 reg_base_value=reg_value,
                                                 readable_reg=readable_reg,
                                                 field_value=enum_value))

                read_callback_mock.reset_mock()
                write_callback_mock.reset_mock()

    def _single_enum_field_read_and_write_test(
            self,
            fut: Union[FieldEnumReadOnly, FieldEnumReadOnly, FieldEnumWriteOnly],
            full_enum_def: dict[str, tuple[int, Optional[str], Optional[str]]],
            is_sw_readable: bool,
            is_sw_writable: bool) -> None:
        """
        Check the ability to read and write to integer (non-encoded) field
        """

        # the lsb, msb, high, low, bitmask, inv_bitmask and max_value are all checked as part of
        # `test_field_properties` so do not need to checked here. Similarly, the properties of
        # parent register are checked as part of `test_register_properties`

        self.__single_enum_field_simulator_read_and_write_test(fut=fut,
                                                               is_sw_readable=is_sw_readable,
                                                               is_sw_writable=is_sw_writable)

        # split the enum definition from the full enum definition
        self._test_enum_def_rdl_name_desc_(fut=fut, full_enum_def=full_enum_def)
        enum_definition = self._full_to_reduced_enum_conversion(full_enum_def)

        if is_sw_readable:
            if not isinstance(fut, (FieldEnumReadOnly, FieldEnumReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a readable field')
            self.__single_enum_field_read_test(fut=fut,
                                               enum_definition=enum_definition)

        if is_sw_writable:
            if not isinstance(fut, (FieldEnumWriteOnly, FieldEnumReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a writable field')
            self.__single_enum_field_write_test(fut=fut,
                                                enum_definition=enum_definition)

    def _single_register_read_and_write_test(self, *,
                                             rut: Union[RegReadOnly, RegReadWrite, RegWriteOnly],
                                             has_sw_readable: bool,
                                             has_sw_writable: bool,
                                             readable_fields: set[str],
                                             writeable_fields: set[str]) -> None:

        # the register properties are tested separately so are available to be used here

        self._test_field_iterators(rut=rut,
                                   has_sw_readable=has_sw_readable,
                                   has_sw_writable=has_sw_writable,
                                   readable_fields=readable_fields,
                                   writeable_fields=writeable_fields)

        self.__single_register_simulator_read_and_write_test(rut=rut,
                                                             has_sw_readable=has_sw_readable,
                                                             has_sw_writable=has_sw_writable)

        if has_sw_readable:
            if not isinstance(rut, (RegReadOnly, RegReadWrite)):
                raise TypeError('Test can not proceed as the rut is not a readable register')
            self.__single_reg_read_test(rut=rut)
            # check the read fields and read context manager
            self.__single_reg_read_fields_and_context_test(rut=rut)
        else:
            # test that a non-readable register has no read method and
            # attempting one generates and error
            with self.assertRaises(AttributeError):
                _= rut.read(0)  # type: ignore[union-attr,call-arg]

        if has_sw_writable:
            if not isinstance(rut, (RegWriteOnly, RegReadWrite)):
                raise TypeError('Test can not proceed as the rut is not a writable register')
            self.__single_reg_write_test(rut=rut)
            if has_sw_readable:
                if not isinstance(rut, RegReadWrite):
                    raise TypeError('Test can not proceed as the rut is not a read '
                                    'and writable register')
                self.__single_reg_write_fields_and_context_test(rut)
            else:
                if not isinstance(rut, RegWriteOnly):
                    raise TypeError('Test can not proceed as the rut is not a writable register')
                self.__single_reg_full_write_fields_test(rut)

        else:
            # test that a non-writable register has no write method and
            # attempting one generates and error
            with self.assertRaises(AttributeError):
                rut.write(0)  # type: ignore[union-attr,call-arg]

    def __single_reg_read_test(self, rut: Union[RegReadOnly, RegReadWrite]) -> None:

        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback', return_value=1) as read_callback_mock:
            for reg_value in [0, 1, rut.max_value, random_reg_value(rut)]:
                read_callback_mock.reset_mock()
                read_callback_mock.return_value = reg_value
                self.assertEqual(rut.read(), reg_value)
                read_callback_mock.assert_called_once_with(
                    addr=rut.address,
                    width=rut.width,
                    accesswidth=rut.accesswidth)
            write_callback_mock.assert_not_called()

    def __single_reg_write_test(self, rut: Union[RegWriteOnly, RegReadWrite]) -> None:
        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback', return_value=0) as read_callback_mock:
            for reg_value in [0, 1, rut.max_value, random_reg_value(rut)]:
                write_callback_mock.reset_mock()
                read_callback_mock.return_value = reg_value
                rut.write(reg_value)
                write_callback_mock.assert_called_once_with(
                    addr=rut.address,
                    width=rut.width,
                    accesswidth=rut.accesswidth,
                    data=reg_value)
            read_callback_mock.assert_not_called()

            # test writing a value beyond the register range is blocked with an exception
            # being raised
            with self.assertRaises(ValueError):
                rut.write(-1)

            with self.assertRaises(ValueError):
                rut.write(rut.max_value + 1)

    def __single_reg_read_fields_and_context_test(self,
                                                  rut: Union[RegReadOnly, RegReadWrite]) -> None:
        """
        Check the `read_fields` and `single_read` methods
        """

        # build up a register value, starting with a random register value
        reg_value = RandomReg(rut).value

        with patch.object(self, 'write_callback') as write_callback_mock, \
            patch.object(self, 'read_callback', return_value=reg_value) as read_callback_mock:

            # build the expected return structure
            ref_read_fields = { field.inst_name: field.read() for field in rut.readable_fields }
            read_callback_mock.reset_mock()

            self.assertDictEqual(rut.read_fields(), ref_read_fields)
            read_callback_mock.assert_called_once_with(
                addr=rut.address,
                width=rut.width,
                accesswidth=rut.accesswidth)
            read_callback_mock.reset_mock()

            with rut.single_read() as rut_context_inst:
                context_ref_read_fields = {field.inst_name: field.read()
                                           for field in rut_context_inst.readable_fields}
            self.assertDictEqual(ref_read_fields, context_ref_read_fields)
            read_callback_mock.assert_called_once_with(
                addr=rut.address,
                width=rut.width,
                accesswidth=rut.accesswidth)

            write_callback_mock.assert_not_called()

    def __single_reg_write_fields_and_context_test(self, rut: RegReadWrite) -> None:
        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback', return_value=0) as read_callback_mock:
            # fix for #196 (excessive test time) if the number of fields is greater than 4
            # the combinations are reduced to only tests combinations of three plus the full
            # set
            num_writable_fields = len(list(rut.writable_fields))
            if num_writable_fields > 4:
                perms_iterator: Iterable[int] = chain(range(1, 4), [num_writable_fields])
            else:
                perms_iterator = range(1, num_writable_fields + 1)
            for fields_to_write in chain.from_iterable(
                    (combinations(rut.writable_fields, perms) for perms in perms_iterator)):

                reg_sequence = RegWriteTestSequence(rut, fields=fields_to_write)

                # read/write without verify
                read_callback_mock.return_value = reg_sequence.start_value
                with rut.single_read_modify_write(verify=False) as reg_session:
                    for field_name, field_value in reg_sequence.write_sequence.items():
                        field = reg_session.get_child_by_system_rdl_name(field_name)
                        field.write(field_value)

                write_callback_mock.assert_called_once_with(
                    addr=rut.address,
                    width=rut.width,
                    accesswidth=rut.accesswidth,
                    data=reg_sequence.value)
                read_callback_mock.assert_called_once()
                write_callback_mock.reset_mock()
                read_callback_mock.reset_mock()

                # read/write/verify pass
                with rut.single_read_modify_write(verify=True) as reg_session:
                    for field_name, field_value in reg_sequence.write_sequence.items():
                        field = reg_session.get_child_by_system_rdl_name(field_name)
                        field.write(field_value)
                    read_callback_mock.return_value = reg_sequence.value

                write_callback_mock.assert_called_once_with(
                    addr=rut.address,
                    width=rut.width,
                    accesswidth=rut.accesswidth,
                    data=reg_sequence.value)
                self.assertEqual(read_callback_mock.call_count, 2)
                write_callback_mock.reset_mock()
                read_callback_mock.reset_mock()

                # read/write/verify pass
                with self.assertRaises(RegisterWriteVerifyError):
                    with rut.single_read_modify_write(verify=True) as reg_session:
                        for field_name, field_value in reg_sequence.write_sequence.items():
                            field = reg_session.get_child_by_system_rdl_name(field_name)
                            field.write(field_value)
                        read_callback_mock.return_value = reg_sequence.value ^ reg_session.max_value

                write_callback_mock.reset_mock()
                read_callback_mock.reset_mock()

                # check the write_fields
                read_callback_mock.return_value = reg_sequence.start_value
                # make the kwargs by replacing the field names with the safe versions
                kwargs = { rut.systemrdl_python_child_name_map[unsafe_name] : value
                           for unsafe_name, value in reg_sequence.write_sequence.items() }
                rut.write_fields(**kwargs)
                write_callback_mock.assert_called_once_with(
                    addr=rut.address,
                    width=rut.width,
                    accesswidth=rut.accesswidth,
                    data=reg_sequence.value)
                read_callback_mock.assert_called_once()
                write_callback_mock.reset_mock()
                read_callback_mock.reset_mock()

    def __single_reg_full_write_fields_test(self, rut: RegWriteOnly) -> None:
        """
        Test the `write_fields` method of a Write Only Register
        """
        with patch.object(self, 'write_callback') as write_callback_mock, \
                patch.object(self, 'read_callback') as read_callback_mock:
            # in the case of a write only register the only legal case is a full field write in
            # one go
            reg_sequence = RegWriteZeroStartTestSequence(rut, fields=rut.writable_fields)

            # make the kwargs by replacing the field names with the safe versions
            kwargs = { rut.systemrdl_python_child_name_map[unsafe_name] : value
                       for unsafe_name, value in reg_sequence.write_sequence.items() }
            rut.write_fields(**kwargs)
            write_callback_mock.assert_called_once_with(
                addr=rut.address,
                width=rut.width,
                accesswidth=rut.accesswidth,
                data=reg_sequence.value)
            read_callback_mock.assert_not_called()
            write_callback_mock.reset_mock()
            read_callback_mock.reset_mock()

    def __single_register_simulator_read_and_write_test(
            self,
            rut: Union[RegReadOnly, RegReadWrite, RegWriteOnly],
            has_sw_readable: bool,
            has_sw_writable: bool) -> None:

        sim_register = self.simulator_instance.register_by_full_name(rut.full_inst_name)

        self.assertIsInstance(sim_register, (SimRegister, SimMemoryRegister))
        register_read_callback = Mock()
        register_write_callback = Mock()

        if has_sw_readable:
            if not isinstance(rut, (RegReadOnly, RegReadWrite)):
                raise TypeError('Test can not proceed as the rut is not a readable register')
            # register read checks
            # update the value via the backdoor in the simulator
            random_value = random_reg_value(rut)
            sim_register.value = random_value
            self.assertEqual(rut.read(), random_value)
            # up to now the callback should not have been called
            sim_register.read_callback = register_read_callback
            sim_register.write_callback = register_write_callback
            random_value = random_reg_value(rut)
            sim_register.value = random_value
            self.assertEqual(rut.read(), random_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_called_once_with(value=random_value)
            register_write_callback.reset_mock()
            register_read_callback.reset_mock()
            sim_register.value = random_value
            sim_register.read_callback = None
            sim_register.write_callback = None
            self.assertEqual(rut.read(), random_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_not_called()

        if has_sw_writable:
            if not isinstance(rut, (RegWriteOnly, RegReadWrite)):
                raise TypeError('Test can not proceed as the rut is not a writable register')
            # register write checks
            random_value = random_reg_value(rut)
            rut.write(random_value)
            self.assertEqual(sim_register.value, random_value)
            # up to now the callback should not have been called
            sim_register.read_callback = register_read_callback
            sim_register.write_callback = register_write_callback
            random_value = random_reg_value(rut)
            rut.write(random_value)
            self.assertEqual(sim_register.value, random_value)
            register_write_callback.assert_called_once_with(value=random_value)
            register_read_callback.assert_not_called()
            register_write_callback.reset_mock()
            register_read_callback.reset_mock()
            sim_register.read_callback = None
            sim_register.write_callback = None
            random_value = random_reg_value(rut)
            rut.write(random_value)
            self.assertEqual(sim_register.value, random_value)
            if has_sw_readable:
                if not isinstance(rut, RegReadWrite):
                    raise TypeError('Test can not proceed as the rut is not a read '
                                    'and writable register')
                self.assertEqual(rut.read(), random_value)

    def __single_int_field_simulator_read_and_write_test(
            self,
            fut: Union[FieldReadOnly, FieldWriteOnly, FieldReadWrite],
            is_sw_readable: bool,
            is_sw_writable: bool) -> None:
        #pylint:disable=too-many-statements

        sim_register = self.simulator_instance.register_by_full_name(
            fut.parent_register.full_inst_name)
        self.assertIsInstance(sim_register, (SimRegister, SimMemoryRegister))
        sim_field = self.simulator_instance.field_by_full_name(fut.full_inst_name)
        self.assertIsInstance(sim_field, SimField)
        register_read_callback = Mock()
        register_write_callback = Mock()
        field_read_callback = Mock()
        field_write_callback = Mock()

        # pylint:disable-next=protected-access
        readable_reg = fut.parent_register._is_readable

        if is_sw_readable:
            # register read checks
            # update the register value via the backdoor in the simulator
            if not isinstance(fut, (FieldReadOnly, FieldReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a readable field')

            random_field_value = random_int_field_value(fut)
            random_value = reg_value_for_field_read_with_random_base(
                fut=fut,
                field_value=random_field_value)
            sim_register.value = random_value
            self.assertEqual(fut.read(), random_field_value)
            # update the field value via the backdoor in the simulator
            previous_register_value = random_value

            random_field_value = random_int_field_value(fut)
            sim_field.value = random_field_value
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=previous_register_value,
                field_value=random_field_value)
            self.assertEqual(sim_register.value, random_value)
            self.assertEqual(fut.read(), random_field_value)
            # hook up the callbacks to check they work correctly
            random_field_value = random_int_field_value(fut)
            random_value = reg_value_for_field_read_with_random_base(
                fut=fut,
                field_value=random_field_value)
            sim_register.value = random_value
            sim_register.read_callback = register_read_callback
            sim_register.write_callback = register_write_callback
            sim_field.read_callback = field_read_callback
            sim_field.write_callback = field_write_callback
            self.assertEqual(fut.read(), random_field_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_called_once_with(value=random_value)
            field_write_callback.assert_not_called()
            field_read_callback.assert_called_once_with(value=random_field_value)
            # revert the callbacks and check again
            register_write_callback.reset_mock()
            register_read_callback.reset_mock()
            field_write_callback.reset_mock()
            field_read_callback.reset_mock()
            sim_register.read_callback = None
            sim_register.write_callback = None
            sim_field.read_callback = None
            sim_field.write_callback = None
            #random_field_value = random_int_field_value(fut)
            random_value = reg_value_for_field_read_with_random_base(
                fut=fut,
                field_value=random_field_value)
            sim_register.value = random_value
            self.assertEqual(fut.read(), random_field_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_not_called()
            field_write_callback.assert_not_called()
            field_read_callback.assert_not_called()

        if is_sw_writable:
            # register write checks
            # update the register value via the backdoor in the simulator, then perform a field
            # write and make sure it is updated

            if not isinstance(fut, (FieldWriteOnly, FieldReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a writable field')

            if readable_reg:
                initial_reg_random_value = random_field_parent_reg_value(fut)
                sim_register.value = initial_reg_random_value
            else:
                # if the register is not readable the write assumes the rest of the register is 0
                initial_reg_random_value = 0

            random_field_value = random_int_field_value(fut)
            sim_field.value = random_field_value
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=initial_reg_random_value,
                field_value=random_field_value)
            fut.write(random_field_value)
            self.assertEqual(sim_register.value, random_value)

            register_write_callback.assert_not_called()
            register_read_callback.assert_not_called()
            field_write_callback.assert_not_called()
            field_read_callback.assert_not_called()
            # hook up the call backs
            sim_register.read_callback = None
            sim_register.write_callback = register_write_callback
            sim_field.read_callback = None
            sim_field.write_callback = field_write_callback
            random_field_value = random_int_field_value(fut)
            fut.write(random_field_value)
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=initial_reg_random_value,
                field_value=random_field_value)
            self.assertEqual(sim_register.value, random_value)
            register_write_callback.assert_called_once_with(
                value=random_value)
            field_write_callback.assert_called_once_with(
                value=random_field_value)
            register_read_callback.assert_not_called()
            field_read_callback.assert_not_called()
            # revert the callbacks and check again
            register_write_callback.reset_mock()
            register_read_callback.reset_mock()
            field_write_callback.reset_mock()
            field_read_callback.reset_mock()
            sim_register.write_callback = None
            sim_field.write_callback = None
            random_field_value = random_int_field_value(fut)
            fut.write(random_field_value)
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=initial_reg_random_value,
                field_value=random_field_value)
            self.assertEqual(sim_register.value, random_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_not_called()
            field_write_callback.assert_not_called()
            field_read_callback.assert_not_called()

    def __single_enum_field_simulator_read_and_write_test(
            self,
            fut: Union[FieldEnumReadOnly, FieldEnumWriteOnly, FieldEnumReadWrite],
            is_sw_readable: bool,
            is_sw_writable: bool) -> None:
        # pylint:disable=too-many-statements

        sim_register = self.simulator_instance.register_by_full_name(
            fut.parent_register.full_inst_name)
        self.assertIsInstance(sim_register, (SimRegister, SimMemoryRegister))
        sim_field = self.simulator_instance.field_by_full_name(fut.full_inst_name)
        self.assertIsInstance(sim_field, SimField)
        register_read_callback = Mock()
        register_write_callback = Mock()
        field_read_callback = Mock()
        field_write_callback = Mock()

        # pylint:disable-next=protected-access
        readable_reg = fut.parent_register._is_readable

        if is_sw_readable:
            # register read checks
            # update the register value via the backdoor in the simulator

            if not isinstance(fut, (FieldEnumReadOnly, FieldEnumReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a readable field')

            random_field_value = random_encoded_field_value(fut)
            random_value = reg_value_for_field_read_with_random_base(
                fut=fut,
                field_value=random_field_value.value)

            sim_register.value = random_value
            self.assertEqual(fut.read(), random_field_value)
            # update the field value via the backdoor in the simulator
            previous_register_value = random_value
            random_field_value = random_encoded_field_value(fut)
            sim_field.value = random_field_value.value
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=previous_register_value,
                field_value=random_field_value.value)
            self.assertEqual(sim_register.value, random_value)
            self.assertEqual(fut.read(), random_field_value)


            # hook up the callbacks to check they work correctly
            random_field_value = random_encoded_field_value(fut)
            random_value = reg_value_for_field_read_with_random_base(
                fut=fut,
                field_value=random_field_value.value)

            sim_register.value = random_value
            sim_register.read_callback = register_read_callback
            sim_register.write_callback = register_write_callback
            sim_field.read_callback = field_read_callback
            sim_field.write_callback = field_write_callback
            self.assertEqual(fut.read(), random_field_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_called_once_with(value=random_value)
            field_write_callback.assert_not_called()
            field_read_callback.assert_called_once_with(value=random_field_value.value)

            # revert the callbacks and check again
            register_write_callback.reset_mock()
            register_read_callback.reset_mock()
            field_write_callback.reset_mock()
            field_read_callback.reset_mock()
            sim_register.read_callback = None
            sim_register.write_callback = None
            sim_field.read_callback = None
            sim_field.write_callback = None
            random_field_value = random_encoded_field_value(fut)
            random_value = reg_value_for_field_read_with_random_base(
                fut=fut,
                field_value=random_field_value.value)

            sim_register.value = random_value
            self.assertEqual(fut.read(), random_field_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_not_called()
            field_write_callback.assert_not_called()
            field_read_callback.assert_not_called()


        if is_sw_writable:
            # register write checks
            # update the register value via the backdoor in the simulator, then perform a field
            # write and make sure it is updated

            if not isinstance(fut, (FieldEnumWriteOnly, FieldEnumReadWrite)):
                raise TypeError('Test can not proceed as the fut is not a writable field')

            if readable_reg:
                initial_reg_random_value = random_field_parent_reg_value(fut)
                sim_register.value = initial_reg_random_value
            else:
                # if the register is not readable the write assumes the rest of the register is 0
                initial_reg_random_value = 0


            random_field_value = random_encoded_field_value(fut)
            sim_field.value = random_field_value.value
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=initial_reg_random_value,
                field_value=random_field_value.value)
            fut.write(random_field_value)


            self.assertEqual(sim_register.value, random_value)

            register_write_callback.assert_not_called()
            register_read_callback.assert_not_called()
            field_write_callback.assert_not_called()
            field_read_callback.assert_not_called()
            # hook up the call backs
            sim_register.read_callback = None
            sim_register.write_callback = register_write_callback
            sim_field.read_callback = None
            sim_field.write_callback = field_write_callback
            random_field_value = random_encoded_field_value(fut)
            fut.write(random_field_value)
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=initial_reg_random_value,
                field_value=random_field_value.value)
            self.assertEqual(sim_register.value, random_value)
            register_write_callback.assert_called_once_with(
                value=random_value)
            field_write_callback.assert_called_once_with(
                value=random_field_value.value)
            register_read_callback.assert_not_called()
            field_read_callback.assert_not_called()
            # revert the callbacks and check again
            register_write_callback.reset_mock()
            register_read_callback.reset_mock()
            field_write_callback.reset_mock()
            field_read_callback.reset_mock()
            sim_register.write_callback = None
            sim_field.write_callback = None
            random_field_value = random_encoded_field_value(fut)
            fut.write(random_field_value)
            random_value = reg_value_for_field_read(
                fut=fut,
                reg_base_value=initial_reg_random_value,
                field_value=random_field_value.value)
            self.assertEqual(sim_register.value, random_value)
            register_write_callback.assert_not_called()
            register_read_callback.assert_not_called()
            field_write_callback.assert_not_called()
            field_read_callback.assert_not_called()

    def _single_memory_read_and_write_test(self, *,
                                           mut: Union[MemoryReadOnly,
                                                      MemoryReadOnlyLegacy,
                                                      MemoryWriteOnly,
                                                      MemoryWriteOnlyLegacy,
                                                      MemoryReadWrite,
                                                      MemoryReadWriteLegacy],
                                           is_sw_readable: bool,
                                           is_sw_writable: bool,
                                           readable_registers: NodeIterators,
                                           writeable_registers: NodeIterators) -> None:

        # the register memory are tested separately so are available to be used here

        self._test_register_iterators(dut=mut,
                                      readable_registers=readable_registers,
                                      writeable_registers=writeable_registers)



        if is_sw_readable:
            if not isinstance(mut, (MemoryReadOnly, MemoryReadOnlyLegacy,
                                    MemoryReadWrite, MemoryReadWriteLegacy)):
                raise TypeError('Test can not proceed as the mut is not a readable memory')
            self.__single_memory_read_test(mut=mut)
            self.__single_memory_simulator_read_test(mut=mut)
        else:
            # test that a non-readable memory has no read method and
            # attempting one generates and error
            with self.assertRaises(AttributeError):
                _= mut.read(0)  # type: ignore[union-attr,call-arg]

        if is_sw_writable:
            if not isinstance(mut, (MemoryWriteOnly, MemoryWriteOnlyLegacy,
                                    MemoryReadWrite, MemoryReadWriteLegacy)):
                raise TypeError('Test can not proceed as the mut is not a writable memory')
            self.__single_memory_write_test(mut=mut)
            self.__single_memory_simulator_write_test(mut=mut)
        else:
            # test that a non-writable memory has no write method and
            # attempting one generates and error
            with self.assertRaises(AttributeError):
                mut.write(0)  # type: ignore[union-attr,call-arg]

    def __single_memory_read_test(
            self,
            mut: Union[MemoryReadOnly, MemoryReadOnlyLegacy,
                       MemoryReadWrite, MemoryReadWriteLegacy]
        ) -> None:

        with patch.object(self, 'write_callback') as write_callback_mock, \
            patch.object(self, 'read_callback', return_value=0) as read_callback_mock:

            # single entry read test
            for entry, value in product(
                    [0, mut.entries-1, random_memory_entry(mut)],
                    [0, mut.max_entry_value, random_memory_entry_value(mut)]):
                read_callback_mock.return_value = value

                if self.legacy_block_access:
                    if not isinstance(mut, (MemoryReadOnlyLegacy, MemoryReadWriteLegacy)):
                        raise TypeError(f'Memory should be legacy type but got {type(mut)}')
                    self.assertEqual(mut.read(start_entry=entry, number_entries=1),
                                     Array(mut.array_typecode, [value]))
                else:
                    if not isinstance(mut, (MemoryReadOnly, MemoryReadWrite)):
                        raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
                    self.assertEqual(mut.read(start_entry=entry, number_entries=1),[value])

                read_callback_mock.assert_called_once_with(
                    addr=mut.address + (entry * mut.width_in_bytes),
                    width=mut.width,
                    accesswidth=mut.accesswidth)
                read_callback_mock.reset_mock()

            # multi-entry read
            # check a multi-entry read, if the memory is small do the entire memory, however, if
            # it is large limit the number of entries to 10
            entries_to_test = mut.entries if mut.entries < 10 else 10
            rand_data_list = [random_memory_entry_value(mut) for _ in range(entries_to_test)]
            # the following needs to have the same parameters as the callback so has some unused
            # args
            # pylint:disable-next=unused-argument
            def read_data_mock(addr: int, width: int, accesswidth: int) -> int:
                mem_entry = (addr - mut.address) // mut.width_in_bytes
                return rand_data_list[mem_entry]
            read_callback_mock.side_effect = read_data_mock

            if self.legacy_block_access:
                if not isinstance(mut, (MemoryReadOnlyLegacy, MemoryReadWriteLegacy)):
                    raise TypeError(f'Memory should be legacy type but got {type(mut)}')
                self.assertEqual(mut.read(start_entry=0,number_entries=entries_to_test),
                                 Array(mut.array_typecode, rand_data_list))
            else:
                if not isinstance(mut, (MemoryReadOnly, MemoryReadWrite)):
                    raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
                self.assertEqual(mut.read(start_entry=0, number_entries=entries_to_test),
                                 rand_data_list)

            write_callback_mock.assert_not_called()

    def __single_memory_write_test(
            self,
            mut: Union[MemoryWriteOnly, MemoryWriteOnlyLegacy,
                       MemoryReadWrite, MemoryReadWriteLegacy]
        ) -> None:

        # this function will simplify once all the legacy modes are removed later so we have
        # allowed more branches for now
        # pylint:disable=too-many-branches

        with patch.object(self, 'write_callback') as write_callback_mock, \
            patch.object(self, 'read_callback', return_value=0) as read_callback_mock:

            # single entry write test
            for entry, value in product(
                    [0, mut.entries - 1, random_memory_entry(mut)],
                    [0, mut.max_entry_value, random_memory_entry_value(mut)]):

                if self.legacy_block_access:
                    if not isinstance(mut, (MemoryWriteOnlyLegacy, MemoryReadWriteLegacy)):
                        raise TypeError(f'Memory should be legacy type but got {type(mut)}')
                    mut.write(start_entry=entry, data = Array(mut.array_typecode, [value]))
                else:
                    if not isinstance(mut, (MemoryWriteOnly, MemoryReadWrite)):
                        raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
                    mut.write(start_entry=entry, data = [value])

                write_callback_mock.assert_called_once_with(
                    addr=mut.address + (entry * mut.width_in_bytes),
                    width=mut.width,
                    accesswidth=mut.accesswidth,
                    data=value)
                write_callback_mock.reset_mock()

            # multi-entry read
            # check a multi-entry read, if the memory is small do the entire memory, however, if
            # it is large limit the number of entries to 10
            entries_to_test = mut.entries if mut.entries < 10 else 10
            rand_data_list = [random_memory_entry_value(mut) for _ in range(entries_to_test)]

            if self.legacy_block_access:
                if not isinstance(mut, (MemoryWriteOnlyLegacy, MemoryReadWriteLegacy)):
                    raise TypeError(f'Memory should be legacy type but got {type(mut)}')
                mut.write(start_entry=0, data=Array(mut.array_typecode, rand_data_list))
            else:
                if not isinstance(mut, (MemoryWriteOnly, MemoryReadWrite)):
                    raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
                mut.write(start_entry=0, data=rand_data_list)

            calls = [call(addr=mut.address + (entry * mut.width_in_bytes),
                          width=mut.width,
                          accesswidth=mut.accesswidth,
                          data=rand_data_list[entry]) for entry in range(entries_to_test)]
            write_callback_mock.assert_has_calls(calls, any_order=False)

            # check invalid write values bounce
            if self.legacy_block_access:
                if not isinstance(mut, (MemoryWriteOnlyLegacy, MemoryReadWriteLegacy)):
                    raise TypeError(f'Memory should be legacy type but got {type(mut)}')
                # depending the hardware array sizes the error may be at the point the array is
                # constructed on internal
                dummy_array = Array(mut.array_typecode, [0])
                if dummy_array.itemsize > mut.width_in_bytes:
                    with self.assertRaises(ValueError):
                        mut.write(start_entry=0, data=Array(mut.array_typecode,
                                                            [mut.max_entry_value + 1]))
                else:
                    with self.assertRaises(OverflowError):
                        mut.write(start_entry=0, data=Array(mut.array_typecode,
                                                            [mut.max_entry_value + 1]))
                with self.assertRaises(OverflowError):
                    mut.write(start_entry=0, data=Array(mut.array_typecode,[-1]))
            else:
                if not isinstance(mut, (MemoryWriteOnly, MemoryReadWrite)):
                    raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
                with self.assertRaises(ValueError):
                    mut.write(start_entry=0, data=[mut.max_entry_value + 1])
                with self.assertRaises(ValueError):
                    mut.write(start_entry=0, data=[-1])

            read_callback_mock.assert_not_called()

    def __single_memory_simulator_read_test(
            self,
            mut: Union[MemoryReadOnly, MemoryReadOnlyLegacy,
                       MemoryReadWrite, MemoryReadWriteLegacy],
        ) -> None:

        sim_memory = self.simulator_instance.memory_by_full_name(mut.full_inst_name)
        self.assertIsInstance(sim_memory, SimMemory)

        # single entry read test
        for entry, value in product(
                [0, mut.entries - 1, random_memory_entry(mut)],
                [0, mut.max_entry_value, random_memory_entry_value(mut)]):

            sim_memory.value[entry] = value

            if self.legacy_block_access:
                if not isinstance(mut, (MemoryReadOnlyLegacy, MemoryReadWriteLegacy)):
                    raise TypeError(f'Memory should be legacy type but got {type(mut)}')
                self.assertEqual(mut.read(start_entry=entry, number_entries=1),
                                 Array(mut.array_typecode, [value]))
            else:
                if not isinstance(mut, (MemoryReadOnly, MemoryReadWrite)):
                    raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
                self.assertEqual(mut.read(start_entry=entry, number_entries=1), [value])

        # multi-entry read
        # check a multi-entry read, if the memory is small do the entire memory, however, if
        # it is large limit the number of entries to 10
        entries_to_test = mut.entries if mut.entries < 10 else 10
        rand_data_list = [random_memory_entry_value(mut) for _ in range(entries_to_test)]
        for entry in range(entries_to_test):
            sim_memory.value[entry] = rand_data_list[entry]

        if self.legacy_block_access:
            if not isinstance(mut, (MemoryReadOnlyLegacy, MemoryReadWriteLegacy)):
                raise TypeError(f'Memory should be legacy type but got {type(mut)}')
            self.assertEqual(mut.read(start_entry=0,number_entries=entries_to_test),
                             Array(mut.array_typecode, rand_data_list))
        else:
            if not isinstance(mut, (MemoryReadOnly, MemoryReadWrite)):
                raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
            self.assertEqual(mut.read(start_entry=0, number_entries=entries_to_test),
                             rand_data_list)

    def __single_memory_simulator_write_test(
            self,
            mut: Union[MemoryWriteOnly, MemoryWriteOnlyLegacy,
                       MemoryReadWrite, MemoryReadWriteLegacy]
        ) -> None:

        sim_memory = self.simulator_instance.memory_by_full_name(mut.full_inst_name)
        self.assertIsInstance(sim_memory, SimMemory)

        # single entry write test
        for entry, value in product(
                [0, mut.entries - 1, random_memory_entry(mut)],
                [0, mut.max_entry_value, random_memory_entry_value(mut)]):

            if self.legacy_block_access:
                if not isinstance(mut, (MemoryWriteOnlyLegacy, MemoryReadWriteLegacy)):
                    raise TypeError(f'Memory should be legacy type but got {type(mut)}')
                mut.write(start_entry=entry, data=Array(mut.array_typecode, [value]))
            else:
                if not isinstance(mut, (MemoryWriteOnly, MemoryReadWrite)):
                    raise TypeError(
                        f'Memory should be non-legacy type but got {type(mut)}')
                mut.write(start_entry=entry, data=[value])

            self.assertEqual(sim_memory.value[entry], value)

        # multi-entry read
        # check a multi-entry read, if the memory is small do the entire memory, however, if
        # it is large limit the number of entries to 10
        entries_to_test = mut.entries if mut.entries < 10 else 10
        rand_data_list = [random_memory_entry_value(mut) for _ in range(entries_to_test)]

        if self.legacy_block_access:
            if not isinstance(mut, (MemoryWriteOnlyLegacy, MemoryReadWriteLegacy)):
                raise TypeError(f'Memory should be legacy type but got {type(mut)}')
            mut.write(start_entry=0, data=Array(mut.array_typecode, rand_data_list))
        else:
            if not isinstance(mut, (MemoryWriteOnly, MemoryReadWrite)):
                raise TypeError(f'Memory should be non-legacy type but got {type(mut)}')
            mut.write(start_entry=0, data=rand_data_list)

        for entry in range(entries_to_test):
            self.assertEqual(sim_memory.value[entry], rand_data_list[entry])
