"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filter = exports.flatten = exports.handler = void 0;
const util_arn_parser_1 = require("@aws-sdk/util-arn-parser");
const auth_app_1 = require("@octokit/auth-app");
const auth_token_1 = require("@octokit/auth-token");
const auth_unauthenticated_1 = require("@octokit/auth-unauthenticated");
const rest_1 = require("@octokit/rest");
const aws_sdk_1 = require("aws-sdk");
const execute_github_api_call_1 = require("./execute-github-api-call");
const get_secret_value_1 = require("./get-secret-value");
const auth_1 = require("../auth");
exports.handler = async (event) => {
    console.log(`Request of type ${event.RequestType} received`);
    console.log("Payload: %j", event);
    const call = parse(event.ResourceProperties[event.RequestType]);
    if (!call) {
        return;
    }
    const octokitOptions = {};
    const { strategy, secret } = event.ResourceProperties.Auth;
    switch (strategy) {
        case auth_1.AuthenticationStrategy.AUTH_APP:
            // https://github.com/octokit/authentication-strategies.js/#github-app-or-installation-authentication
            octokitOptions.authStrategy = auth_app_1.createAppAuth;
            octokitOptions.auth = JSON.parse(await get_secret_value_1.getSecretValue(secret));
            break;
        case auth_1.AuthenticationStrategy.AUTH_TOKEN:
            const parameterArn = util_arn_parser_1.parse(secret);
            const parameterName = parameterArn.resource.replace(/^parameter/, "");
            const ssm = new aws_sdk_1.SSM({ region: parameterArn.region });
            const getParameterResponse = await ssm.getParameter({ Name: parameterName, WithDecryption: true }).promise();
            // https://github.com/octokit/auth-token.js#readme
            octokitOptions.authStrategy = auth_token_1.createTokenAuth(getParameterResponse.Parameter.Value);
            break;
        case auth_1.AuthenticationStrategy.UNAUTHENTICATED:
        // https://github.com/octokit/auth-unauthenticated.js#createunauthenticatedauth-options
        default:
            octokitOptions.authStrategy = auth_unauthenticated_1.createUnauthenticatedAuth;
    }
    const octokit = new rest_1.Octokit(octokitOptions);
    try {
        const response = await execute_github_api_call_1.executeGithubApiCall(octokit, call);
        console.debug("Response: %j", response);
        const flattened = exports.flatten(response.data ?? {});
        const responsePath = call.physicalResourceId?.responsePath;
        const physicalResourceId = responsePath ? flattened[responsePath] : call.physicalResourceId?.id;
        return {
            PhysicalResourceId: physicalResourceId,
            Data: exports.filter(flattened, call.outputPaths),
        };
    }
    catch (e) {
        const err = e;
        if (!call.ignoreErrorCodesMatching || !new RegExp(call.ignoreErrorCodesMatching).test(err.message)) {
            throw e;
        }
    }
    return;
};
/**
 * Parse the GitHub API call passed in ResourceProperties OnCreate, OnUpdate and OnDelete
 *
 * @internal
 */
const parse = (stringified) => {
    return stringified ? JSON.parse(stringified) : undefined;
};
/**
 * Flattens a nested object and materializes the key paths dot separated.
 *
 * @internal
 */
exports.flatten = (object) => {
    const _flatten = (path, child) => {
        if ("object" == typeof child && null != child) {
            return Object.entries(child).reduce((result, [key, value]) => Object.assign(result, _flatten(path.concat([key]), value)), {});
        }
        return { [path.join(".")]: child };
    };
    return _flatten([], object);
};
/**
 * Filters objects keys that starts with key names provided in outputPaths.
 *
 * @internal
 */
exports.filter = (object, outputPaths) => {
    if (!outputPaths) {
        return object;
    }
    const result = {};
    for (const [key, value] of Object.entries(object)) {
        for (const outputPath of outputPaths) {
            if (key.startsWith(outputPath)) {
                result[key] = value;
            }
        }
    }
    return result;
};
//# sourceMappingURL=data:application/json;base64,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