"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubActionsSecretRepository = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const provider_1 = require("./custom-resource-provider/provider");
const github_custom_resource_1 = require("./github-custom-resource");
/**
 * Manage an GitHib Actions Repository secret.
 *
 * ```typescript
 * // The GitHub API authentication secret
 * const auth = secrets_manager.Secret.fromSecretNameV2(scope, "Auth", "cdk-github/github-token");
 *
 * // The AWS SecretsManager Secret to configure as GitHub Action secret.
 * const secret = secrets_manager.Secret.fromSecretNameV2(scope, "Secret", "any-secret/example");
 *
 * new GithubActionsSecretRepository(scope, "GithubRepo", {
 *   owner: "pepperize",
 *   repositoryName: "cdk-github",
 *   secretName: "example",
 *   source: GithubActionsSecret.fromSecretsManager(secret, "some-json-field"),
 *   authOptions: AuthOptions.appAuth(auth),
 *   removalPolicy: RemovalPolicy.DESTROY,
 * });
 * ```
 */
class GithubActionsSecretRepository extends github_custom_resource_1.GithubCustomResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const provider = provider_1.GithubCustomResourceProvider.getOrCreate(this);
        const authOptions = props.authOptions;
        authOptions._grantRead(provider);
        props.source._grantRead(provider);
        const onCreate = {
            endpoint: "actions",
            method: "createOrUpdateRepoSecret",
            parameters: {
                owner: props.owner,
                repository_name: props.repositoryName,
                secret_name: props.secretName,
                value: props.source,
            },
            outputPaths: [],
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${props.owner}::${props.repositoryName}::${props.secretName}`),
        };
        const onDelete = props.removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY
            ? {
                endpoint: "actions",
                method: "deleteOrgSecret",
                parameters: {
                    owner: props.owner,
                    repository_name: props.repositoryName,
                    secret_name: props.secretName,
                },
                outputPaths: [],
            }
            : undefined;
        this._resource = new aws_cdk_lib_1.CustomResource(this, "Resource", {
            serviceToken: provider.serviceToken,
            resourceType: "Custom::GithubActions::RepositorySecret",
            properties: {
                Create: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Update: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Delete: onDelete && aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onDelete) }),
                Auth: props.authOptions._auth,
            },
        });
    }
}
exports.GithubActionsSecretRepository = GithubActionsSecretRepository;
//# sourceMappingURL=data:application/json;base64,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