"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubActionsSecretOrganization = exports.Visibility = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const provider_1 = require("./custom-resource-provider/provider");
const github_custom_resource_1 = require("./github-custom-resource");
var Visibility;
(function (Visibility) {
    Visibility["ALL"] = "all";
    Visibility["PRIVATE"] = "private";
    Visibility["SELECTED"] = "selected";
})(Visibility = exports.Visibility || (exports.Visibility = {}));
/**
 * Manage an GitHib Actions organization secret.
 *
 * ```typescript
 * // The GitHub API authentication secret
 * const auth = secrets_manager.Secret.fromSecretNameV2(scope, "Auth", "cdk-github/github-token");
 *
 * // The AWS SecretsManager Secret to configure as GitHub Action secret.
 * const secret = secrets_manager.Secret.fromSecretNameV2(scope, "Secret", "any-secret/example");
 *
 * new GithubActionsSecretOrganization(scope, "GithubRepo", {
 *   organizationName: "pepperize",
 *   secretName: "example",
 *   source: GithubActionsSecret.fromSecretsManager(secret, "some-json-field"),
 *   visibility: Visibility.ALL,
 *   authOptions: AuthOptions.appAuth(auth),
 *   removalPolicy: RemovalPolicy.DESTROY,
 * });
 * ```
 */
class GithubActionsSecretOrganization extends github_custom_resource_1.GithubCustomResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const provider = provider_1.GithubCustomResourceProvider.getOrCreate(this);
        const authOptions = props.authOptions;
        authOptions._grantRead(provider);
        props.source._grantRead(provider);
        const onCreate = {
            endpoint: "actions",
            method: "createOrUpdateOrgSecret",
            parameters: {
                org: props.organizationName,
                secret_name: props.secretName,
                value: props.source,
                visibility: props.visibility,
            },
            outputPaths: [],
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${props.organizationName}::${props.secretName}`),
        };
        const onDelete = props.removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY
            ? {
                endpoint: "actions",
                method: "deleteOrgSecret",
                parameters: {
                    org: props.organizationName,
                    secret_name: props.secretName,
                },
                outputPaths: [],
            }
            : undefined;
        this._resource = new aws_cdk_lib_1.CustomResource(this, "Resource", {
            serviceToken: provider.serviceToken,
            resourceType: "Custom::GithubActions::OrganizationSecret",
            properties: {
                Create: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Update: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Delete: onDelete && aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onDelete) }),
                Auth: props.authOptions._auth,
            },
        });
    }
}
exports.GithubActionsSecretOrganization = GithubActionsSecretOrganization;
//# sourceMappingURL=data:application/json;base64,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