"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubActionsSecretEnvironment = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const provider_1 = require("./custom-resource-provider/provider");
const github_custom_resource_1 = require("./github-custom-resource");
/**
 * Manage an GitHib Actions environment secret.
 *
 * ```typescript
 * // The GitHub API authentication secret
 * const auth = secrets_manager.Secret.fromSecretNameV2(scope, "Auth", "cdk-github/github-token");
 *
 * // The AWS SecretsManager Secret to configure as GitHub Action secret.
 * const secret = secrets_manager.Secret.fromSecretNameV2(scope, "Secret", "any-secret/example");
 *
 * new GithubActionsSecretEnvironment(scope, "GithubRepo", {
 *   repositoryId: "558989134",
 *   environmentName: "production",
 *   secretName: "example",
 *   source: GithubActionsSecret.fromSecretsManager(secret, "some-json-field"),
 *   authOptions: AuthOptions.appAuth(auth),
 *   removalPolicy: RemovalPolicy.DESTROY,
 * });
 * ```
 */
class GithubActionsSecretEnvironment extends github_custom_resource_1.GithubCustomResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const provider = provider_1.GithubCustomResourceProvider.getOrCreate(this);
        const authOptions = props.authOptions;
        authOptions._grantRead(provider);
        props.source._grantRead(provider);
        const onCreate = {
            endpoint: "actions",
            method: "createOrUpdateEnvironmentSecret", // https://github.com/octokit/plugin-rest-endpoint-methods.js/blob/main/docs/actions/createOrUpdateEnvironmentSecret.md
            parameters: {
                repository_id: props.repositoryId,
                environment_name: props.environmentName,
                secret_name: props.secretName,
                value: props.source,
            },
            outputPaths: [],
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${props.repositoryId}::${props.environmentName}::${props.secretName}`),
        };
        const onDelete = props.removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY
            ? {
                endpoint: "actions",
                method: "deleteOrgSecret", // https://github.com/octokit/plugin-rest-endpoint-methods.js/blob/main/docs/actions/deleteEnvironmentSecret.md
                parameters: {
                    repository_id: props.repositoryId,
                    environment_name: props.environmentName,
                    secret_name: props.secretName,
                },
                outputPaths: [],
            }
            : undefined;
        this._resource = new aws_cdk_lib_1.CustomResource(this, "Resource", {
            serviceToken: provider.serviceToken,
            resourceType: "Custom::GithubActions::EnvironmentSecret",
            properties: {
                Create: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Update: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Delete: onDelete && aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onDelete) }),
                Auth: props.authOptions._auth,
            },
        });
    }
}
exports.GithubActionsSecretEnvironment = GithubActionsSecretEnvironment;
//# sourceMappingURL=data:application/json;base64,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