"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubActionsSecretOrganization = exports.Visibility = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const provider_1 = require("./custom-resource-provider/provider");
const github_custom_resource_1 = require("./github-custom-resource");
var Visibility;
(function (Visibility) {
    Visibility["ALL"] = "all";
    Visibility["PRIVATE"] = "private";
    Visibility["SELECTED"] = "selected";
})(Visibility || (exports.Visibility = Visibility = {}));
/**
 * Manage an GitHib Actions organization secret.
 *
 * ```typescript
 * // The GitHub API authentication secret
 * const auth = secrets_manager.Secret.fromSecretNameV2(scope, "Auth", "cdk-github/github-token");
 *
 * // The AWS SecretsManager Secret to configure as GitHub Action secret.
 * const secret = secrets_manager.Secret.fromSecretNameV2(scope, "Secret", "any-secret/example");
 *
 * new GithubActionsSecretOrganization(scope, "GithubRepo", {
 *   organizationName: "pepperize",
 *   secretName: "example",
 *   source: GithubActionsSecret.fromSecretsManager(secret, "some-json-field"),
 *   visibility: Visibility.ALL,
 *   authOptions: AuthOptions.appAuth(auth),
 *   removalPolicy: RemovalPolicy.DESTROY,
 * });
 * ```
 */
class GithubActionsSecretOrganization extends github_custom_resource_1.GithubCustomResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const provider = provider_1.GithubCustomResourceProvider.getOrCreate(this);
        const authOptions = props.authOptions;
        authOptions._grantRead(provider);
        props.source._grantRead(provider);
        const onCreate = {
            endpoint: "actions",
            method: "createOrUpdateOrgSecret", // https://github.com/octokit/plugin-rest-endpoint-methods.js/blob/main/docs/actions/createOrUpdateOrgSecret.md
            parameters: {
                org: props.organizationName,
                secret_name: props.secretName,
                value: props.source,
                visibility: props.visibility,
            },
            outputPaths: [],
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`${props.organizationName}::${props.secretName}`),
        };
        const onDelete = props.removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY
            ? {
                endpoint: "actions",
                method: "deleteOrgSecret",
                parameters: {
                    org: props.organizationName,
                    secret_name: props.secretName,
                },
                outputPaths: [],
            }
            : undefined;
        this._resource = new aws_cdk_lib_1.CustomResource(this, "Resource", {
            serviceToken: provider.serviceToken,
            resourceType: "Custom::GithubActions::OrganizationSecret",
            properties: {
                Create: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Update: aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onCreate) }),
                Delete: onDelete && aws_cdk_lib_1.Lazy.uncachedString({ produce: () => aws_cdk_lib_1.Stack.of(this).toJsonString(onDelete) }),
                Auth: props.authOptions._auth,
            },
        });
    }
}
exports.GithubActionsSecretOrganization = GithubActionsSecretOrganization;
//# sourceMappingURL=data:application/json;base64,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