from pixelarraycloudstorage.client import AsyncClient
from typing import Dict, Any, Optional, List, Tuple, AsyncGenerator
import os
import aiohttp
import mimetypes
import math
import time


class FileStorageManagerAsync(AsyncClient):
    def __init__(self, api_key: str):
        super().__init__(api_key)

    async def get_namespace(self) -> Optional[str]:
        """
        description:
            获取用户命名空间
        return:
            - namespace: 命名空间
        """
        user_info, success = await self._request("GET", "/api/auth/user_info")
        if not success:
            raise RuntimeError("获取用户信息失败，无法确定命名空间")
        namespace = user_info.get("namespace")
        if not namespace:
            raise RuntimeError("当前账号未绑定命名空间")
        return namespace

    async def upload(
        self,
        file_path: str,
        remote_path: Optional[str] = None,
    ) -> Tuple[Dict[str, Any], bool]:
        """
        description:
            上传文件（合并了初始化、分片上传、完成上传三个步骤）
        parameters:
            file_path: 文件路径（str）
            remote_path: 远端保存路径，缺省时沿用file_path并自动创建多级目录
        return:
            - data: 结果数据
            - success: 是否成功
        """
        final_result: Dict[str, Any] = {}
        final_success = False
        async for progress in self.upload_stream(
            file_path=file_path, remote_path=remote_path
        ):
            event = progress.get("event")
            if event == "error":
                return {}, False
            if event == "complete" and progress.get("success"):
                final_result = progress.get("result", {})
                final_success = True
        return final_result, final_success

    async def upload_stream(
        self,
        file_path: str,
        remote_path: Optional[str] = None,
    ) -> AsyncGenerator[Dict[str, Any], None]:
        """
        description:
            上传文件（流式，返回生成器，包含进度信息）
        parameters:
            remote_path: 远端保存路径（可选，默认沿用file_path）
        """
        chunk_size = 2 * 1024 * 1024  # 2MB
        upload_start = time.perf_counter()
        with open(file_path, "rb") as f:
            file_bytes = f.read()

        total_size = len(file_bytes)

        file_name = remote_path.split("/")[-1]
        mime_type = mimetypes.guess_type(file_path)[0]
        init_data = {
            "filename": file_name,
            "file_type": mime_type,
            "total_size": total_size,
            "file_path": remote_path,
        }
        namespace = await self.get_namespace()
        if namespace:
            init_data["namespace"] = namespace

        init_result, success = await self._request(
            "POST", "/api/file_storage/upload/init", json=init_data
        )
        if not success:
            yield {
                "event": "error",
                "percentage": 0,
                "total_chunks": 0,
                "remaining_chunks": 0,
                "total_bytes": total_size,
                "processed_bytes": 0,
                "speed": 0,
                "message": "初始化上传失败",
                "success": False,
            }
            return

        upload_id = init_result.get("upload_id")
        chunk_urls = init_result.get("chunk_urls", [])
        total_chunks = len(chunk_urls)

        if not upload_id or not chunk_urls:
            yield {
                "event": "error",
                "percentage": 0,
                "total_chunks": 0,
                "remaining_chunks": 0,
                "total_bytes": total_size,
                "processed_bytes": 0,
                "speed": 0,
                "message": "缺少上传ID或分片信息",
                "success": False,
            }
            return

        yield {
            "event": "init",
            "percentage": 0,
            "total_chunks": total_chunks,
            "remaining_chunks": total_chunks,
            "total_bytes": total_size,
            "processed_bytes": 0,
            "speed": 0,
            "message": "初始化完成，开始上传分片",
            "success": True,
        }

        parts: List[Dict[str, Any]] = []
        uploaded_bytes = 0

        async with aiohttp.ClientSession() as session:
            for idx, chunk_info in enumerate(chunk_urls):
                part_number = chunk_info.get("part_number")
                url = chunk_info.get("url")
                start = idx * chunk_size
                end = min(start + chunk_size, total_size)
                chunk_data = file_bytes[start:end]

                if not url or not part_number:
                    percentage = (
                        0
                        if total_size == 0
                        else min((uploaded_bytes / total_size) * 100, 100)
                    )
                    yield {
                        "event": "error",
                        "percentage": percentage,
                        "total_chunks": total_chunks,
                        "remaining_chunks": total_chunks - idx,
                        "total_bytes": total_size,
                        "processed_bytes": uploaded_bytes,
                        "speed": 0,
                        "message": "分片信息缺失",
                        "success": False,
                    }
                    return

                chunk_start = time.perf_counter()
                try:
                    async with session.put(url, data=chunk_data) as resp:
                        if resp.status != 200:
                            raise RuntimeError(f"分片上传失败，状态码：{resp.status}")
                        etag = resp.headers.get("ETag", "").strip('"')
                        parts.append(
                            {
                                "part_number": part_number,
                                "etag": etag,
                            }
                        )
                except Exception as exc:
                    percentage = (
                        0
                        if total_size == 0
                        else min((uploaded_bytes / total_size) * 100, 100)
                    )
                    yield {
                        "event": "error",
                        "percentage": percentage,
                        "total_chunks": total_chunks,
                        "remaining_chunks": max(total_chunks - idx, 0),
                        "total_bytes": total_size,
                        "processed_bytes": uploaded_bytes,
                        "speed": 0,
                        "message": f"分片上传异常：{exc}",
                        "success": False,
                    }
                    return

                uploaded_bytes += len(chunk_data)
                duration = max(time.perf_counter() - chunk_start, 1e-6)
                speed = len(chunk_data) / duration
                percentage = (
                    100
                    if total_size == 0
                    else min((uploaded_bytes / total_size) * 100, 100)
                )

                yield {
                    "event": "chunk",
                    "percentage": percentage,
                    "total_chunks": total_chunks,
                    "remaining_chunks": max(total_chunks - (idx + 1), 0),
                    "total_bytes": total_size,
                    "processed_bytes": uploaded_bytes,
                    "chunk_index": idx,
                    "chunk_size": len(chunk_data),
                    "speed": speed,
                    "message": f"分片{idx + 1}/{total_chunks}上传完成",
                    "success": True,
                }

        complete_data = {
            "upload_id": upload_id,
            "parts": parts,
        }
        complete_result, success = await self._request(
            "POST", "/api/file_storage/upload/complete", json=complete_data
        )
        if not success:
            yield {
                "event": "error",
                "percentage": 100,
                "total_chunks": total_chunks,
                "remaining_chunks": 0,
                "total_bytes": total_size,
                "processed_bytes": total_size,
                "speed": 0,
                "message": "完成上传失败",
                "success": False,
            }
            return

        total_duration = max(time.perf_counter() - upload_start, 1e-6)
        yield {
            "event": "complete",
            "percentage": 100,
            "total_chunks": total_chunks,
            "remaining_chunks": 0,
            "total_bytes": total_size,
            "processed_bytes": total_size,
            "speed": total_size / total_duration if total_duration else 0,
            "message": "上传完成",
            "success": True,
            "result": complete_result,
        }

    async def list_files(
        self,
        remote_path: Optional[str] = None,
        page: int = 1,
        page_size: int = 50,
    ) -> Tuple[List[Dict[str, Any]], bool]:
        """
        description:
            获取文件列表
        parameters:
            remote_path: 目标文件夹/文件的远端路径（可选，默认根目录）
            page: 页码（可选）
            page_size: 每页数量（可选）
        return:
            - data: 文件列表数据
            - success: 是否成功
        """
        data = {
            "page": page,
            "page_size": page_size,
        }
        if remote_path is not None:
            data["remote_path"] = remote_path
        namespace = await self.get_namespace()
        if namespace:
            data["namespace"] = namespace

        result, success = await self._request(
            "POST", "/api/file_storage/files/list", json=data
        )
        if not success:
            return {}, False
        return result, True

    async def delete_file(
        self,
        remote_path: str,
    ) -> Tuple[Dict[str, Any], bool]:
        """
        description:
            删除文件或文件夹
        parameters:
            remote_path: 目标文件/文件夹的远端路径
        return:
            - data: 结果数据
            - success: 是否成功
        """
        payload: Dict[str, Any] = {"remote_path": remote_path}
        namespace = await self.get_namespace()
        if namespace:
            payload["namespace"] = namespace
        data, success = await self._request(
            "POST",
            "/api/file_storage/files/delete_by_path",
            json=payload,
        )
        if not success:
            return {}, False
        return data, True

    async def download(
        self,
        remote_path: str,
        save_path: str,
    ) -> Tuple[Dict[str, Any], bool]:
        """
        description:
            下载文件
        parameters:
            file_path: 文件在云端的file_path
            save_path: 保存路径
        return:
            - data: 下载结果数据
            - success: 是否成功
        """
        final_result: Dict[str, Any] = {}
        final_success = False
        async for progress in self.download_stream(remote_path, save_path):
            event = progress.get("event")
            if event == "error":
                return {}, False
            if event == "complete" and progress.get("success"):
                final_result = progress.get("result", {})
                final_success = True
        return final_result, final_success

    async def download_stream(
        self,
        remote_path: str,
        save_path: str,
    ) -> AsyncGenerator[Dict[str, Any], None]:
        """
        description:
            下载文件（流式，返回生成器）
        """
        chunk_size = 2 * 1024 * 1024
        signed_url_data, success = await self._request(
            "POST",
            "/api/file_storage/files/download_by_path",
            json={"file_path": remote_path},
        )
        if not success:
            yield {
                "event": "error",
                "percentage": 0,
                "total_chunks": 0,
                "remaining_chunks": 0,
                "total_bytes": 0,
                "processed_bytes": 0,
                "speed": 0,
                "message": "生成签名URL失败",
                "success": False,
            }
            return

        signed_url = signed_url_data.get("signed_url")
        file_record = signed_url_data.get("file_record", {}) or {}
        total_size = file_record.get("file_size", 0) or 0

        if not signed_url:
            yield {
                "event": "error",
                "percentage": 0,
                "total_chunks": 0,
                "remaining_chunks": 0,
                "total_bytes": total_size,
                "processed_bytes": 0,
                "speed": 0,
                "message": "签名URL为空",
                "success": False,
            }
            return

        total_chunks = math.ceil(total_size / chunk_size) if total_size else 0

        yield {
            "event": "init",
            "percentage": 0,
            "total_chunks": total_chunks,
            "remaining_chunks": total_chunks,
            "total_bytes": total_size,
            "processed_bytes": 0,
            "speed": 0,
            "message": "开始下载文件",
            "success": True,
        }

        os.makedirs(os.path.dirname(save_path) or ".", exist_ok=True)

        downloaded_bytes = 0
        chunk_index = 0
        download_start = time.perf_counter()

        async with aiohttp.ClientSession() as session:
            try:
                async with session.get(signed_url) as resp:
                    if resp.status != 200:
                        raise RuntimeError(f"文件下载失败，状态码：{resp.status}")

                    header_size = resp.headers.get("Content-Length")
                    if total_size == 0 and header_size:
                        try:
                            total_size = int(header_size)
                            total_chunks = (
                                math.ceil(total_size / chunk_size) if total_size else 0
                            )
                        except ValueError:
                            total_size = 0

                    with open(save_path, "wb") as f:
                        async for chunk in resp.content.iter_chunked(chunk_size):
                            chunk_start = time.perf_counter()
                            chunk_index += 1
                            downloaded_bytes += len(chunk)
                            f.write(chunk)

                            chunk_duration = max(
                                time.perf_counter() - chunk_start, 1e-6
                            )
                            instant_speed = len(chunk) / chunk_duration
                            percentage = (
                                0
                                if total_size == 0
                                else min((downloaded_bytes / total_size) * 100, 100)
                            )
                            remaining = (
                                max(total_chunks - chunk_index, 0)
                                if total_chunks
                                else 0
                            )

                            yield {
                                "event": "chunk",
                                "percentage": percentage,
                                "total_chunks": total_chunks,
                                "remaining_chunks": remaining,
                                "total_bytes": total_size,
                                "processed_bytes": downloaded_bytes,
                                "chunk_index": chunk_index - 1,
                                "chunk_size": len(chunk),
                                "speed": instant_speed,
                                "message": f"分片{chunk_index}/{total_chunks or '?'}下载完成",
                                "success": True,
                            }
            except Exception as exc:
                yield {
                    "event": "error",
                    "percentage": 0,
                    "total_chunks": total_chunks,
                    "remaining_chunks": total_chunks,
                    "total_bytes": total_size,
                    "processed_bytes": downloaded_bytes,
                    "speed": 0,
                    "message": f"下载过程中发生错误：{exc}",
                    "success": False,
                }
                return

        total_duration = max(time.perf_counter() - download_start, 1e-6)
        result = {
            "total_size": total_size,
            "success": True,
        }
        yield {
            "event": "complete",
            "percentage": 100,
            "total_chunks": total_chunks,
            "remaining_chunks": 0,
            "total_bytes": total_size,
            "processed_bytes": total_size if total_size else downloaded_bytes,
            "speed": (total_size or downloaded_bytes) / max(total_duration, 1e-6),
            "message": "下载完成",
            "success": True,
            "result": result,
        }
